! f90 include file

!_______________________________________________________________________
!
   function mfMax_mfArray( A, dim, nanflag ) result( out )

      type(mfArray)                          :: A
      integer,          intent(in), optional :: dim
      character(len=*), intent(in), optional :: nanflag
      type(mfArray)                          :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, j, nrow, ncol, col_length
      type(mfArray) :: At
      logical :: nan_is_discarded

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfMax", "E",                               &
                            "real array required!" )
         go to 99
      end if

      if( present(nanflag) ) then
         if( to_lower(nanflag) == "omitnan" ) then
            nan_is_discarded = .true.
         else if( to_lower(nanflag) == "includenan" ) then
            nan_is_discarded = .false.
         else
            call PrintMessage( "mfMax", "E",                            &
                               "bad argument for 'nanflag':",           &
                               "it must be equal to 'omitNaN' or 'includeNaN'!" )
            go to 99
         end if
      else
         nan_is_discarded = .true.
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      out%data_type = MF_DT_DBLE
      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! max by column (as in Matlab)
            out%shape = [ 1, ncol ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               if( nan_is_discarded ) then
                  do j = 1, ncol
                     out%double(1,j) = mf_maxval_omitNaN(A%double(:,j))
                  end do
               else ! NaN is included
                  do j = 1, ncol
                     out%double(1,j) = mf_maxval_includeNaN(A%double(:,j))
                  end do
               end if
            else if( A%data_type == MF_DT_SP_DBLE ) then
               if( nan_is_discarded ) then
                  do j = 1, ncol
                     col_length = A%j(j+1) - A%j(j)
                     if( col_length == 0 ) then
                        out%double(1,j) = 0.0d0
                     else
                        out%double(1,j) = mf_maxval_omitNaN(A%a(A%j(j):A%j(j+1)-1))
                        ! possible zero(s) must be taken into account
                        if( col_length < nrow ) then
                           out%double(1,j) = mf_max_omitNaN( out%double(1,j), 0.0d0 )
                        end if
                     end if
                  end do
               else ! NaN is included
                  do j = 1, ncol
                     col_length = A%j(j+1) - A%j(j)
                     if( col_length == 0 ) then
                        out%double(1,j) = 0.0d0
                     else
                        out%double(1,j) = mf_maxval_includeNaN(A%a(A%j(j):A%j(j+1)-1))
                        ! possible zero(s) must be taken into account
                        if( col_length < nrow ) then
                           out%double(1,j) = mf_max_includeNaN( out%double(1,j), 0.0d0 )
                        end if
                     end if
                  end do
               end if
            end if
            out%prop%triu = TRUE
         else if( dim == 2 ) then
            ! max by row
            out%shape = [ nrow, 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               if( nan_is_discarded ) then
                  do i = 1, nrow
                     out%double(i,1) = mf_maxval_omitNaN(A%double(i,:))
                  end do
               else ! NaN is included
                  do i = 1, nrow
                     out%double(i,1) = mf_maxval_includeNaN(A%double(i,:))
                  end do
               end if
            else if( A%data_type == MF_DT_SP_DBLE ) then
               ! getting the transpose of A
               ! (.t. operator not available, because it is in the
               !  OPS module)
               At%shape = [ ncol, nrow ]
               At%data_type = A%data_type
               allocate( At%a(size(A%a)) )

               allocate( At%i(size(A%i)) )

               allocate( At%j(At%shape(2)+1) )

               call csc_transp( nrow, ncol, A%a, A%i, A%j,              &
                                At%a, At%i, At%j )
               if( nan_is_discarded ) then
                  do j = 1, At%shape(2)
                     col_length = At%j(j+1) - At%j(j)
                     if( col_length == 0 ) then
                        out%double(j,1) = 0.0d0
                     else
                        out%double(j,1) = mf_maxval_omitNaN(At%a(At%j(j):At%j(j+1)-1))
                        ! possible zero(s) must be taken into account
                        if( col_length < At%shape(1) ) then
                           out%double(j,1) = mf_max_omitNaN( out%double(j,1), 0.0d0 )
                        end if
                     end if
                  end do
               else ! NaN is included
                  do j = 1, At%shape(2)
                     col_length = At%j(j+1) - At%j(j)
                     if( col_length == 0 ) then
                        out%double(j,1) = 0.0d0
                     else
                        out%double(j,1) = mf_maxval_includeNaN(At%a(At%j(j):At%j(j+1)-1))
                        ! possible zero(s) must be taken into account
                        if( col_length < At%shape(1) ) then
                           out%double(j,1) = mf_max_includeNaN( out%double(j,1), 0.0d0 )
                        end if
                     end if
                  end do
               end if
               call msRelease( At )
            end if
            out%prop%tril = TRUE
         else
            call PrintMessage( "mfMax", "E",                            &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( nrow == 1 .or. ncol == 1 ) then
            ! vector (dense !)
            out%shape = [ 1, 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( nrow == 1 ) then
               if( nan_is_discarded ) then
                  out%double(1,1) = mf_maxval_omitNaN(A%double(1,:))
               else ! NaN is included
                  out%double(1,1) = mf_maxval_includeNaN(A%double(1,:))
               end if
            else if( ncol == 1 ) then
               if( nan_is_discarded ) then
                  out%double(1,1) = mf_maxval_omitNaN(A%double(:,1))
               else ! NaN is included
                  out%double(1,1) = mf_maxval_includeNaN(A%double(:,1))
               end if
            end if
            out%prop%tril = TRUE
            out%prop%triu = TRUE
         else
            ! matrix -> processing by column
            out%shape = [ 1, ncol ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               if( nan_is_discarded ) then
                  do j = 1, ncol
                     out%double(1,j) = mf_maxval_omitNaN(A%double(:,j))
                  end do
               else ! NaN is included
                  do j = 1, ncol
                     out%double(1,j) = mf_maxval_includeNaN(A%double(:,j))
                  end do
               end if
            else if( A%data_type == MF_DT_SP_DBLE ) then
               if( nan_is_discarded ) then
                  do j = 1, ncol
                     col_length = A%j(j+1) - A%j(j)
                     if( col_length == 0 ) then
                        out%double(1,j) = 0.0d0
                     else
                        out%double(1,j) = mf_maxval_omitNaN(A%a(A%j(j):A%j(j+1)-1))
                        ! possible zero(s) must be taken into account
                        if( col_length < nrow ) then
                           out%double(1,j) = mf_max_omitNaN( out%double(1,j), 0.0d0 )
                        end if
                     end if
                  end do
               else ! NaN is included
                  do j = 1, ncol
                     col_length = A%j(j+1) - A%j(j)
                     if( col_length == 0 ) then
                        out%double(1,j) = 0.0d0
                     else
                        out%double(1,j) = mf_maxval_includeNaN(A%a(A%j(j):A%j(j+1)-1))
                        ! possible zero(s) must be taken into account
                        if( col_length < nrow ) then
                           out%double(1,j) = mf_max_includeNaN( out%double(1,j), 0.0d0 )
                        end if
                     end if
                  end do
               end if
            end if
            out%prop%triu = TRUE
         end if
      end if

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfMax_mfArray
!_______________________________________________________________________
!
   function mfMax_mfArray_mfArray( A, B, nanflag ) result( out )

      type(mfArray)                          :: A, B
      character(len=*), intent(in), optional :: nanflag
      type(mfArray)                          :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: nrow, ncol, nzmax, jerr, nzaplb
      logical :: sorted_version
      integer :: status
      integer :: i, j
      logical :: nan_is_discarded

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      if( mfIsEmpty(A) .or. mfIsEmpty(B) ) then
         ! result will be empty
         go to 99
      end if

      if( .not. ( mfIsReal(A) .and. mfIsReal(B) ) ) then
         call PrintMessage( "mfMax", "E",                               &
                            "real arrays required!" )
         go to 99
      end if

      if( any(A%shape /= B%shape) ) then
         call PrintMessage( "mfMax", "E",                               &
                            "mfArray 'A' and 'B' must have the same shape!" )
         go to 99
      end if

      if( present(nanflag) ) then
         if( to_lower(nanflag) == "omitnan" ) then
            nan_is_discarded = .true.
         else if( to_lower(nanflag) == "includenan" ) then
            nan_is_discarded = .false.
         else
            call PrintMessage( "mfMax", "E",                            &
                               "bad argument for 'nanflag':",           &
                               "it must be equal to 'omitNaN' or 'includeNaN'!" )
            go to 99
         end if
      else
         nan_is_discarded = .true.
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( A%data_type == MF_DT_SP_DBLE .and.                            &
          B%data_type == MF_DT_SP_DBLE ) then

         out%data_type = MF_DT_SP_DBLE
         out%shape = A%shape
         ! nzmax choice: at least nnzaplb, but preserve extra rooms
         ! from initial sparse arrays.
         call nnzaplb( nrow, ncol, a%i, a%j, b%i, b%j, nzaplb )
         nzmax = max( max(mfNzmax(a),mfNzmax(b)), nzaplb )
         call msAssign( out, mfSpAlloc(nrow,ncol,nzmax) )
         ! check row_sorted strategy
         if( mfIsRowSorted( A ) .and. mfIsRowSorted( B ) ) then
            sorted_version = .true.
         else
            if( MF_SP_AUTO_ROW_SORTED ) then
               if( .not. A%status_temporary ) then
                  if( .not. mfIsRowSorted( A ) ) then
                     call msRowSort( A )
                     call PrintMessage( "mfMax", "I",                   &
                                        "sparse matrix 'A' have been sorted!", &
                                        "(If you don't want that MUESLI sort itself,", &
                                        " you can use the 'msSetAutoRowSorted' routine to", &
                                        " change this behavior.)" )
                  end if
               end if
               if( .not. B%status_temporary ) then
                  if( .not. mfIsRowSorted( B ) ) then
                     call msRowSort( B )
                     call PrintMessage( "mfMax", "I",                   &
                                        "sparse matrix 'B' have been sorted!", &
                                        "(If you don't want that MUESLI sort itself,", &
                                        " you can use the 'msSetAutoRowSorted' routine to", &
                                        " change this behavior.)" )
                  end if
               end if
               if( mfIsRowSorted( A ) .and. mfIsRowSorted( B ) ) then
                  sorted_version = .true.
               else
                  sorted_version = .false.
               end if
            else ! row sorted under the user control
               sorted_version = .false.
               call PrintMessage( "mfMax", "W",                         &
                                  "at least one sparse matrix is not row sorted!",&
                                  "(sorting it permits to use a better algo.)" )
            end if
         end if
         if( sorted_version ) then
            if( nan_is_discarded ) then
               call maxab1_omitNaN( nrow, ncol, A%a, A%i, A%j, B%a, B%i, B%j, &
                                    out%a, out%i, out%j, nzmax, jerr )
            else ! NaN is included
               call maxab1_includeNaN( nrow, ncol, A%a, A%i, A%j, B%a, B%i, B%j, &
                                       out%a, out%i, out%j, nzmax, jerr )
            end if
         else
            if( nan_is_discarded ) then
               call maxab_omitNaN( nrow, ncol, A%a, A%i, A%j, B%a, B%i, B%j, &
                                   out%a, out%i, out%j, nzmax, jerr )
            else ! NaN is included
               call maxab_includeNaN( nrow, ncol, A%a, A%i, A%j, B%a, B%i, B%j, &
                                      out%a, out%i, out%j, nzmax, jerr )
            end if
         end if
         if( jerr /= 0 ) then
            write(STDERR,*) "(MUESLI mfMax:) internal error: maxab fails!"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop
         end if
         if( sorted_version ) then
            out%row_sorted = TRUE
         else
            out%row_sorted = UNKNOWN
         end if

      else if( A%data_type == MF_DT_DBLE .and.                          &
               B%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_DBLE
         out%shape = A%shape
         allocate( out%double(out%shape(1),out%shape(2)) )

         if( nan_is_discarded ) then
            out%double = mf_max_omitNaN( A%double, B%double )
         else ! NaN is included
            out%double = mf_max_includeNaN( A%double, B%double )
         end if

      else

         call PrintMessage( "mfMax", "E",                               &
                            "arrays cannot be sparse/dense!",           &
                            "(not yet finished)" )
         go to 99

      end if

      out%prop%tril = UNKNOWN
      out%prop%triu = UNKNOWN
      if( A%prop%symm == TRUE .and. B%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      else
         out%prop%symm = UNKNOWN
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, B%units, status )
         if( status /= 0 ) then
            call PrintMessage( "mfMax", "E",                            &
                               "the physical dimensions of the two mfArrays A and B", &
                               "are not consistent!" )
            go to 99
         end if
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )
      call msAutoRelease( A, B )

#endif
   end function mfMax_mfArray_mfArray
!_______________________________________________________________________
!
   function mfMax_mfArray_real8( A, b, nanflag ) result( out )

      type(mfArray)                              :: A
      real(kind=MF_DOUBLE), intent(in)           :: b
      character(len=*),     intent(in), optional :: nanflag
      type(mfArray)                              :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: nrow, ncol
      integer :: status
      integer :: i, j
      logical :: nan_is_discarded

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfMax", "E",                               &
                            "dense real array required!",               &
                            "(sparse array not yet supported)" )
         go to 99
      end if

      if( present(nanflag) ) then
         if( to_lower(nanflag) == "omitnan" ) then
            nan_is_discarded = .true.
         else if( to_lower(nanflag) == "includenan" ) then
            nan_is_discarded = .false.
         else
            call PrintMessage( "mfMax", "E",                            &
                               "bad argument for 'nanflag':",           &
                               "it must be equal to 'omitNaN' or 'includeNaN'!" )
            go to 99
         end if
      else
         nan_is_discarded = .true.
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      out%data_type = MF_DT_DBLE
      out%shape = A%shape
      allocate( out%double(nrow,ncol) )

      if( nan_is_discarded ) then
         out%double = mf_max_omitNaN( A%double, b )
      else ! NaN is included
         out%double = mf_max_includeNaN( A%double, b )
      end if

      out%prop%tril = UNKNOWN
      out%prop%triu = UNKNOWN
      out%prop%symm = UNKNOWN
      out%prop%posd = UNKNOWN

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfMax", "E",                            &
                               "the mfArray A should be non-dimensional!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfMax_mfArray_real8
!_______________________________________________________________________
!
   subroutine msMax( out, A, dim, nanflag )

      type(mfArray)                          :: A
      integer,          intent(in), optional :: dim
      character(len=*), intent(in), optional :: nanflag
      type(mf_Out)                           :: out
      !------ API end ------
#ifdef _DEVLP

      type(mfArray), pointer :: va, ia

      integer :: i, j, nrow, ncol, col_length
      type(mfArray) :: At
      logical :: nan_is_discarded

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! we must have two output arguments
      if( out%n /= 2 ) then
         call PrintMessage( "msMax", "E",                               &
                            "two output args required!",                &
                            "syntax is : call msMax ( mfOut(v,i), a [, dim] )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, A ) ) then
         call PrintMessage( "msMax", "E",                               &
                            "output arguments cannot be tempo, or cannot share", &
                            "same memory as another input argument." )
         go to 99
      end if

      if( present(nanflag) ) then
         if( to_lower(nanflag) == "omitnan" ) then
            nan_is_discarded = .true.
         else if( to_lower(nanflag) == "includenan" ) then
            nan_is_discarded = .false.
         else
            call PrintMessage( "msMax", "E",                            &
                               "bad argument for 'nanflag':",           &
                               "it must be equal to 'omitNaN' or 'includeNaN'!" )
            go to 99
         end if
      else
         nan_is_discarded = .true.
      end if

      va => out%ptr1
      ia => out%ptr2
      call msSilentRelease( va, ia )

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "msMax", "E",                               &
                            "real array required!" )
         go to 99
      end if

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      va%data_type = MF_DT_DBLE
      ia%data_type = MF_DT_DBLE

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! max by column (as in Matlab)
            va%shape = [ 1, ncol ]
            ia%shape = [ 1, ncol ]
            allocate( va%double(va%shape(1),va%shape(2)) )

            allocate( ia%double(ia%shape(1),ia%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               if( nan_is_discarded ) then
                  do j = 1, ncol
                     call ms_maxval_loc_omitNaN( A%double(:,j),         &
                                                 va%double(1,j),        &
                                                 ia%double(1,j) )
                  end do
               else ! NaN is included
                  do j = 1, ncol
                     call ms_maxval_loc_includeNaN( A%double(:,j),      &
                                                    va%double(1,j),     &
                                                    ia%double(1,j) )
                  end do
               end if
            else if( A%data_type == MF_DT_SP_DBLE ) then
               do j = 1, ncol
                  ! check number of non zeros in the column
                  col_length = A%j(j+1) - A%j(j)
                  if( col_length == 0 ) then
                     va%double(1,j) = 0.0d0
                     ia%double(1,j) = 1
                  else
                     if( nan_is_discarded ) then
                        call ms_maxval_loc_omitNaN( A%a(A%j(j):A%j(j+1)-1), &
                                                    va%double(1,j),     &
                                                    ia%double(1,j) )
                     else ! NaN is included
                        call ms_maxval_loc_includeNaN( A%a(A%j(j):A%j(j+1)-1), &
                                                       va%double(1,j),  &
                                                       ia%double(1,j) )
                     end if
                     ! possible zero(s) must be taken into account
                     if( col_length < nrow ) then
                        if( va%double(1,j) < 0.0d0 ) then
                           va%double(1,j) = 0.0d0
                           ia%double(1,j) = first_index_not_in(A%i(A%j(j):A%j(j+1)-1))
                        else
                           ! transform index in a dense row number
                           ia%double(1,j) = A%i( A%j(j)-1 + nint(ia%double(1,j)) )
                        end if
                     end if
                  end if
               end do
            end if
         else if( dim == 2 ) then
            ! max by row
            va%shape = [ nrow, 1 ]
            ia%shape = [ nrow, 1 ]
            allocate( va%double(va%shape(1),va%shape(2)) )

            allocate( ia%double(ia%shape(1),ia%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               if( nan_is_discarded ) then
                  do i = 1, nrow
                     call ms_maxval_loc_omitNaN( A%double(i,:),         &
                                                 va%double(i,1),        &
                                                 ia%double(i,1) )
                  end do
               else ! NaN is included
                  do i = 1, nrow
                     call ms_maxval_loc_includeNaN( A%double(i,:),      &
                                                    va%double(i,1),     &
                                                    ia%double(i,1) )
                  end do
               end if
            else if( A%data_type == MF_DT_SP_DBLE ) then
               ! getting the transpose of A
               ! (.t. operator not available, because it is in the
               !  OPS module)
               At%shape = [ ncol, nrow ]
               At%data_type = A%data_type
               allocate( At%a(size(A%a)) )

               allocate( At%i(size(A%i)) )

               allocate( At%j(At%shape(2)+1) )

               call csc_transp( nrow, ncol, A%a, A%i, A%j,              &
                                At%a, At%i, At%j )
               do j = 1, At%shape(2)
                  ! check number of non zeros in the column of A^t
                  col_length = At%j(j+1) - At%j(j)
                  if( col_length == 0 ) then
                     va%double(j,1) = 0.0d0
                     ia%double(j,1) = 1
                  else
                     if( nan_is_discarded ) then
                        call ms_maxval_loc_omitNaN( At%a(At%j(j):At%j(j+1)-1), &
                                                    va%double(j,1),     &
                                                    ia%double(j,1) )
                     else ! NaN is included
                        call ms_maxval_loc_includeNaN( At%a(At%j(j):At%j(j+1)-1), &
                                                       va%double(j,1),  &
                                                       ia%double(j,1) )
                     end if
                     ! possible zero(s) must be taken into account
                     if( col_length < At%shape(1) ) then
                        if( va%double(j,1) < 0.0d0 ) then
                           va%double(j,1) = 0.0d0
                           ia%double(j,1) = first_index_not_in(At%i(At%j(j):At%j(j+1)-1))
                        else
                           ! transform index in a dense col number
                           ia%double(j,1) = At%i( At%j(j)-1 + nint(ia%double(j,1)) )
                        end if
                     end if
                  end if
               end do
            end if
            call msSilentRelease( At )
         else
            call PrintMessage( "msMax", "E",                            &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( nrow == 1 .or. ncol == 1 ) then
            ! vector (dense !)
            va%shape = [ 1, 1 ]
            ia%shape = [ 1, 1 ]
            allocate( va%double(1,1) )

            allocate( ia%double(1,1) )

            if( nrow == 1 ) then
               if( nan_is_discarded ) then
                  call ms_maxval_loc_omitNaN( A%double(1,:),            &
                                              va%double(1,1),           &
                                              ia%double(1,1) )
               else ! NaN is included
                  call ms_maxval_loc_includeNaN( A%double(1,:),         &
                                                 va%double(1,1),        &
                                                 ia%double(1,1) )
               end if
            else if( ncol == 1 ) then
               if( nan_is_discarded ) then
                  call ms_maxval_loc_omitNaN( A%double(:,1),            &
                                              va%double(1,1),           &
                                              ia%double(1,1) )
               else ! NaN is included
                  call ms_maxval_loc_includeNaN( A%double(:,1),         &
                                                 va%double(1,1),        &
                                                 ia%double(1,1) )
               end if
            end if
         else
            ! matrix -> processing by column
            va%shape = [ 1, ncol ]
            ia%shape = [ 1, ncol ]
            allocate( va%double(va%shape(1),va%shape(2)) )

            allocate( ia%double(ia%shape(1),ia%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               if( nan_is_discarded ) then
                  do j = 1, ncol
                     call ms_maxval_loc_omitNaN( A%double(:,j),         &
                                                 va%double(1,j),        &
                                                 ia%double(1,j) )
                  end do
               else ! NaN is included
                  do j = 1, ncol
                     call ms_maxval_loc_includeNaN( A%double(:,j),      &
                                                    va%double(1,j),     &
                                                    ia%double(1,j) )
                  end do
               end if
            else if( A%data_type == MF_DT_SP_DBLE ) then
               do j = 1, ncol
                  ! check number of non zeros in the column
                  col_length = A%j(j+1) - A%j(j)
                  if( col_length == 0 ) then
                     va%double(1,j) = 0.0d0
                     ia%double(1,j) = 1
                  else
                     if( nan_is_discarded ) then
                        call ms_maxval_loc_omitNaN( A%a(A%j(j):A%j(j+1)-1), &
                                                    va%double(1,j),     &
                                                    ia%double(1,j) )
                     else ! NaN is included
                        call ms_maxval_loc_includeNaN( A%a(A%j(j):A%j(j+1)-1), &
                                                       va%double(1,j),  &
                                                       ia%double(1,j) )
                     end if
                     ! possible zero(s) must be taken into account
                     if( col_length < nrow ) then
                        if( va%double(1,j) < 0.0d0 ) then
                           va%double(1,j) = 0.0d0
                           ia%double(1,j) = first_index_not_in(A%i(A%j(j):A%j(j+1)-1))
                        else
                           ! transform index in a dense row number
                           ia%double(1,j) = A%i( A%j(j)-1 + nint(ia%double(1,j)) )
                        end if
                     end if
                  end if
               end do
            end if
         end if
      end if

      va%prop%symm = FALSE
      va%prop%posd = FALSE
      ia%prop%symm = FALSE
      ia%prop%posd = FALSE

      if( mf_phys_units ) then
         va%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msMax
