! f90 include file

!-----------------------------------------------------------------------
! numerical progression using percentages, with prints in erase mode
!
! Copyright É. Canot 2003-2025 -- IPR/CNRS
!_______________________________________________________________________
!
   subroutine msPrepProgress_int( start, end, disp_times, estimator )

      integer,              intent(in)           :: start, end
      logical,              intent(in), optional :: disp_times
      character(len=3),     intent(in), optional :: estimator
      !------ API end ------

#ifdef _DEVLP
      ! special case where arguments 'start' and 'end' are integers

      call msPrepProgress_dble( dble(start), dble(end),                 &
                                disp_times, estimator )

#endif
   end subroutine msPrepProgress_int
!_______________________________________________________________________
!
   subroutine msPrepProgress_dble( start, end, disp_times, estimator )

      real(kind=MF_DOUBLE), intent(in)           :: start, end
      logical,              intent(in), optional :: disp_times
      character(len=3),     intent(in), optional :: estimator
      !------ API end ------

#ifdef _DEVLP
      ! (call must be placed just before the loop)

      !  prepare constant data used by 'msPrintProgress'
      !
      !  values monitoring the loop in the calling program:
      !  start (in) : start value
      !  end   (in) : end value
      !
      !  constants initialized here are passed to other routines
      !  via a common.
      !
      !  Two choices for the estimator :
      !    1 : "GLE" : Global Linear Estimator
      !                (more efficient, more stable, less accurate)
      !    2 : "LPE" : Local Power Estimator (solve a small LSQ pb),
      !                involving a power law of the index iterator,
      !                based on the ten last values.
      !                (less efficient, less stable, more accurate)
      !    3 : "new" : (in test) Power Estimator using only three points:
      !                the origin points (of course (0,0)), the current
      !                point and the intermediate between the current one
      !                and the origin one.
      !    4 : "CLE" : (in test) Cubic Local Estimator using the last
      !                15 points.
      !   Default is 1 ("GLE")

      real(kind=MF_DOUBLE) :: pc_fact, cpu_init,                        &
                              val_min, val_max, val_period,             &
                              left_time_old
      integer :: clock_rate, clock_init, estim, nb_period_past
      logical :: reverse, times
      common /mf_echo_progress/ pc_fact, cpu_init,                      &
                                val_min, val_max, val_period,           &
                                clock_rate, clock_init, estim, nb_period_past, &
                                reverse, times, left_time_old

      character(len=80) :: string

   !------ end of declarations -- execution starts hereafter  ------

      if( start < end ) then
         val_min = start
         val_max = end
         reverse = .false.
      else if( start > end ) then
         val_min = end
         val_max = start
         reverse = .true.
      else
         call PrintMessage( "msPrepProgress", "W",                      &
                            "nothing to do" )
         pc_fact = MF_NAN
         return
      end if

      ! factor for computing the percentage in the next routine
      pc_fact = 100.0d0/(val_max-val_min)

      ! computing the jump (want to print only at a resolution of 0.1 %)
      val_period = (val_max-val_min)/1000.0d0

      nb_period_past = -1

      if( present(disp_times) ) then
         times = disp_times
      else
         times = .true.
      end if

      estim = 1 ! Default is GLE
      if( present(estimator) ) then
         if( to_lower(estimator) == "gle" ) then
            estim = 1
         else if( to_lower(estimator) == "lpe" ) then
            estim = 2
            call cpu_time( cpu_init )
         else if( to_lower(estimator) == "new" ) then
            estim = 3
            call cpu_time( cpu_init )
         else if( to_lower(estimator) == "cle" ) then
            estim = 4
            call cpu_time( cpu_init )
         end if
      end if

      if( times ) then
         call system_clock( count_rate=clock_rate, count=clock_init )
         string = "  0.0 % (time left =   0h  0m  0s -- estim. remain. time =    h   m   s)"
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
         call put_string_on_term_no_adv( trim(string)//char(0) )
#else
         write(STDOUT,"(A)",advance="no") trim(string)
         call msFlush(STDOUT)
#endif
         left_time_old = -1.0d0
      else
         string = "  0.0 %"
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
         call put_string_on_term_no_adv( trim(string)//char(0) )
#else
         write(STDOUT,"(A)",advance="no") trim(string)
         call msFlush(STDOUT)
#endif
      end if

#endif
   end subroutine msPrepProgress_dble
!_______________________________________________________________________
!
   subroutine msPrintProgress_int( val )

      integer, intent(in) :: val
      !------ API end ------

#ifdef _DEVLP
      ! special case where arguments 'start' and 'end' are integers

      call msPrintProgress_dble( dble(val) )

#endif
   end subroutine msPrintProgress_int
!_______________________________________________________________________
!
   subroutine msPrintProgress_dble( val )

      real(kind=MF_DOUBLE), intent(in) :: val
      !------ API end ------

#ifdef _DEVLP
      ! (call should be placed at the end of the loop)

      ! print the percentage of work done on STDOUT
      !
      ! at each call of the current routine, the percentage of work done
      ! is printed, after erasing the previous write.

      real(kind=MF_DOUBLE) :: pc_fact, cpu_init,                        &
                              val_min, val_max, val_period,             &
                              left_time_old
      integer :: clock_rate, clock_init, estim, nb_period_past
      logical :: reverse, times
      common /mf_echo_progress/ pc_fact, cpu_init,                      &
                                val_min, val_max, val_period,           &
                                clock_rate, clock_init, estim, nb_period_past, &
                                reverse, times, left_time_old

      real(kind=MF_DOUBLE) :: percent, left_time, total_time,           &
                              cpu, left_cpu
      real(kind=MF_DOUBLE), save :: cpu_save
      character(len=80) :: string
      integer :: clock, rem_time, k
      integer, save :: clock_save
      integer :: hrs_1, min_1, sec_1, hrs_2, min_2, sec_2

      integer :: nb_period
      integer, save :: index = 0

      integer, parameter :: index_max = 10
      real(kind=MF_DOUBLE), save :: time_log10(index_max,2) = 0.0d0

      integer, parameter :: index_max_15 = 15
      real(kind=MF_DOUBLE), save :: time_array(index_max_15,2) = 0.0d0

      real(kind=MF_DOUBLE) :: p_fit, cte_fit
      real(kind=MF_DOUBLE) :: d_val, x, coeff_4(4)

      ! new Power Estimator (in test)
      type(mfArray), save :: table_time
      integer :: i_middle, i_size
      real(kind=MF_DOUBLE) :: a, b, left_time_mid, val_mid, delta

   !------ end of declarations -- execution starts hereafter  ------

      if( times ) then
         call system_clock( count=clock )
         left_time = dble(clock-clock_init)/clock_rate
      end if

      nb_period = (val - val_min)/val_period
      if( nb_period > nb_period_past ) then
         nb_period_past = nb_period_past + 1
      else
         ! The return is conditional, and depends of the displayed time...
         if( nint(left_time) == nint(left_time_old) ) then
            return
         end if
      end if

      if( reverse ) then
         percent = (val_max-val)*pc_fact
      else
         percent = (val-val_min)*pc_fact
      end if

      if( times ) then

         left_time_old = left_time

         if( estim == 1 ) then
            ! Global Linear Estimator (1)
            if( reverse ) then
               d_val = (val_max-val)
               if( d_val == 0.0d0 ) then
                  ! not huge(1.0d0) because at line 303,
                  ! it will be converted to an integer
                  total_time = huge(1)
               else
                  total_time = left_time/d_val*(val_max-val_min)
               end if
            else
               d_val = (val-val_min)
               if( d_val == 0.0d0 ) then
                  ! not huge(1.0d0) because at line 303,
                  ! it will be converted to an integer
                  total_time = huge(1)
               else
                  total_time = left_time/d_val*(val_max-val_min)
               end if
            end if
         else if( estim == 2 ) then
            call cpu_time( cpu )
            left_cpu = cpu - cpu_init
            ! Local Power Estimator (1)
            if( index < index_max ) then
               if( index == 0 ) then
                  cpu_save = cpu_init
               end if
               index = index + 1
               time_log10(index,1) = log10( val )
               time_log10(index,2) = log10( cpu-cpu_save )
               cpu_save = cpu
               ! GLE
               total_time = left_time/(val-val_min)*(val_max-val_min)
               total_time = total_time*left_time/left_cpu
            else
               time_log10 = eoshift(time_log10,shift=1,dim=1)
               time_log10(index_max,1) = log10( val )
               time_log10(index,2) = log10( cpu-cpu_save )
               cpu_save = cpu
               ! LPE : cost(i) = cte*i**p
               !       total = int( cost(i), i = 0..N )
               !             = cte/(p+1)*N^(p+1)
               !
               ! progress_x, progress_y, progress_linfit are global
               ! variables declared in the 'polyfun' module
               call msAssign( progress_x, .t. mf(time_log10(:,1)) )
               call msAssign( progress_y, .t. mf(time_log10(:,2)) )
               call msAssign( progress_linfit, mfPolyfit( progress_x, progress_y, n=1 ) )
               p_fit = mfGet(progress_linfit,1)
               cte_fit = 10.0d0**mfDble(mfGet(progress_linfit,2))
               total_time = cte_fit/(p_fit+1.0d0)                       &
                  *( val_max**(p_fit+1.0d0) - val_min**(p_fit+1.0d0) )
               total_time = total_time*left_time/left_cpu
            end if
         else if( estim == 3 ) then ! ("new", in test)
            table_time = table_time .vc. mf([ left_time, val ])
            ! the new estimator needs at least three points
            i_size = size(table_time,1)
!!print *
            if( i_size >= 2 ) then
               ! find the middle of the table
               i_middle = i_size/2
               left_time_mid = mfGet( table_time, i_middle, 1 )
               val_mid       = mfGet( table_time, i_middle, 2 )
!!print *, "left_time_mid = ", left_time_mid, " val_mid = ", val_mid
!!print *, "left_time = ", left_time, " val = ", val
               ! first case: time is function of val
               a = (left_time_mid-left_time*val_mid/val) / (val_mid**2-val*val_mid)
               b = (left_time-a*val**2) / val
               total_time = a*val_max**2 + b*val_max
!!print *, "a = ", a, " b = ", b, " total_time = ", total_time, " [val,time]"
               ! second case: val is function of time
               a = (val_mid-val*left_time_mid/left_time) / (left_time_mid**2-left_time*left_time_mid)
               b = (val-a*left_time**2) / left_time
               delta = b**2 + 4*a*val_max
               total_time = (-b+sqrt(delta))/ (2*a)
!!print *, "a = ", a, " b = ", b, " total_time = ", total_time, " [time,val]"
!### TODO ?: comment choisir entre l'une ou l'autre des formules ?
            else
               ! use the linear estimator
               d_val = (val-val_min)
               if( d_val == 0.0d0 ) then
                  ! not huge(1.0d0) because at line 303,
                  ! it will be converted to an integer
                  total_time = huge(1)
               else
                  total_time = left_time/(val-val_min)*(val_max-val_min)
               end if
!!print *, "(linear) total_time = ", total_time
            end if
         else ! estim == 4 ("LCE", in test)
            ! Cubic Local Estimator
            if( index < index_max_15 ) then
               index = index + 1
               time_array(index,1) = val
               time_array(index,2) = left_time
               ! GLE because time_array is not yet fulfilled...
               total_time = left_time/(val-val_min)*(val_max-val_min)
               total_time = total_time*left_time/left_cpu
            else
               time_array = eoshift(time_array,shift=1,dim=1)
               time_array(index_max_15,1) = val
               time_array(index,2)        = left_time
               ! LCE : cost(i) = cte*i**3
               !       total = int( cost(i), i = 0..N )
               !             = cte/(p+1)*N^(p+1)
               !
               ! progress_x, progress_y, progress_linfit are global
               ! variables declared in the 'polyfun' module
               call msAssign( progress_x, .t. mf(time_array(:,1)) )
               call msAssign( progress_y, .t. mf(time_array(:,2)) )
               coeff_4(1:4) = mfPolyfit( progress_x, progress_y, n=3 )
               x = val_max - val
               total_time = coeff_4(1)*x**3 + coeff_4(2)*x**2 +         &
                            coeff_4(3)*x    + coeff_4(4)
            end if
         end if

         call sec_2_hms( left_time, hrs_1, min_1, sec_1 )
         rem_time = nint(max(0.0d0, total_time-left_time))
         call sec_2_hms( rem_time, hrs_2, min_2, sec_2 )

         write(string,"(F5.1,A,I3,A,2(I2,A),A,I3,A,2(I2,A))")           &
               percent, " % (time left = ",                             &
               hrs_1, "h ", min_1, "m ", sec_1, "s ",                   &
               "-- estim. remain. time = ",                             &
               hrs_2, "h ", min_2, "m ", sec_2, "s)"
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
         call put_string_back_on_term_no_adv( trim(string)//char(0) )
#else
         call go_home_on_term()
         write(STDOUT,"(A)",advance="no") trim(string)
         call msFlush(STDOUT)
#endif

      else

         write(string,"(F5.1,A)") percent, " %"
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
         call put_string_back_on_term_no_adv( trim(string)//char(0) )
#else
         call go_home_on_term()
         write(STDOUT,"(A)",advance="no") trim(string)
         call msFlush(STDOUT)
#endif

      end if

#endif
   end subroutine msPrintProgress_dble
!_______________________________________________________________________
!
   subroutine msPostProgress()
      !------ API end ------

#ifdef _DEVLP
      ! (call must be placed just after the loop)
      !
      !  complementary print after the loop

      real(kind=MF_DOUBLE) :: pc_fact, cpu_init,                        &
                              val_min, val_max, val_period,             &
                              left_time_old
      integer :: clock_rate, clock_init, estim, nb_period_past
      logical :: reverse, times
      common /mf_echo_progress/ pc_fact, cpu_init,                      &
                                val_min, val_max, val_period,           &
                                clock_rate, clock_init, estim, nb_period_past, &
                                reverse, times, left_time_old

      real(kind=MF_DOUBLE) :: left_time
      character(len=80) :: string
      integer :: clock, k
      integer :: hrs_1, min_1, sec_1

   !------ end of declarations -- execution starts hereafter  ------

      if( times ) then

         call system_clock( count=clock )
         left_time = dble(clock-clock_init)/clock_rate

         call sec_2_hms( left_time, hrs_1, min_1, sec_1 )

         write(string,"(A,I3,A,2(I2,A),A40)") " done. (time left = ",   &
               hrs_1, "h ", min_1, "m ", sec_1, "s)", " "
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
         call put_string_back_on_term( string(1:72)//char(0) )
#else
         call go_home_on_term()
         write(STDOUT,"(A)") string(1:72)
         call msFlush(STDOUT)
#endif

      else

         string = " done.  "
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
         call put_string_back_on_term( string(1:7)//char(0) )
#else
         call go_home_on_term()
         write(STDOUT,"(A)") string(1:7)
         call msFlush(STDOUT)
#endif

      end if

      call msSilentRelease( progress_x, progress_y, progress_linfit )

#endif
   end subroutine msPostProgress
