!_______________________________________________________________________
!
   function TriMesh_tri( x, y, tri,                                     &
                         icol, linewidth, height,                       &
                         tri_num, nod_num,                              &
                         boundary_only )                                &
   result( handle )

      type(mfArray)                    :: x, y, tri
      integer,              intent(in) :: icol
      real(kind=MF_DOUBLE), intent(in) :: linewidth, height
      logical,              intent(in) :: tri_num, nod_num, boundary_only

      integer :: handle
      !------ API end ------

      ! pointers for manipulating mfArray out of fml module
      real(kind=MF_DOUBLE), pointer :: x_ptr(:,:), y_ptr(:,:),          &
                                       tri_ptr(:,:)

      integer :: mf_message_level_save
      integer :: p_dim, index_min, index_max, nn, ntri
      real(kind=MF_DOUBLE), pointer :: xy_sto_pg(:,:)
      integer, pointer :: tri_sto_pg(:,:)
      logical :: find_in_col

      integer, allocatable :: coo_i(:), coo_j(:)
      real(kind=MF_DOUBLE), allocatable :: coo_val(:)
      integer :: nz

      integer :: i, j, k, k_tmp, kk1, kk2, i_edge, i1, i2
      integer :: it, f1, f2, n1, n2, ni, nj, itmp
      integer :: nf_edge, nn_edge, nnz_tri_sp, nnz_nod_sparse,          &
                 nb_subparts
      real(kind=MF_DOUBLE) :: val
      real(kind=MF_DOUBLE), pointer :: xy_sto_edge(:,:)
      integer, pointer :: list_nod_nums(:), list_end_subparts(:)
      type(mfArray) :: tri_sp, mf_list_end_subparts

      character(len=*), parameter :: ROUTINE_NAME = "mf/msTriMesh"

   !------ end of declarations -- execution starts hereafter  ------

      handle = 0

      call msInitArgs( x, y, tri )

      ! checking that 'x' is allocated
      if( mfIsEmpty(x) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' is empty!" )
         go to 99
      end if

      ! 'x' must be Real
      if( .not. mfIsReal(x) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' must be real!" )
         go to 99
      end if

      ! 'x' cannot be sparse
      if( mfIsSparse(x) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' cannot be sparse!" )
         go to 99
      end if

      call msPointer( x, x_ptr, no_crc=.true., intern_call=.true. )

      ! checking that 'y' is allocated
      if( mfIsEmpty(y) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'y' is empty!" )
         go to 99
      end if

      ! 'y' must be Real
      if( .not. mfIsReal(y) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'y' must be real!" )
         go to 99
      end if

      ! 'y' cannot be sparse
      if( mfIsSparse(y) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'y' cannot be sparse!" )
         go to 99
      end if

      call msPointer( y, y_ptr, no_crc=.true., intern_call=.true. )

      ! checking that 'tri' is allocated
      if( mfIsEmpty(tri) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'tri' is empty!" )
         go to 99
      end if

      ! 'tri' must be Real
      if( .not. mfIsReal(tri) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'tri' must be real!" )
         go to 99
      end if

      ! 'tri' cannot be sparse
      if( mfIsSparse(tri) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'tri' cannot be sparse!" )
         go to 99
      end if

      call msPointer( tri, tri_ptr, no_crc=.true. )

      ! checking that 'x' is a vector
      if( size(x_ptr,1)/=1 .and. size(x_ptr,2)/=1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' must be a vector!" )
         go to 99
      end if

      ! checking that 'y' is a vector
      if( size(y_ptr,1)/=1 .and. size(y_ptr,2)/=1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'y' must be a vector!" )
         go to 99
      end if

      ! checking that 'x' and 'y' have the same shape
      if( any( shape(x_ptr) /= shape(y_ptr) ) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' and 'y' must have the same shape!" )
         go to 99
      end if

      if( size(x_ptr,1) /= 1 ) then
         p_dim = 1
      else
         p_dim = 2
      end if

      nn = size(x_ptr,p_dim)
      ntri = size(tri,1)

      ! checking that 'tri' array is a valid triangulation
      if( size(tri,2) /= 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'tri' is not a valid triangulation!",      &
                            "(array must have 3 columns)" )
         go to 99
      end if
      index_min = minval(tri_ptr)
      index_max = maxval(tri_ptr)
      if( index_min /= 1 .and. index_max /= nn ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'tri' is not a valid triangulation!",      &
                            "('tri' indices must be ranged between 1 and the nb of elements in 'x')" )
         go to 99
      end if

      if( boundary_only ) then

         ! Count the number of edge faces
         ! (in the absence of connectivity, we have to simulate a drawing)

         ! nf_edge is determined by the following way: we record all
         ! anonymous faces, referenced by the node indices (n1,n2)
         ! in the sparse matrix 'tri_sp', initially large enough to
         ! contain all the faces. At the end, it will contain only 1 or 2;
         ! -> edge faces correspond to elements storing a 1.

         ! 3*nn is an estimation of the total number of faces for a large,
         ! hexagonal mesh.
         allocate( coo_i(2*3*nn), coo_j(2*3*nn), coo_val(2*3*nn) )
         nz = 0
         do it = 1, ntri
            ! examine the three faces of this triangle
            do f1 = 1, 3
               f2 = f1 + 1
               if( f2 > 3 ) f2 = 1
               ! (i,j) is the couple of nodes defining a face of the triangle
               n1 = tri_ptr(it,f1)
               n2 = tri_ptr(it,f2)
               if( n2 < n1 ) then
                  k_tmp = n1
                  n1 = n2
                  n2 = k_tmp
               end if
               ! Below, using the "added" option during the conversion
               ! COO -> CSC, helps a lot in the filling of the matrix:
               ! this avoid getting a value (costly operation) and testing
               ! this value.
               nz = nz + 1
               coo_i(nz) = n1
               coo_j(nz) = n2
               coo_val(nz) = 1.0d0
            end do
         end do

         tri_sp = mfSpImport( coo_i(1:nz), coo_j(1:nz), coo_val(1:nz),  &
                              nn, nn, duplicated_entries="added" )
         deallocate( coo_i, coo_j, coo_val )

         ! Removing all 2s in the sparse matrix ...
         tri_sp = 2.0d0*mfSpOnes(tri_sp) - tri_sp
         ! ... and compact it (because nn is usually very large in
         !     comparison with nf).
         call msSpReAlloc( tri_sp, "minimal" )
         nf_edge = mfNnz( tri_sp )

         ! As the boundaries (including the internal ones) are closed
         ! curves, the number of nodes involved is equal to the number
         ! of edge faces.
         nn_edge = nf_edge

         ! Retrieve the boundary path, seeing 'tri_sp' as a matrix
         ! of adjacency.

         allocate( xy_sto_edge(nn_edge,2) )
         allocate( list_nod_nums(nn_edge) )
#ifndef _OPTIM
xy_sto_edge(:,:) = MF_NAN
list_nod_nums(:) = 0
#endif

         ! Makes tri_sp symmetric
         tri_sp = tri_sp + .t. tri_sp

         mf_list_end_subparts = MF_EMPTY

         ! Search for continuous faces
         call init_search_faces( ibegin=1 )

         do i = 1, nf_edge-1 ! loop over all faces

            ! Record the current face
            if( i == 1 ) then
               if( p_dim == 1 ) then
                  xy_sto_edge(i+1,1) = x_ptr(nj,1)
                  xy_sto_edge(i+1,2) = y_ptr(nj,1)
               else
                  xy_sto_edge(i+1,1) = x_ptr(1,nj)
                  xy_sto_edge(i+1,2) = y_ptr(1,nj)
               end if
            else ! i /= 1
               if( ni == list_nod_nums(i) ) then
                  if( p_dim == 1 ) then
                     xy_sto_edge(i+1,1) = x_ptr(nj,1)
                     xy_sto_edge(i+1,2) = y_ptr(nj,1)
                  else
                     xy_sto_edge(i+1,1) = x_ptr(1,nj)
                     xy_sto_edge(i+1,2) = y_ptr(1,nj)
                  end if
                  list_nod_nums(i+1) = nj
               else
                  if( p_dim == 1 ) then
                     xy_sto_edge(i+1,1) = x_ptr(ni,1)
                     xy_sto_edge(i+1,2) = y_ptr(ni,1)
                  else
                     xy_sto_edge(i+1,1) = x_ptr(1,ni)
                     xy_sto_edge(i+1,2) = y_ptr(1,ni)
                  end if
                  list_nod_nums(i+1) = ni
               end if
            end if

            ! Find the next face
            if( find_in_col ) then
               kk1 = tri_sp%j(nj)
               kk2 = kk1 + 1
               ! elements of col nj are stored in a(kk1:kk2)
               if( tri_sp%a(kk1) == 1 ) then
                  ! kk1 is the new face
                  ni = tri_sp%i(kk1)
                  ! tag the face
                  tri_sp%a(kk1) = -1.0d0
               else if( tri_sp%a(kk2) == 1 ) then
                  ! kk2 is the new face
                  ni = tri_sp%i(kk2)
                  ! tag the face
                  tri_sp%a(kk2) = -1.0d0
               else
                  mf_list_end_subparts = mf_list_end_subparts .hc. mf(i)
                  ! removing all -1s in tri_sp
                  tri_sp = ( tri_sp + mfSpOnes(tri_sp) ) / 2.0d0
                  call init_search_faces( ibegin=i+1 )
                  cycle
               end if
               call msSet( -1.0d0, tri_sp, nj, ni )
            else ! find in row
               ! transpose the position (ni,nj)
               itmp = ni
               ni = nj
               nj = itmp
               ! now find in col
               kk1 = tri_sp%j(nj)
               kk2 = kk1 + 1
               if( tri_sp%a(kk1) == 1 ) then
                  ! kk1 is the new face
                  ni = tri_sp%i(kk1)
                  ! tag the face
                  tri_sp%a(kk1) = -1.0d0
               else if( tri_sp%a(kk2) == 1 ) then
                  ! kk2 is the new face
                  ni = tri_sp%i(kk2)
                  ! tag the face
                  tri_sp%a(kk2) = -1.0d0
               else
                  mf_list_end_subparts = mf_list_end_subparts .hc. mf(i)
                  ! removing all -1s in tri_sp
                  tri_sp = ( tri_sp + mfSpOnes(tri_sp) ) / 2.0d0
                  call init_search_faces( ibegin=i+1 )
                  cycle
               end if
               call msSet( -1.0d0, tri_sp, nj, ni )
               ! transpose back the position
               itmp = ni
               ni = nj
               nj = itmp
            end if

            find_in_col = .not. find_in_col

         end do
         mf_list_end_subparts = mf_list_end_subparts .hc. mf(nf_edge)

         nb_subparts = size( mf_list_end_subparts )
         allocate( list_end_subparts(nb_subparts) )
         list_end_subparts(:) = mf_list_end_subparts

         call msRelease( tri_sp, mf_list_end_subparts )

         handle = MeshBoundary_unstruct( xy_sto_edge, list_end_subparts, &
                                         list_nod_nums,                 &
                                         icol, linewidth, height, nod_num )

      else

         allocate( xy_sto_pg(nn,2) )

         allocate( tri_sto_pg(ntri,3) )

         if( p_dim == 1 ) then
            xy_sto_pg(:,1) = x_ptr(:,1)
            xy_sto_pg(:,2) = y_ptr(:,1)
         else
            xy_sto_pg(:,1) = x_ptr(1,:)
            xy_sto_pg(:,2) = y_ptr(1,:)
         end if
         tri_sto_pg(:,:) = tri_ptr(:,:)

         handle = MeshTri_tri( xy_sto_pg, tri_sto_pg, icol, linewidth,  &
                               height, tri_num, nod_num )

      end if

      call msFreePointer( x, x_ptr )
      call msFreePointer( y, y_ptr )
      call msFreePointer( tri, tri_ptr )

 99   continue

      call msFreeArgs( x, y, tri )
      call msAutoRelease( x, y, tri )

   contains
   !____________________________________________________________________
   !
      subroutine init_search_faces( ibegin )

         integer, intent(in) :: ibegin
         !------ API end ------

         integer :: i, j

         ! (ni,nj) must be shared with the main routine; they are not
         ! local variables.

      !------ end of declarations -- execution starts hereafter  ------

         ! Found the first element available in tri_sp
         do j = 1, nn ! loop over columns
            if( tri_sp%j(j+1) == tri_sp%j(j) ) then
               ! no element in this column
               cycle
            end if
            ni = tri_sp%i(1)
            nj = j
            exit
         end do
         ! tag the face and it's symmetric
         tri_sp%a(1) = -1.0d0
         call msSet( -1.0d0, tri_sp, nj, ni )
         i = ibegin
         if( p_dim == 1 ) then
            xy_sto_edge(i,1) = x_ptr(ni,1)
            xy_sto_edge(i,2) = y_ptr(ni,1)
         else
            xy_sto_edge(i,1) = x_ptr(1,ni)
            xy_sto_edge(i,2) = y_ptr(1,ni)
         end if
         list_nod_nums(i) = ni
         list_nod_nums(i+1) = nj

         find_in_col = .true.

      end subroutine init_search_faces
   !____________________________________________________________________
   !
   end function TriMesh_tri
!_______________________________________________________________________
!
   function TriMesh_fac( tri_connect,                                   &
                         icol, linewidth, height,                       &
                         tri_num, nod_num, fac_num,                     &
                         boundary_only )                                &
   result( handle )

      type(mfTriConnect),   intent(in) :: tri_connect
      integer,              intent(in) :: icol
      real(kind=MF_DOUBLE), intent(in) :: linewidth, height
      logical,              intent(in) :: tri_num, nod_num, fac_num,    &
                                          boundary_only

      integer :: handle
      !------ API end ------

      ! From tri_connect, we need only the following components: n_xy, face_n

      real(kind=MF_DOUBLE), pointer :: xy_sto_pg(:,:)
      integer, pointer :: face_sto_pg(:,:), tri_sto_pg(:,:), perm_face(:)

      integer :: nn, nf, ntri, nf_edge, nn_edge, nb_subparts
      integer :: ier
      logical :: find_in_col

      integer, allocatable :: coo_i(:), coo_j(:)
      real(kind=MF_DOUBLE), allocatable :: coo_val(:)
      integer :: nz

      ! continuous paths through nearly all the faces
      type(mf_Int_List), allocatable :: connected_faces(:)
      integer, allocatable :: conn_fac_sizes(:)
      integer :: i, j, k, k_tmp, kk1, kk2, i_edge, i1, i2
      integer :: f1, f2, n1, n2, nf_old, ni, nj, itmp

      real(kind=MF_DOUBLE), pointer :: xy_sto_edge(:,:)
      integer, pointer :: list_nod_nums(:), list_end_subparts(:),       &
                          list_fac_nums(:)
      type(mfArray) :: tri_sp, mf_list_end_subparts

      character(len=*), parameter :: ROUTINE_NAME = "mf/msTriMesh"

   !------ end of declarations -- execution starts hereafter  ------

      handle = 0

      if( .not. tri_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'tri_connect' doesn't seem to be a valid triangle connectivity!", &
                            "-> did you initialize it using msBuildTriConnect?" )
         go to 99
      end if

      nn = size( tri_connect%n_xy, 1 )

      if( boundary_only ) then

         ! Get all the faces of the boundaries. In the 'face_tri' table,
         ! the second index should be negative or zero for such faces.
         nf = size( tri_connect%face_tri, 1 )

         ! Count edge faces
         nf_edge = 0
         do i = 1, nf
            if( tri_connect%face_tri(i,2) <= 0 ) then
               nf_edge = nf_edge + 1
            end if
         end do

         ! Fill a tri_sp structure
         ! (same way as in routine 'TriMesh_tri', but here we store
         !  the face number instead of '1')
         allocate( coo_i(nf_edge), coo_j(nf_edge), coo_val(nf_edge) )
         nz = 0
         do i = 1, nf
            if( tri_connect%face_tri(i,2) <= 0 ) then
               ! get the nodes of this face
               n1 = tri_connect%face_n(i,1)
               n2 = tri_connect%face_n(i,2)
               nz = nz + 1
               coo_i(nz) = n1
               coo_j(nz) = n2
               coo_val(nz) = dble(i)
            end if
         end do

         tri_sp = mfSpImport( coo_i, coo_j, coo_val, nn, nn )
         deallocate( coo_i, coo_j, coo_val )

         ! For a closed curve, the number of nodes is equal to the
         ! number of edges.
         nn_edge = nf_edge

         ! Retrieve the boundary path, seeing 'tri_sp' as a matrix
         ! of adjacency.

         allocate( xy_sto_edge(nn_edge,2) )
         allocate( list_nod_nums(nn_edge) )
         allocate( list_fac_nums(nf_edge) )
#ifndef _OPTIM
xy_sto_edge(:,:) = MF_NAN
list_nod_nums(:) = 0
list_fac_nums(:) = 0
#endif

         ! Makes tri_sp symmetric
         tri_sp = tri_sp + .t. tri_sp

         mf_list_end_subparts = MF_EMPTY

         ! Search for continuous faces
         call init_search_faces( ibegin=1 )

         do i = 1, nf_edge-1 ! loop over all faces

            ! Record the current face
            if( i == 1 ) then
               xy_sto_edge(i+1,1) = tri_connect%n_xy(nj,1)
               xy_sto_edge(i+1,2) = tri_connect%n_xy(nj,2)
            else ! i /= 1
               if( ni == list_nod_nums(i) ) then
                  xy_sto_edge(i+1,1) = tri_connect%n_xy(nj,1)
                  xy_sto_edge(i+1,2) = tri_connect%n_xy(nj,2)
                  list_nod_nums(i+1) = nj
               else
                  xy_sto_edge(i+1,1) = tri_connect%n_xy(ni,1)
                  xy_sto_edge(i+1,2) = tri_connect%n_xy(ni,2)
                  list_nod_nums(i+1) = ni
               end if
            end if

            ! Find the next face
            if( find_in_col ) then
               kk1 = tri_sp%j(nj)
               kk2 = kk1 + 1
               ! elements of col nj are stored in a(kk1:kk2)
               if( tri_sp%a(kk1) > 0.0d0 ) then
                  ! kk1 is the new face
                  ni = tri_sp%i(kk1)
                  ! save save the face number
                  list_fac_nums(i+1) = tri_sp%a(kk1)
                  ! tag the face
                  tri_sp%a(kk1) = -1.0d0
               else if( tri_sp%a(kk2) > 0.0d0 ) then
                  ! kk2 is the new face
                  ni = tri_sp%i(kk2)
                  ! save save the face number
                  list_fac_nums(i+1) = tri_sp%a(kk2)
                  ! tag the face
                  tri_sp%a(kk2) = -1.0d0
               else
                  mf_list_end_subparts = mf_list_end_subparts .hc. mf(i)
                  ! removing all -1s in tri_sp
                  ! (no, the double statement below is not a joke!)
                  tri_sp = tri_sp + mfSpOnes(tri_sp)
                  tri_sp = tri_sp - mfSpOnes(tri_sp)
                  call init_search_faces( ibegin=i+1 )
                  cycle
               end if
               call msSet( -1.0d0, tri_sp, nj, ni )
            else ! find in row
               ! transpose the position (ni,nj)
               itmp = ni
               ni = nj
               nj = itmp
               ! now find in col
               kk1 = tri_sp%j(nj)
               kk2 = kk1 + 1
               if( tri_sp%a(kk1) > 0.0d0 ) then
                  ! kk1 is the new face
                  ni = tri_sp%i(kk1)
                  ! save save the face number
                  list_fac_nums(i+1) = tri_sp%a(kk1)
                  ! tag the face
                  tri_sp%a(kk1) = -1.0d0
               else if( tri_sp%a(kk2) > 0.0d0 ) then
                  ! kk2 is the new face
                  ni = tri_sp%i(kk2)
                  ! save save the face number
                  list_fac_nums(i+1) = tri_sp%a(kk2)
                  ! tag the face
                  tri_sp%a(kk2) = -1.0d0
               else
                  mf_list_end_subparts = mf_list_end_subparts .hc. mf(i)
                  ! removing all -1s in tri_sp
                  ! (no, the double statement below is not a joke!)
                  tri_sp = tri_sp + mfSpOnes(tri_sp)
                  tri_sp = tri_sp - mfSpOnes(tri_sp)
                  call init_search_faces( ibegin=i+1 )
                  cycle
               end if
               call msSet( -1.0d0, tri_sp, nj, ni )
               ! transpose back the position
               itmp = ni
               ni = nj
               nj = itmp
            end if

            find_in_col = .not. find_in_col

         end do
         mf_list_end_subparts = mf_list_end_subparts .hc. mf(nf_edge)

         nb_subparts = size( mf_list_end_subparts )
         allocate( list_end_subparts(nb_subparts) )
         list_end_subparts(:) = mf_list_end_subparts

         call msRelease( tri_sp, mf_list_end_subparts )

         handle = MeshBoundary_unstruct( xy_sto_edge, list_end_subparts, &
                                         list_nod_nums,                 &
                                         icol, linewidth, height,       &
                                         nod_num, list_fac_nums, fac_num )

      else !############################################################

!### TODO: required?
         if( tri_connect%face_oriented ) then
!!            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
!!                               "'tri_connect' cannot be used in this routine when faces are oriented!", &
!!                               "-> recall msBuildTriConnect with 'check_face_orient'=.false." )
!!            go to 99
pause "msTriMesh: Warning: I don't follow the requirement about 'check_face_orient'=.false."
         end if

         allocate( xy_sto_pg(nn,2) )
         xy_sto_pg = tri_connect%n_xy

         nf = size( tri_connect%face_n, 1 )
         allocate( face_sto_pg(nf,2) )
         ! face_sto_pg = tri_connect%face_n      was the old way, not used now

         ! this permutation keep in memory the change in face numbering
         allocate( perm_face(nf) )

         ! Storing faces in a special way:
         !  1) they are ordered in such a way that they form multiple
         !     continuous paths through the domain
         !  2) their orientation may be changed also, to satisfy (1).
         call msTriNodeNeighbors( tri_connect, connected_faces=connected_faces )
         allocate( conn_fac_sizes(nn) )
         do i = 1, nn
            conn_fac_sizes(i) = size( connected_faces(i)%list )
         end do

         nf_old = nf
         nf = 0
         do while( .not. conn_fac_sizes_empty() )

            ! select the first item in the first non-empty list
            do i = 1, nn
               if( conn_fac_sizes(i) /= 0 ) exit
            end do
            f1 = connected_faces(i)%list(1)
            n1 = tri_connect%face_n(f1,1)
            n2 = tri_connect%face_n(f1,2)
            nf = nf + 1
            face_sto_pg(nf,1) = n1
            face_sto_pg(nf,2) = n2
            perm_face(nf) = f1

            call build_cont_path()

         end do
         if( nf /= nf_old ) then
            print *, "(Muesli FGL:) internal error"
            print *, "  msTriMesh_fac: nf computed from building the continuous"
            print *, "  paths doesn't match the true value of nf."
            stop
         end if

         ntri = size( tri_connect%tri_n, 1 )
         allocate( tri_sto_pg(ntri,3) )
         tri_sto_pg = tri_connect%tri_n

         handle = MeshTri_fac( xy_sto_pg, face_sto_pg, perm_face,       &
                               tri_sto_pg, icol, linewidth, height,     &
                               tri_num, nod_num, fac_num )

      end if

 99   continue

   contains
   !____________________________________________________________________
   !
      subroutine build_cont_path()

         !------ API end ------

         logical :: found
         integer :: j, f2, n, nb_faces_in_path
         integer :: nn1, nn2, itmp

         ! the loop needs f1, n1, n2
         do
            ! remove f1 from the list based on n1
            !   locate f1 in the list
            n = conn_fac_sizes(n1)
            found = .false.
            do j = 1, n
               if( connected_faces(n1)%list(j) == f1 ) then
                  found = .true.
                  exit
               end if
            end do
            if( .not. found ) then
               print *, "(Muesli FGL:) internal error"
               print *, "  build_cont_path: f1 not found in list of connected"
               print *, "  faces based on n1"
               stop
            end if
            !   shift the part of the list after j (if not the last item)
            if( j < n ) then
               connected_faces(n1)%list(j:n-1) = connected_faces(n1)%list(j+1:n)
            end if
            connected_faces(n1)%list(n) = 0
            conn_fac_sizes(n1) = conn_fac_sizes(n1) - 1

            ! remove f1 from the list based on n2, and take its successor
            !   locate f1 in the list
            n = conn_fac_sizes(n2)
            found = .false.
            do j = 1, n
               if( connected_faces(n2)%list(j) == f1 ) then
                  found = .true.
                  exit
               end if
            end do
            if( .not. found ) then
               print *, "(Muesli FGL:) internal error"
               print *, "  build_cont_path: f1 not found in list of connected"
               print *, "  faces based on n2"
               stop
            end if
            !   shift the part of the list after j (if not the last item)
            if( j == n ) then
               f2 = connected_faces(n2)%list(1)
            else
               f2 = connected_faces(n2)%list(j+1)
               connected_faces(n2)%list(j:n-1) = connected_faces(n2)%list(j+1:n)
            end if
            connected_faces(n2)%list(n) = 0
            conn_fac_sizes(n2) = conn_fac_sizes(n2) - 1
            if( f2 == 0 .or. n == 1 ) then
               exit
            else
               nn1 = tri_connect%face_n(f2,1)
               nn2 = tri_connect%face_n(f2,2)
               ! change face orientation to enforce continuity in the path
               if( nn1 /= face_sto_pg(nf,2) ) then
                  itmp = nn1
                  nn1 = nn2
                  nn2 = itmp
               end if
               nf = nf + 1
               face_sto_pg(nf,1) = nn1
               face_sto_pg(nf,2) = nn2
               perm_face(nf) = f2
            end if
            n1 = n2
            ! find n2
            if( tri_connect%face_n(f2,1) == n1 ) then
               n2 = tri_connect%face_n(f2,2)
            else
               n2 = tri_connect%face_n(f2,1)
            end if
            f1 = f2
         end do

      end subroutine build_cont_path
   !____________________________________________________________________
   !
      function conn_fac_sizes_empty() result( bool )

         logical :: bool
         !------ API end ------

         bool = all( conn_fac_sizes(:) == 0 )

      end function conn_fac_sizes_empty
   !____________________________________________________________________
   !
      subroutine init_search_faces( ibegin )

         integer, intent(in) :: ibegin
         !------ API end ------

         integer :: i, j

         ! Found the first element available in tri_sp
         do j = 1, nn ! loop over columns
            if( tri_sp%j(j+1) == tri_sp%j(j) ) then
               ! no element in this column
               cycle
            end if
            ni = tri_sp%i(1)
            nj = j
            exit
         end do
         i = ibegin
         ! save save the face number
         list_fac_nums(i) = tri_sp%a(1)
         ! tag the face and it's symmetric
         tri_sp%a(1) = -1.0d0
         call msSet( -1.0d0, tri_sp, nj, ni )
         xy_sto_edge(i,1) = tri_connect%n_xy(ni,1)
         xy_sto_edge(i,2) = tri_connect%n_xy(ni,2)
         list_nod_nums(i) = ni
         list_nod_nums(i+1) = nj

         find_in_col = .true.

      end subroutine init_search_faces
   !____________________________________________________________________
   !
   end function TriMesh_fac
!_______________________________________________________________________
!
   function MeshTri_tri( xy_sto_pg, tri_sto_pg,                         &
                         icol, linewidth, height,                       &
                         tri_num, nod_num )                             &
   result( handle )

      real(kind=MF_DOUBLE), pointer    :: xy_sto_pg(:,:)
      integer,              pointer    :: tri_sto_pg(:,:)
      integer,              intent(in) :: icol
      real(kind=MF_DOUBLE), intent(in) :: linewidth, height
      logical,              intent(in) :: tri_num, nod_num

      integer :: handle
      !------ API end ------

      type(mf_win_info), pointer :: win
      type(grobj_elem), pointer :: grobj

      real(kind=MF_DOUBLE) :: range(4)

      integer :: ntri, nn, hdle, icol_txt

      integer :: itmp
      character(len=3) :: answer
      logical :: device_has_cursor

   !------ end of declarations -- execution starts hereafter  ------

      ! les [nouveaux] axes doivent être prêts
      range(1) = minval( xy_sto_pg(:,1) )
      range(2) = maxval( xy_sto_pg(:,1) )
      range(3) = minval( xy_sto_pg(:,2) )
      range(4) = maxval( xy_sto_pg(:,2) )
      call mf_prepare_axes( CURRENT_WIN_ID, range )

      ! plotting something is always clipped at viewport
      if( X11_DEVICE ) then
         ! caution: axes must have been defined before
         call X11_clip_on_viewport()
      end if

      ntri = size(tri_sto_pg,1)

      call pgbbuf()

      win => mf_win_db(CURRENT_WIN_ID)

!++++++++++++++++++ Storage in DB ++++++++++++++++++

      ! new grobj
      if( win%mf_win_db_active ) then
         ! create a new grobj and insert it in the linked list
         call create_grobj( win, grobj )
      else
         ! just allocate the grobj
         allocate( grobj )
      end if

      grobj%struct%cmd = "trimesh_by_tri"
      grobj%struct%range = range
      grobj%struct%abs_mat => xy_sto_pg
      grobj%struct%tab_2d_1 => tri_sto_pg
      grobj%struct%npt = ntri
      grobj%struct%color = icol
      grobj%struct%linewidth = linewidth
      grobj%struct%height_text = height
      grobj%struct%bool1 = tri_num
      grobj%struct%bool2 = nod_num

      if( tri_num ) then
         ! Write the triangle number
         call decode_col_rgb( [0.0d0,0.0d0,0.6d0], icol_txt )
         grobj%struct%npt2 = ntri
         grobj%struct%linestyle = icol_txt
      end if

      if( nod_num ) then
         ! Write the node number
         nn = size(xy_sto_pg,1)
         call decode_col_rgb( [0.6d0,0.0d0,0.0d0], icol_txt )
         grobj%struct%npt3 = nn
         grobj%struct%marker = icol_txt
      end if

      if( win%mf_win_db_active ) then
         hdle = mf_win_get_free_handle(CURRENT_WIN_ID)
         win%handles(hdle)%ptr => grobj
         grobj%struct%hdle = hdle
         handle = encode_handle( CURRENT_WIN_ID, hdle )
      end if

!+++++++++++++++++++++++++++++++++++++++++++++++++++

      win%blank = .false.
      win%empty = .false.

!------------------ Drawing GrObj ------------------

      ! inquiring if the device has a cursor
      call pgqinf( "CURSOR", answer, itmp )
      if( to_lower(answer) == "yes" ) then
         device_has_cursor = .true.
         itmp = gr_set_cursor_shape( MF_WATCH_CURSOR )
      else
         device_has_cursor = .false.
      end if

      call mf_trimesh_by_tri_draw( grobj )

      call pgebuf()

      if( device_has_cursor ) then
         itmp = gr_set_cursor_shape( MF_LEFT_ARROW_CURSOR )
      end if

!---------------------------------------------------

      if( .not. win%mf_win_db_active ) then
         call delete_grobj_inside( grobj )
         deallocate( grobj )
      end if

   end function MeshTri_tri
!_______________________________________________________________________
!
   function MeshTri_fac( xy_sto_pg, face_sto_pg, perm_face, tri_sto_pg, &
                         icol, linewidth, height,                       &
                         tri_num, nod_num, fac_num )                    &
   result( handle )

      real(kind=MF_DOUBLE), pointer    :: xy_sto_pg(:,:)
      integer,              pointer    :: face_sto_pg(:,:), perm_face(:)
      integer,              pointer    :: tri_sto_pg(:,:)
      integer,              intent(in) :: icol
      real(kind=MF_DOUBLE), intent(in) :: linewidth, height
      logical,              intent(in) :: tri_num, nod_num, fac_num

      integer :: handle
      !------ API end ------

      type(mf_win_info), pointer :: win
      type(grobj_elem), pointer :: grobj

      real(kind=MF_DOUBLE) :: range(4)
      integer :: nf, nn, hdle
      integer :: ntri, icol_txt

      integer :: itmp
      character(len=3) :: answer
      logical :: device_has_cursor

   !------ end of declarations -- execution starts hereafter  ------

      ! les [nouveaux] axes doivent être prêts
      range(1) = minval( xy_sto_pg(:,1) )
      range(2) = maxval( xy_sto_pg(:,1) )
      range(3) = minval( xy_sto_pg(:,2) )
      range(4) = maxval( xy_sto_pg(:,2) )
      call mf_prepare_axes( CURRENT_WIN_ID, range )

      ! plotting something is always clipped at viewport
      if( X11_DEVICE ) then
         ! caution: axes must have been defined before
         call X11_clip_on_viewport()
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      nf = size(face_sto_pg,1)

!++++++++++++++++++ Storage in DB ++++++++++++++++++

      ! new grobj
      if( win%mf_win_db_active ) then
         ! create a new grobj and insert it in the linked list
         call create_grobj( win, grobj )
      else
         ! just allocate the grobj
         allocate( grobj )
      end if

      grobj%struct%cmd = "trimesh_by_fac"
      grobj%struct%range = range
      grobj%struct%abs_mat => xy_sto_pg
      grobj%struct%tab_2d_1 => face_sto_pg
      grobj%struct%ir => perm_face
      grobj%struct%color = icol
      grobj%struct%npt = nf
      grobj%struct%linewidth = linewidth
      grobj%struct%height_text = height
      grobj%struct%bool1 = tri_num
      grobj%struct%bool2 = nod_num
      grobj%struct%bool3 = fac_num

      if( tri_num ) then
         ntri = size(tri_sto_pg,1)
         call decode_col_rgb( [0.0d0,0.0d0,0.6d0], icol_txt )
         grobj%struct%tab_2d_2 => tri_sto_pg
         grobj%struct%npt2 = ntri
         grobj%struct%linestyle = icol_txt
      else
         deallocate( tri_sto_pg )
      end if

      if( nod_num ) then
         nn = size(xy_sto_pg,1)
         call decode_col_rgb( [0.6d0,0.0d0,0.0d0], icol_txt )
         grobj%struct%npt3 = nn
         grobj%struct%marker = icol_txt
      end if

      if( fac_num ) then
         call decode_col_rgb( [0.0d0,0.6d0,0.0d0], icol_txt )
         grobj%struct%alpha_transp = icol_txt
      end if

      if( win%mf_win_db_active ) then
         hdle = mf_win_get_free_handle(CURRENT_WIN_ID)
         win%handles(hdle)%ptr => grobj
         grobj%struct%hdle = hdle
         handle = encode_handle( CURRENT_WIN_ID, hdle )
      end if

!+++++++++++++++++++++++++++++++++++++++++++++++++++

      win%blank = .false.
      win%empty = .false.

!------------------ Drawing GrObj ------------------

      ! inquiring if the device has a cursor
      call pgqinf( "CURSOR", answer, itmp )
      if( to_lower(answer) == "yes" ) then
         device_has_cursor = .true.
         itmp = gr_set_cursor_shape( MF_WATCH_CURSOR )
      else
         device_has_cursor = .false.
      end if

      call mf_trimesh_by_fac_draw( grobj )

      call pgebuf()

      if( device_has_cursor ) then
         itmp = gr_set_cursor_shape( MF_LEFT_ARROW_CURSOR )
      end if

!---------------------------------------------------

      if( .not. win%mf_win_db_active ) then
         call delete_grobj_inside( grobj )
         deallocate( grobj )
      end if

   end function MeshTri_fac
!_______________________________________________________________________
!
   function MeshBoundary_unstruct( xy_sto_edge, list_end_subparts,      &
                                   list_nod_nums,                       &
                                   icol, linewidth, height, nod_num,    &
                                   list_fac_nums, fac_num )             &
   result( handle )

      real(kind=MF_DOUBLE), pointer              :: xy_sto_edge(:,:)
      integer,              pointer              :: list_end_subparts(:), &
                                                    list_nod_nums(:)
      integer,              intent(in)           :: icol
      real(kind=MF_DOUBLE), intent(in)           :: linewidth, height
      logical,              intent(in)           :: nod_num
      integer,              pointer,    optional :: list_fac_nums(:)
      logical,              intent(in), optional :: fac_num

      integer :: handle
      !------ API end ------

      type(mf_win_info), pointer :: win
      type(grobj_elem), pointer :: grobj

      real(kind=MF_DOUBLE) :: range(4)

      integer :: nf_edge, hdle, icol_txt

      integer :: itmp
      character(len=3) :: answer
      logical :: device_has_cursor

   !------ end of declarations -- execution starts hereafter  ------

      ! les [nouveaux] axes doivent être prêts
      range(1) = minval( xy_sto_edge(:,1) )
      range(2) = maxval( xy_sto_edge(:,1) )
      range(3) = minval( xy_sto_edge(:,2) )
      range(4) = maxval( xy_sto_edge(:,2) )
      call mf_prepare_axes( CURRENT_WIN_ID, range )

      ! plotting something is always clipped at viewport
      if( X11_DEVICE ) then
         ! caution: axes must have been defined before
         call X11_clip_on_viewport()
      end if

      nf_edge = size(xy_sto_edge,1)

      call pgbbuf()

      win => mf_win_db(CURRENT_WIN_ID)

!++++++++++++++++++ Storage in DB ++++++++++++++++++

      ! new grobj
      if( win%mf_win_db_active ) then
         ! create a new grobj and insert it in the linked list
         call create_grobj( win, grobj )
      else
         ! just allocate the grobj
         allocate( grobj )
      end if

      grobj%struct%cmd = "mesh_boundary_unstruct"
      grobj%struct%range = range
      grobj%struct%abs_mat => xy_sto_edge
      grobj%struct%ir => list_nod_nums
      grobj%struct%jc => list_end_subparts
      if( present(list_fac_nums) ) then
         grobj%struct%col_tab => list_fac_nums
      else
         grobj%struct%col_tab => null()
      end if
      grobj%struct%npt = nf_edge
      grobj%struct%color = icol
      grobj%struct%linewidth = linewidth
      grobj%struct%height_text = height
      grobj%struct%bool1 = nod_num
      if( present(fac_num) ) then
         grobj%struct%bool2 = fac_num
      else
         grobj%struct%bool2 = .false.
      end if

      if( nod_num ) then
         call decode_col_rgb( [0.6d0,0.0d0,0.0d0], icol_txt )
         grobj%struct%marker = icol_txt
      end if

      if( grobj%struct%bool2 ) then
         call decode_col_rgb( [0.0d0,0.6d0,0.0d0], icol_txt )
         grobj%struct%alpha_transp = icol_txt
      end if

      if( win%mf_win_db_active ) then
         hdle = mf_win_get_free_handle(CURRENT_WIN_ID)
         win%handles(hdle)%ptr => grobj
         grobj%struct%hdle = hdle
         handle = encode_handle( CURRENT_WIN_ID, hdle )
      end if

!+++++++++++++++++++++++++++++++++++++++++++++++++++

      win%blank = .false.
      win%empty = .false.

!------------------ Drawing GrObj ------------------

      ! inquiring if the device has a cursor
      call pgqinf( "CURSOR", answer, itmp )
      if( to_lower(answer) == "yes" ) then
         device_has_cursor = .true.
         itmp = gr_set_cursor_shape( MF_WATCH_CURSOR )
      else
         device_has_cursor = .false.
      end if

      call mf_boundary_mesh_unstruct_draw( grobj )

      call pgebuf()

      if( device_has_cursor ) then
         itmp = gr_set_cursor_shape( MF_LEFT_ARROW_CURSOR )
      end if

!---------------------------------------------------

      if( .not. win%mf_win_db_active ) then
         call delete_grobj_inside( grobj )
         deallocate( grobj )
      end if

   end function MeshBoundary_unstruct
