!_______________________________________________________________________
!
   subroutine msImWrite( A, cmap, filename, fmt, indexed )

      type(mfArray), intent(in) :: A, cmap
      character(len=*), intent(in) :: filename
      character(len=*), intent(in), optional :: fmt
      logical,          intent(in), optional :: indexed
      !------ API end ------

      ! Two image indexing schemes:
      !
      ! 1) real valued pixels
      !    (default one or if 'indexed_image'=.false.)
      !    => (fractional) real values are stored in 'array';
      !       these real values are ranged in [0.0,1.0].
      !       Transparent pixels correspond to a NaN value.
      !
      ! 2) indexed pixels
      !    => integer values are stored in 'array';
      !       these indexes are ranged in [1,N], where N is the number
      !       of colors in the colormap (including the NONE color for
      !       transparent pixels).
      !       NONE color is coded as the triplet (R,G,B) = (NaN,NaN,NaN),
      !       at the beginning of the colormap.

      ! supported image formats are :
      !
      ! XPM + any other format supported by 'convert' of ImageMagick

      ! pointers for manipulating mfArray out of fml module
      real(kind=MF_DOUBLE), pointer :: A_f90(:,:)

      logical :: has_extension, indexed_image
      integer :: length, i
      character(len=80) :: imagename
      character(len=9) :: fmt_img

      character(len=*), parameter :: ROUTINE_NAME = "msImWrite"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, cmap )

      ! some checks
      if( mfIsEmpty(A) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "image A is empty!" )
         return
      end if

      if( mfIsEmpty(cmap) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "colormap cmap is empty!" )
         return
      end if

      if( size(cmap,2) /= 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "second dimension of the colormap cmap is not 3!" )
         return
      end if

      ! quick access to internal structure of the mfArray 'A'
      call msPointer( A, A_f90, no_crc=.true. )

      ! further checks according to the indexing scheme
      if( present(indexed) ) then
         indexed_image = indexed
         ! in the 'indexed pixels' indexing scheme, the value of
         ! all elements of the image 'A' should be integers greater than 1
         if( any(dble(nint(A_f90)) /= A_f90) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "elements of image A must be integers greater", &
                               "than or equal to one!" )
            return
         end if
      else
         indexed_image = .false.
         ! in the 'real valued pixels' indexing scheme, the value of
         ! all elements of the image 'A' should be in [0.,1.]
         if( minval(A_f90) < 0.0d0 .or. 1.0d0 < maxval(A_f90) ) then
            ! help the user in detecting that all elements are integers,
            ! and in such a case, the user perhaps used the 'indexed pixels'
            ! indexing scheme instead...
            if( all(dble(nint(A_f90)) == A_f90) ) then
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "elements of image A must be ranged in [0.,1.]!", &
                                  "(in your case, all elements are integer, so it may", &
                                  " indicate that you are using the 'indexed pixels'", &
                                  " storing scheme?)" )
            else
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "elements of image A must be ranged in [0.,1.]!" )
            end if
            return
         end if
         ! the colormap 'cmap' cannot contain NaN values
         if( any(mfIsNaN(cmap)) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "In the 'real valued pixels' indexing scheme,", &
                               "elements of colormap 'cmap' cannot be NaN!" )
         end if
      end if

      length = len_trim(filename)

      has_extension = .false.

      if( present(fmt) ) then
         fmt_img = to_lower(fmt)
      else
         ! image format is based on extension
         i = index( trim(filename), "/", back=.true. )
         if( i /= 0 ) then
            imagename = filename(i+1:length)
         else
            imagename = filename
         end if
         i = index( trim(imagename), ".", back=.true. )
         if( i == 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "image format not detected!",            &
                               "You should supply the format." )
            return
         end if
         has_extension = .true.
         fmt_img = to_lower( imagename(i+1:length) )
      end if

         select case( to_lower(fmt_img) )
         case( "xpm" )
            call write_xpm( A, cmap, filename, indexed_image )
         case default
#ifdef _HAS_IM_CONVERT
            call write_any( A, cmap, filename, indexed_image )
#else
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "image format not supported!",           &
                               "You should install the ImageMagick package." )
#endif
      end select

      call msFreeArgs( A, cmap )
      call msAutoRelease( A, cmap )

      call msFreePointer( A, A_f90 )

   end subroutine msImWrite
