!_______________________________________________________________________
!
   function mfGinput( event, key )                                      &
   result ( out )

      logical,     intent(in), optional :: event
      logical,     intent(in), optional :: key
      type(mfArray)                     :: out
      !------ API end ------

      ! Returns in the mfArray 'out' only the coordinates (x,y).
      !
      ! See the subroutine version for other arguments or other returned
      ! values.
      !
      ! Usually ('event'=TRUE, which is the default), the routine waits for
      ! an event (key pressed or mouse click) before returning the pointer
      ! position. When 'event'=FALSE, the routine returns immediately, and
      ! returns in 'out' the 3 following values:
      !   (x,y)  pointer coordinates
      !      z   a flag indicating if the pointer is inside the window (1)
      !          or outside (0).
      ! Moreover, if 'key' is present and equal to TRUE (in addition to
      ! 'event'=FALSE), 'out' contains a 4th element, which is ichar(key),
      ! the key pressed.
      !
      !---------------------------------------------------------------------
      ! Implementation notes:
      ! If used with the keyboard only, two X11 events occur: indeed,
      ! there are two call to 'pgband' (the first one for MouseDown, the
      ! second one for MouseUp).

      real(kind=MF_DOUBLE) :: xref, yref, xc, yc
      character :: ch
      character(len=4) :: win_id_char
      integer :: itmp, inside
      logical :: wait_event, key_returned
      integer :: axis_manual_save_x, axis_manual_save_y

      character(len=*), parameter :: ROUTINE_NAME = "mfGinput"

      type(mf_win_info), pointer :: win

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. X11_DEVICE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "X11 device is ""off"".",                   &
                            "('" // trim(ROUTINE_NAME) // "' is an interactive routine which requires X11)" )
         return
      end if

      if( CURRENT_WIN_ID == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "no X11 device selected." )
         return
      end if

      if( present(event) ) then
         wait_event = event
         if( present(key) ) then
            key_returned = key
         else
            key_returned = .false.
         end if
      else
         if( present(key) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'key' optional arg cannot be present when 'event' is not present." )
            return
         end if
         wait_event = .true.
      end if

      call set_focus_mfplot_win( 1 )

      if( wait_event ) then

         ! usual call

         win => mf_win_db(CURRENT_WIN_ID)
         if( all( win%current_axes(:) == 0.0d0 ) ) then
            ! set the whole graphic env, keeping the 'manual' status
            axis_manual_save_x = win%axis_manual_x
            axis_manual_save_y = win%axis_manual_y
            call msAxis( [ 0.0d0, 1.0d0, 0.0d0, 1.0d0 ] )
            win%axis_manual_x = axis_manual_save_x
            win%axis_manual_y = axis_manual_save_y
         end if

         write(STDOUT,*)
         write(STDOUT,"(A,I0,A)") "  Entering interactive mode in Figure ", &
                                  CURRENT_WIN_ID, " ..."
         write(STDOUT,"(A,I0,A)") '  (2 events expected; any key different to "L", "M" or "R" to quit)'
         write(STDOUT,*) " [click with the mouse to get a point]"
         write(STDOUT,*)

         call grsci(1) ! std foreground
         call grsls(1) ! continuous line

         ! avoid passing NaN values to MFPLOT
         xref = 0.
         yref = 0.
         xc = 0.
         yc = 0.

         ! wait for a click (mouse down or equivalent key pressed)
         if( pgband(MF_CROSSHAIR_CURSOR,7,0,xref,yref,xc,yc,ch,         &
                    .false., win%axis_scale_x, win%axis_scale_y) == 0 ) then
            write(win_id_char,"(I0)") CURRENT_WIN_ID
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "no cursor device for win ID: " // win_id_char )
            return
         end if

         ch = to_upper(ch)
         if( ch=="L" .or. ch=="M" .or. ch=="R" ) then
            ! wait for a click (mouse up)
            if( pgband(MF_CROSSHAIR_CURSOR,7,0,xref,yref,xc,yc,ch,      &
                       .false., win%axis_scale_x, win%axis_scale_y) == 0 ) then
               write(win_id_char,"(I0)") CURRENT_WIN_ID
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "no cursor device for win ID: " // win_id_char )
               return
            end if
            out = [ xc, yc ]
            if( win%axis_scale_x == 2 ) then
               call msSet( 10.0d0**xc, out, 1 )
            else if( win%axis_scale_x == 0 ) then
               write(STDERR,*) "(MUESLI mfGinput:) internal error: x-axis scale undefined!"
               call msMuesliTrace( pause="no" )
               stop
            end if
            if( win%axis_scale_y == 2 ) then
               call msSet( 10.0d0**yc, out, 2 )
            else if( win%axis_scale_y == 0 ) then
               write(STDERR,*) "(MUESLI mfGinput:) internal error: y-axis scale undefined!"
               call msMuesliTrace( pause="no" )
               stop
            end if
         end if

         ! if ESCAPE key has been pressed, cursor shape is changed back to
         ! normal in the X11 driver (xw_read_cursor routine in x11_driver.c)
         if( ichar(ch) /= 27 ) then ! ESCAPE key not pressed
            ! change cursor shape to normal
            if( grnormcurs_ec() == 0 ) then
               write(win_id_char,"(I0)") CURRENT_WIN_ID
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "no cursor device for win ID: " // trim(adjustl(win_id_char)) )
               return
            end if
         end if

         itmp = gr_set_cursor_shape( MF_LEFT_ARROW_CURSOR )

         call set_focus_mfplot_win( 0 )

      else

         if( key_returned ) then
            if( pginput(xc,yc,inside,ch) == 0 ) then
               write(win_id_char,"(I0)") CURRENT_WIN_ID
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "no cursor device for win ID: " // win_id_char )
               return
            end if
            out = [ xc, yc, dble(inside), dble(ichar(ch)) ]
         else
            if( pginput(xc,yc,inside) == 0 ) then
               write(win_id_char,"(I0)") CURRENT_WIN_ID
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "no cursor device for win ID: " // win_id_char )
               return
            end if
            out = [ xc, yc, dble(inside) ]
         endif

      end if

      call msReturnArray( out ) ! set to tempo

   end function mfGinput
!_______________________________________________________________________
!
   subroutine msGinput( out, whole_keyboard, rect_size, rect_inside_axes, &
                        event )

      type(mf_Out) :: out
      logical,       intent(in), optional :: whole_keyboard
      type(mfArray), intent(in), optional :: rect_size
      logical,       intent(in), optional :: rect_inside_axes
      logical,       intent(in), optional :: event
      !------ API end ------

      ! Values returned (via mfOut):
      !
      !   coords : coordinates (x,y) at the pointer location when it has
      !            been clicked;
      !
      !   keycode : if whole_keyboard=.false. [default behavior]
      !               keycode = mouse button number (1=left, 2=middle, 3=right)
      !               [even the keyboard keys 'L', 'M', 'R' have been used;
      !                but keycode is EMPTY if any key (other than 'L', 'M'
      !                or 'R') is pressed]
      !             if whole_keyboard=.true.
      !               keycode = ichar(key_pressed)
      !                                  internal
      !                                 MFPLOT val.    ichar(key_pressed)
      !                    wheel up   =   '8'     =>        56
      !                    wheel down =   '2'     =>        50
      !
      !   color (optional) : the triplet [R,G,B] of the pixel just clicked
      !                      by the pointer
      !
      ! Usually (if 'rect_size' is not present), the cursor shown on the
      ! screen has the shape of a big crosshair, whose width and height are
      ! those of the whole selected window; if 'rect_size' is present and
      ! not empty, this mfArray gives the size (width and height) of a
      ! rectangle which follows the pointer during its move.
      !
      ! If 'rect_size' is present, the optional argument 'rect_inside_axes'
      ! is a logical which constrains 'rect_size' to be entirely located
      ! inside the current axes.
      !
      !---------------------------------------------------------------------
      ! Implementation notes:
      ! If use_key is TRUE, only one X11 event occurs (one call to the MFPLOT
      ! 'pgband' routine); on the contrary, two X11 events occur: indeed,
      ! there are two call to 'pgband' (the first one for MouseDown, the
      ! second one for MouseUp).

      type(mfArray), pointer :: coords, keycode, color

      real(kind=MF_DOUBLE) :: xref, yref, xc, yc, width, height, wmax, hmax
      real(kind=MF_DOUBLE) :: red, green, blue
      character :: ch, ch_down
      character(len=4) :: win_id_char
      logical :: wholeKeyboard, rectangle, rectangle_inside_axes
      integer :: itmp
      logical :: wait_event

      character(len=*), parameter :: ROUTINE_NAME = "msGinput"

      type(mf_win_info), pointer :: win

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. X11_DEVICE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "X11 device is ""off"".",                   &
                            "('" // trim(ROUTINE_NAME) // "' is an interactive routine which requires X11)" )
         return
      end if

      if( CURRENT_WIN_ID == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "no X11 device selected." )
         return
      end if

      if( out%n < 2 .and. 3 < out%n ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "calling sequence is:",                     &
                            "call msGinput( mfOut(coords,keycode[,color]), ", &
                            "[whole_keyboard, rect_size, rect_inside_axes])" )
         return
      end if

      coords => out%ptr1
      coords = MF_EMPTY
      keycode => out%ptr2
      keycode = MF_EMPTY
      if( out%n >= 3 ) then
         color => out%ptr3
         color = MF_EMPTY
      end if

      if( present(whole_keyboard) ) then
         wholeKeyboard = whole_keyboard
      else
         wholeKeyboard = .false.
      end if

      if( present(rect_size) ) then
         call msInitArgs( rect_size )
         if( mfIsEmpty(rect_size) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "rect_size argument is present but is empty." )
            rectangle = .false.
         else
            width  = mfGet( rect_size, 1 )
            height = mfGet( rect_size, 2 )
            ! these variables must contain non negative numbers
            if( width <= 0.0d0 .or. height <= 0.0d0 ) then
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "rect_size contains null or negative data!" )
               return
            else
               rectangle = .true.
            end if
         end if
         if( present(rect_inside_axes) ) then
            rectangle_inside_axes = rect_inside_axes
         else
            rectangle_inside_axes = .false.
         end if
      else
         if( present(rect_inside_axes) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "rect_inside_axes cannot be present when", &
                               "rect_size is not present!" )
            return
         end if
         rectangle = .false.
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      if( present(rect_size) ) then
         if( win%axis_scale_x == 2 .or. win%axis_scale_y == 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "rect_size cannot be used in log scale!" )
            return
         end if
      end if

      if( present(event) ) then
         wait_event = event
      else
         wait_event = .true.
      end if

      call set_focus_mfplot_win( 1 )

      write(STDOUT,*)
      write(STDOUT,"(A,I0,A)") "  Entering interactive mode in Figure ", &
                               CURRENT_WIN_ID, " ..."
      if( wholeKeyboard ) then
         write(STDOUT,"(A,I0,A)") "  (1 event expected; ESC to quit)"
         write(STDOUT,*) " [press a key to get a point]"
      else
         write(STDOUT,"(A,I0,A)") '  (2 events expected; any key different to "L", "M" or "R" to quit)'
         write(STDOUT,*) " [click with the mouse to get a point]"
      end if
      write(STDOUT,*)

      call grsci(1) ! std foreground
      call grsls(1) ! continuous line

      ! avoid passing NaN values to MFPLOT
      xref = 0.
      yref = 0.
      xc = 0.
      yc = 0.

      ! wait for a click (mouse down or any key press)
      if( rectangle ) then
         xref = width
         yref = height
         ! MODE = 8 is a new option for drawing a rectangle around
         ! the current pointer position. Three args have been added to
         ! the PGBAND routine.
         if( pgband(MF_CROSSHAIR_CURSOR,8,0,xref,yref,xc,yc,ch,         &
                    rectangle_inside_axes, win%axis_scale_x, win%axis_scale_y ) == 0 ) then
            write(win_id_char,"(I0)") CURRENT_WIN_ID
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "no cursor device for win ID: " // win_id_char )
            return
         end if
      else
         if( pgband(MF_CROSSHAIR_CURSOR,7,0,xref,yref,xc,yc,ch,         &
                    .false., win%axis_scale_x, win%axis_scale_y) == 0 ) then
            write(win_id_char,"(I0)") CURRENT_WIN_ID
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "no cursor device for win ID: " // win_id_char )
            return
         end if
      end if

      if( wholeKeyboard ) then

         if( ichar(ch) /= 27 ) then ! ESCAPE key not pressed
            coords = [ xc, yc ]
            keycode = dble(ichar(ch))
         end if

      else

         ch = to_upper(ch)
         if( ch=="L" .or. ch=="M" .or. ch=="R" ) then ! ESCAPE key not pressed
            ch_down = ch
            ! wait for a click (mouse up)
            if( rectangle ) then
               xref = width
               yref = height
               ! MODE = 8 is a new option for drawing a rectangle around
               ! the current pointer position.
               if( pgband(MF_CROSSHAIR_CURSOR,8,0,xref,yref,xc,yc,ch,   &
                          .false., win%axis_scale_x, win%axis_scale_y) == 0 ) then
                  write(win_id_char,"(I0)") CURRENT_WIN_ID
                  call PrintMessage( trim(ROUTINE_NAME), "E",           &
                                     "no cursor device for win ID: " // win_id_char )
                  return
               end if
            else
               if( pgband(MF_CROSSHAIR_CURSOR,7,0,xref,yref,xc,yc,ch,   &
                          .false., win%axis_scale_x, win%axis_scale_y) == 0 ) then
                  write(win_id_char,"(I0)") CURRENT_WIN_ID
                  call PrintMessage( trim(ROUTINE_NAME), "E",           &
                                     "no cursor device for win ID: " // win_id_char )
                  return
               end if
            end if
            coords = [ xc, yc ]
            if( win%axis_scale_x == 2 ) then
               call msSet( 10.0d0**xc, coords, 1 )
            else if( win%axis_scale_x == 0 ) then
               write(STDERR,*) "(MUESLI msGinput:) internal error: x-axis scale undefined!"
               call msMuesliTrace( pause="no" )
               stop
            end if
            if( win%axis_scale_y == 2 ) then
               call msSet( 10.0d0**yc, coords, 2 )
            else if( win%axis_scale_y == 0 ) then
               write(STDERR,*) "(MUESLI msGinput:) internal error: y-axis scale undefined!"
               call msMuesliTrace( pause="no" )
               stop
            end if
            ch_down = to_upper(ch_down)
            select case( ch_down )
               case("L")
                  keycode = 1.0d0
               case("M")
                  keycode = 2.0d0
               case("R")
                  keycode = 3.0d0
            end select
            if( out%n == 3 ) then
               call GetOnePX( xc, yc, red, green, blue )
               color = [ red, green, blue ]
            end if
         end if

      end if

      ! if ESCAPE key has been pressed, cursor shape is changed back to
      ! normal in the X11 driver (xw_read_cursor routine in xwdriv.c)
      if( ichar(ch) /= 27 ) then ! ESCAPE key not pressed
         ! change cursor shape to normal
         if( grnormcurs_ec() == 0 ) then
            write(win_id_char,"(I0)") CURRENT_WIN_ID
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "no cursor device for win ID: " // trim(adjustl(win_id_char)) )
            return
         end if
      end if

      itmp = gr_set_cursor_shape( MF_LEFT_ARROW_CURSOR )

      if( present(rect_size) ) then
         call msFreeArgs( rect_size )
         call msAutoRelease( rect_size )
      end if

      call set_focus_mfplot_win( 0 )

   end subroutine msGinput
