!_______________________________________________________________________
!
   function mfColormap_noarg( ) result ( colormap )

      type(mfArray) :: colormap
      !------ API end ------

      ! pointers for manipulating mfArray out of fml module
      real(kind=MF_DOUBLE), pointer :: colormap_ptr(:,:)

      integer :: i, ncol, offset
      real(kind=MF_DOUBLE) :: r, g, b
      type(mf_win_info), pointer :: win

      character(len=*), parameter :: ROUTINE_NAME = "mfColormap"

   !------ end of declarations -- execution starts hereafter  ------

      if( CURRENT_WIN_ID == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "no selected device!" )
         return
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      if( .not. win%colormap_init ) then
         call PrintMessage( trim(ROUTINE_NAME), "I",                    &
                            "colormap is not yet initialized.",         &
                            "(selecting the default colormap)" )
         call msColormap( "rainbow" )
      end if

      offset = win%colormap_ci_low - 1
      ncol = win%colormap_ci_high - offset

      call msAssign( colormap, mfZeros(ncol,3) )
      call msPointer( colormap, colormap_ptr, no_crc=.true. )

      do i = 1, ncol
         call grqcr(offset+i, r, g, b)
         colormap_ptr(i,1) = r
         colormap_ptr(i,2) = g
         colormap_ptr(i,3) = b
      end do
      call msFreePointer( colormap, colormap_ptr )

      call msReturnArray( colormap ) ! set to tempo

   end function mfColormap_noarg
!_______________________________________________________________________
!
   function mfColormap_status( string ) result ( bool )

      character(len=*) :: string
      logical :: bool
      !------ API end ------

      type(mf_win_info), pointer :: win

      character(len=*), parameter :: ROUTINE_NAME = "mfColormap"

   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      if( CURRENT_WIN_ID == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "no selected device!" )
         return
      end if

      if( to_lower(trim(string)) /= "init_status" ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            'bad arg ! (did you thought "init_status"?)' )
         return
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      if( win%colormap_init ) then
         bool = .true.
      end if

   end function mfColormap_status
!_______________________________________________________________________
!
   subroutine msColormap_string( name, inverted )

      character(len=*), intent(in)           :: name
      character(len=*), intent(in), optional :: inverted
      !------ API end ------

      logical :: invert

      type(mf_win_info), pointer :: win

      character(len=*), parameter :: ROUTINE_NAME = "msColormap"

   !------ end of declarations -- execution starts hereafter  ------

      if( CURRENT_WIN_ID == 0 ) then
         call msFigure()
         if( CURRENT_WIN_ID == 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "cannot plot: no window created!" )
            return
         end if
      end if

      if( present(inverted) ) then
         if( to_lower(trim(inverted)) /= "inverted" ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               'bad arg ! (did you thought "inverted"?)' )
            return
         end if
         invert = .true.
      else
         invert = .false.
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      select case( to_lower(name) )
         case( "grey" )
            call set_predefined_colormap( 0, invert )
         case( "fusion" )
            ! The old name was "krahina"; this new improved colormap
            ! comes from the FLIR software and has more contrasted colors.
            call set_predefined_colormap( 1, invert )
         case( "bluered" )
            call set_predefined_colormap( 2, invert )
         case( "rainbow", "jet" )
            ! "jet" is also the name used by Matlab.
            call set_predefined_colormap( 3, invert )
         case( "hot" )
            call set_predefined_colormap( 4, invert )
         case( "parula" )
            call set_predefined_colormap( 5, invert )
         case( "flag" )
            call set_predefined_colormap( 6, invert )
         case default
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "bad colormap name!",                    &
                 "('" // trim(name) // "' is not a predefined colormap)" )
            return
      end select
      win%colormap_name = to_lower(name)
      win%colormap_inverted = invert

      call pgscir( win%colormap_ci_low, win%colormap_ci_high )

      win%colormap_init = .true.

      call msRedrawFigure()

   end subroutine msColormap_string
!_______________________________________________________________________
!
   subroutine msColormap_mfArray( colormap, int_max )

      type(mfArray) :: colormap
      integer, optional, intent(in) :: int_max
      !------ API end ------

      ! default : colors are ranged from 0.0 to 1.0
      !
      ! if optional 'int_max' is used, it indicates that entries
      ! are integers whom max value is provided
      ! (typically : 255 => 256 values for each R,G,B (and then
      !                     a subset of 16 millions of colors)

      ! for real entries (i.e. when int_max is not present),
      ! all NaN values are not processed.

      ! pointers for manipulating mfArray out of fml module
      real(kind=MF_DOUBLE), pointer :: colormap_ptr(:,:)

      integer :: i, n1, n2, offset, k
      real(kind=MF_DOUBLE) :: val_min, val_max
      real(kind=MF_DOUBLE) :: r, g, b
      integer :: mf_message_level_save
      logical :: NONE_found
      type(mf_win_info), pointer :: win

      character(len=*), parameter :: ROUTINE_NAME = "msColormap"

   !------ end of declarations -- execution starts hereafter  ------

      if( CURRENT_WIN_ID == 0 ) then
         call msFigure()
         if( CURRENT_WIN_ID == 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "cannot plot: no window created!" )
            return
         end if
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      call msInitArgs( colormap )

      ! checking that 'colormap' is allocated
      if( mfIsEqual(colormap,MF_EMPTY) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'colormap' not initialized!" )
         go to 99
      end if

      ! 'colormap' must be real
      if( .not. mfIsReal(colormap) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'colormap' must be real!" )
         go to 99
      end if

      ! 'colormap' cannot be sparse
      if( mfIsSparse(colormap) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'colormap' cannot be sparse!" )
         go to 99
      end if

      call msPointer( colormap, colormap_ptr, no_crc=.true. )

      n1 = size(colormap_ptr,1)
      n2 = size(colormap_ptr,2)

      ! checking that this array is a valid colormap
      if( n2 /= 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'colormap' array must have the shape = (:,3)" )
         go to 99
      end if

      if( present(int_max) ) then
         ! it must be a positive integer
         if( int_max <= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "optional 'int_max' arg. must be positive!" )
            go to 99
         end if
      end if

      call mf_save_and_disable_fpe( )

      val_min = minval( colormap_ptr )
      val_max = maxval( colormap_ptr )

      if( present(int_max) ) then
         if( val_min < 0.0d0 .or. dble(int_max) < val_max ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "some values of 'colormap' array are out of range [0-1]" )
         end if
      else
         if( val_min < 0.0d0 .or. 1.0d0 < val_max ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "some values of 'colormap' array are out of range [0-1]" )
         end if
      end if

      offset = win%colormap_ci_low - 1

      ! storing the user colormap
      if( allocated(win%user_colormap) ) then
         if( size(win%user_colormap,1) /= n1 ) then
            deallocate( win%user_colormap )
            allocate( win%user_colormap(n1,3) )
         end if
      else
         allocate( win%user_colormap(n1,3) )
      end if
      win%predefined_colormap = .false.

      win%colormap_none_removed = .false.
      ! only one NONE color must be present in the colormap
      ! (and we are going to remove it!)
      NONE_found = .false.

      k = 0
      do i = 1, n1
         r = colormap_ptr(i,1)
         g = colormap_ptr(i,2)
         b = colormap_ptr(i,3)
         if( present(int_max) ) then
            r = r/int_max
            g = g/int_max
            b = b/int_max
         end if
         ! detecting NaN values
         if( r /= r .or. g /= g .or. b /= b ) then
            if( NONE_found ) then
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "to define transparency, the colormap must contain at most one NaN triplet!" )
               go to 99
            end if
            NONE_found = .true.
            cycle
         end if
         k = k + 1
         call grscr(k+offset, r, g, b)
         win%user_colormap(k,:) = [ r, g, b ]
      end do

      win%colormap_ci_high = offset + k
      if( NONE_found ) then
         win%colormap_none_removed = .true.
         ! set last color (not used) to zero, in order to avoid warning
         ! from valgrind...
         win%user_colormap(n1,:) = [ 0.0d0, 0.0d0, 0.0d0 ]
      end if


      win%colormap_used = .true.

      call msFreePointer( colormap, colormap_ptr )

      call pgscir( win%colormap_ci_low, win%colormap_ci_high )

      win%colormap_init = .true.

 99   continue

      call mf_restore_fpe( )

      call msFreeArgs( colormap )
      call msAutoRelease( colormap )

   end subroutine msColormap_mfArray
!_______________________________________________________________________
!
   subroutine set_predefined_colormap( map_ind, inverted )

      integer, intent(in) :: map_ind
      logical, intent(in) :: inverted
      !------ API end ------

      ! Predefined colormap:
      !  0  'grey'     : gray scale (black -> white)
      !  1  'fusion'   : blue -> red -> yellow
      !  2  'bluered'  : blue -> red
      !  3  'rainbow'  : default (from MATLAB, named 'jet')
      !  4  'hot'      : hot (from MATLAB)
      !  5  'parula'   : like the new default colormap of MATLAB since 2014
      !  6  'flag'     : alternating between 4 colors (black, white, red, blue)
      !
      ! Colors are set in MFPLOT via PGSCR (set color representation);
      ! Colors' order is inverted if the boolean 'inverted' is true.

      integer :: i, ncol, offset, ncolm1, i1, i2, i3, i4, i5, i6, k
      integer :: MsgLevel_save
      type(mf_win_info), pointer :: win

      real(kind=MF_DOUBLE) :: r, g, b, x
      real(kind=MF_DOUBLE) :: x1, x2, y1, y2
      real(kind=MF_DOUBLE), allocatable :: xi(:)
      real(kind=MF_DOUBLE), allocatable :: ri(:), gi(:), bi(:)

      ! 'Flag' colormap
      real(kind=MF_DOUBLE), parameter :: flag_rgb(4,3) =                &
            transpose( reshape( [ 1.00d+0, 1.00d+0, 1.00d+0,            &
                                  0.00d+0, 0.00d+0, 1.00d+0,            &
                                  1.00d+0, 0.00d+0, 0.00d+0,            &
                                  0.00d+0, 0.00d+0, 0.00d+0 ], [ 3, 4 ] ) )

      ! 'Parula' colormap (approximation)
      real(kind=MF_DOUBLE), parameter :: parula_red_xy(21,2) =          &
            transpose( reshape( [ 0.0000d+0, 2.3903d-1,                 &
                                  6.3708d-2, 2.6974d-1,                 &
                                  1.2994d-1, 2.8193d-1,                 &
                                  1.9013d-1, 2.6974d-1,                 &
                                  2.0909d-1, 2.5895d-1,                 &
                                  2.6149d-1, 1.8694d-1,                 &
                                  3.0829d-1, 1.7566d-1,                 &
                                  3.3556d-1, 1.5146d-1,                 &
                                  3.5989d-1, 1.4098d-1,                 &
                                  4.1819d-1, 8.6953d-2,                 &
                                  4.6577d-1, 0.0000d+0,                 &
                                  5.2459d-1, 1.4098d-1,                 &
                                  5.7807d-1, 2.2218d-1,                 &
                                  6.4759d-1, 4.3054d-1,                 &
                                  7.4919d-1, 7.7985d-1,                 &
                                  8.1764d-1, 9.5450d-1,                 &
                                  8.4010d-1, 9.9433d-1,                 &
                                  8.6149d-1, 9.9740d-1,                 &
                                  8.8716d-1, 9.9127d-1,                 &
                                  9.3422d-1, 9.6063d-1,                 &
                                  1.0000d+0, 9.7748d-1 ], [ 2, 21 ] ) )

      real(kind=MF_DOUBLE), parameter :: parula_green_xy(11,2) =        &
            transpose( reshape( [ 0.0000d+0, 1.5012d-1,                 &
                                  7.2797d-2, 2.2539d-1,                 &
                                  2.2318d-1, 4.3909d-1,                 &
                                  4.0827d-1, 6.7564d-1,                 &
                                  5.3553d-1, 7.6570d-1,                 &
                                  6.4129d-1, 8.0333d-1,                 &
                                  7.3384d-1, 7.6704d-1,                 &
                                  8.1895d-1, 7.2806d-1,                 &
                                  8.6688d-1, 7.8182d-1,                 &
                                  9.3877d-1, 8.9204d-1,                 &
                                  1.0000d+0, 9.8343d-1 ], [ 2, 11 ] ) )

      real(kind=MF_DOUBLE), parameter :: parula_blue_xy(22,2) =         &
            transpose( reshape( [ 0.0000d+0, 6.6086d-1,                 &
                                  7.3624d-2, 8.5843d-1,                 &
                                  1.6617d-1, 9.7806d-1,                 &
                                  2.2483d-1, 9.9956d-1,                 &
                                  3.2730d-1, 9.2833d-1,                 &
                                  3.8266d-1, 8.8800d-1,                 &
                                  4.7438d-1, 7.6838d-1,                 &
                                  5.6693d-1, 6.0441d-1,                 &
                                  6.5286d-1, 3.8533d-1,                 &
                                  7.0740d-1, 2.4017d-1,                 &
                                  7.4128d-1, 1.7297d-1,                 &
                                  7.6441d-1, 1.5415d-1,                 &
                                  7.9333d-1, 1.7297d-1,                 &
                                  8.0903d-1, 2.0791d-1,                 &
                                  8.2639d-1, 2.3883d-1,                 &
                                  8.3135d-1, 2.4286d-1,                 &
                                  8.3796d-1, 2.4017d-1,                 &
                                  8.5861d-1, 2.2001d-1,                 &
                                  8.9249d-1, 1.8506d-1,                 &
                                  9.3216d-1, 1.5550d-1,                 &
                                  9.6521d-1, 1.2861d-1,                 &
                                  1.0000d+0, 8.0233d-2 ], [ 2, 22 ] ) )

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(CURRENT_WIN_ID)

      offset = win%colormap_ci_low - 1
      ncol = win%colormap_ci_high - offset
      ncolm1 = ncol - 1

      select case( map_ind )

         case( 0 ) !-------------------- grey --------------------

            do i = 1, ncol
               ! real parameter in [0,1]
               x = dble(i-1)/dble(ncolm1)
               r = x
               g = x
               b = x
               call grscr(offset+ind(i), r, g, b)
            enddo

         case( 1 ) !------------------- fusion -------------------

            do i = 1, ncol
               ! real parameter in [0,1]
               x = dble(i-1)/dble(ncolm1)
               if( 0.000000d+00 <= x .and. x < 6.300000d-02 ) then
                  ! interval #1
                  x1 = 0.000000d+00
                  y1 = 0.000000d+00
                  x2 = 6.300000d-02
                  y2 = 3.130435d-02
                  r = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  g = 0.0

                  x1 = 0.000000d+00
                  y1 = 0.000000d+00
                  x2 = 6.300000d-02
                  y2 = 4.226087d-01
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 6.300000d-02 <= x .and. x < 1.326000d-01 ) then
                  ! interval #2
                  x1 = 6.300000d-02
                  y1 = 3.130435d-02
                  x2 = 1.326000d-01
                  y2 = 2.226087d-01
                  r = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  g = 0.0

                  x1 = 6.300000d-02
                  y1 = 4.226087d-01
                  x2 = 1.326000d-01
                  y2 = 5.600000d-01
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 1.326000d-01 <= x .and. x < 3.133253d-01 ) then
                  ! interval #3
                  x1 = 1.326000d-01
                  y1 = 2.226087d-01
                  x2 = 3.133253d-01
                  y2 = 7.113043d-01
                  r = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  g = 0.0

                  x1 = 1.326000d-01
                  y1 = 5.600000d-01
                  x2 = 3.133253d-01
                  y2 = 5.826087d-01
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 3.133253d-01 <= x .and. x < 3.820000d-01 ) then
                  ! interval #4
                  x1 = 3.133253d-01
                  y1 = 7.113043d-01
                  x2 = 3.820000d-01
                  y2 = 7.982610d-01
                  r = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 3.133253d-01
                  y1 = 0.000000d+00
                  x2 = 3.820000d-01
                  y2 = 6.782609d-02
                  g = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 3.133253d-01
                  y1 = 5.826087d-01
                  x2 = 3.820000d-01
                  y2 = 5.130435d-01
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 3.820000d-01 <= x .and. x < 4.435000d-01 ) then
                  ! interval #5
                  x1 = 3.820000d-01
                  y1 = 7.982610d-01
                  x2 = 4.435000d-01
                  y2 = 8.573913d-01
                  r = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 3.820000d-01
                  y1 = 6.782609d-02
                  x2 = 7.575030d-01
                  y2 = 7.356522d-01
                  g = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 3.820000d-01
                  y1 = 5.130435d-01
                  x2 = 4.435000d-01
                  y2 = 3.478261d-01
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 4.435000d-01 <= x .and. x < 5.000000d-01 ) then
                  ! interval #6
                  x1 = 4.435000d-01
                  y1 = 8.573913d-01
                  x2 = 5.000000d-01
                  y2 = 9.008696d-01
                  r = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 3.820000d-01
                  y1 = 6.782609d-02
                  x2 = 7.575030d-01
                  y2 = 7.356522d-01
                  g = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 4.435000d-01
                  y1 = 3.478261d-01
                  x2 = 5.000000d-01
                  y2 = 7.826087d-02
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 5.000000d-01 <= x .and. x < 7.575030d-01 ) then
                  ! interval #7
                  x1 = 5.000000d-01
                  y1 = 9.008696d-01
                  x2 = 7.575030d-01
                  y2 = 1.000000d+00
                  r = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 3.820000d-01
                  y1 = 6.782609d-02
                  x2 = 7.575030d-01
                  y2 = 7.356522d-01
                  g = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 5.000000d-01
                  y1 = 7.826087d-02
                  x2 = 7.575030d-01
                  y2 = 0.000000d+00
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 7.575030d-01 <= x .and. x < 8.163265d-01 ) then
                  ! interval #8
                  r = 1.0

                  x1 = 7.575030d-01
                  y1 = 7.356522d-01
                  x2 = 8.775510d-01
                  y2 = 9.165217d-01
                  g = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 7.575030d-01
                  y1 = 0.000000d+00
                  x2 = 8.163265d-01
                  y2 = 1.391304d-02
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 8.163265d-01 <= x .and. x < 8.775510d-01 ) then
                  ! interval #9
                  r = 1.0

                  x1 = 7.575030d-01
                  y1 = 7.356522d-01
                  x2 = 8.775510d-01
                  y2 = 9.165217d-01
                  g = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 8.163265d-01
                  y1 = 1.391304d-02
                  x2 = 8.775510d-01
                  y2 = 2.295652d-01
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 8.775510d-01 <= x .and. x < 9.423769d-01 ) then
                  ! interval #10
                  r = 1.0

                  x1 = 8.775510d-01
                  y1 = 9.165217d-01
                  x2 = 9.423769d-01
                  y2 = 9.652174d-01
                  g = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 8.775510d-01
                  y1 = 2.295652d-01
                  x2 = 9.423769d-01
                  y2 = 6.765218d-01
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               else if( 9.423769d-01 <= x .and. x <= 1.000000d+00 ) then
                  ! interval #11
                  r = 1.0

                  x1 = 9.423769d-01
                  y1 = 9.652174d-01
                  x2 = 1.000000d+00
                  y2 = 1.000000d+00
                  g = y1 + (x-x1)*(y2-y1)/(x2-x1)

                  x1 = 9.423769d-01
                  y1 = 6.765218d-01
                  x2 = 1.000000d+00
                  y2 = 1.000000d+00
                  b = y1 + (x-x1)*(y2-y1)/(x2-x1)
               end if
               call grscr(offset+ind(i), r, g, b)
            enddo

         case( 2 ) !------------------- bluered ------------------

            do i = 1, ncol
               ! real parameter in [0,1]
               x = dble(i-1)/dble(ncolm1)
               r = x
               g = 0.0
               b = 1.0 - x
               call grscr(offset+ind(i), r, g, b)
            enddo

         case( 3 ) !------------------- rainbow ------------------

            i1 = 1
            i2 = max( 1, nint( 0.125d0 * ncolm1 ) )
            i3 = max( 1, nint( 0.375d0 * ncolm1 ) )
            i4 = max( 1, nint( 0.625d0 * ncolm1 ) )
            i5 = max( 1, nint( 0.875d0 * ncolm1 ) )
            i6 = ncol
            do i = i1, i2                   ! dark blue -> blue
               r = 0.0
               g = 0.0
               if( i1 == i2 ) then
                  b = 0.5625d0
               else
                  b = 0.5625d0*dble(i2-i)/dble(i2-i1)                   &
                    + 1.0000d0*dble(i-i1)/dble(i2-i1)
               end if
               call grscr(offset+ind(i), r, g, b)
            enddo
            do i = i2+1, i3                 ! blue -> cyan
               r = 0.0
               g = dble(i-i2)/dble(i3-i2)
               b = 1.0
               call grscr(offset+ind(i), r, g, b)
            enddo
            do i = i3+1, i4                 ! cyan -> yellow
               r = dble(i-i3)/dble(i4-i3)
               g = 1.0
               b = dble(i4-i)/dble(i4-i3)
               call grscr(offset+ind(i), r, g, b)
            enddo
            do i = i4+1, i5                 ! yellow -> red
               r = 1.0
               g = dble(i5-i)/dble(i5-i4)
               b = 0.0
               call grscr(offset+ind(i), r, g, b)
            enddo
            do i = i5+1, i6                 ! red -> dark red
               r = 1.0000d0*dble(i6-i)/dble(i6-i5)                      &
                 + 0.5625d0*dble(i-i5)/dble(i6-i5)
               g = 0.0
               b = 0.0
               call grscr(offset+ind(i), r, g, b)
            enddo

         case( 4 ) !--------------------- hot --------------------

            i1 = 1
            i2 = max( 1, nint( 0.375d0 * ncolm1 ) )
            i3 = max( 1, nint( 0.750d0 * ncolm1 ) )
            i4 = ncol
            do i = i1, i2                   ! dark red -> red
               if( i1 == i2 ) then
                  r = 0.30d0
               else
                  r = 0.30d0*dble(i2-i)/dble(i2-i1)                     &
                    + 1.00d0*dble(i-i1)/dble(i2-i1)
               end if
               g = 0.0
               b = 0.0
               call grscr(offset+ind(i), r, g, b)
            enddo
            do i = i2+1, i3                   ! red -> yellow
               r = 1.0
               g = dble(i-i2)/dble(i3-i2)
               b = 0.0
               call grscr(offset+ind(i), r, g, b)
            enddo
            do i = i3+1, i4                   ! yellow -> white
               r = 1.0
               g = 1.0
               b = 0.70d0*dble(i-i3)/dble(i4-i3)
               call grscr(offset+ind(i), r, g, b)
            enddo

         case( 5 ) !------------------- parula -------------------

            allocate( xi(ncol), ri(ncol), gi(ncol), bi(ncol) )
            do i = 1, ncol
               ! real parameter in [0,1]
               xi(i) = dble(i-1)/dble(ncolm1)
            enddo
            ! Red component
            MsgLevel_save = mfGetMsgLevel()
            call msSetMsgLevel(1)
            ri(:) = mfSpline( mf(parula_red_xy(:,1)),                   &
                              mf(parula_red_xy(:,2)), mf(xi) )
            ! Green component
            gi(:) = mfSpline( mf(parula_green_xy(:,1)),                 &
                              mf(parula_green_xy(:,2)), mf(xi) )
            ! Blue component
            bi(:) = mfSpline( mf(parula_blue_xy(:,1)),                  &
                              mf(parula_blue_xy(:,2)), mf(xi) )
            call msSetMsgLevel(MsgLevel_save)
            do i = 1, ncol
               if( ri(i) < 0.0d0 ) ri(i) = 0.0d0
               if( ri(i) > 1.0d0 ) ri(i) = 1.0d0
               if( gi(i) < 0.0d0 ) gi(i) = 0.0d0
               if( gi(i) > 1.0d0 ) gi(i) = 1.0d0
               if( bi(i) < 0.0d0 ) bi(i) = 0.0d0
               if( bi(i) > 1.0d0 ) bi(i) = 1.0d0
               call grscr(offset+ind(i), ri(i), gi(i), bi(i))
            enddo

         case( 6 ) !-------------------- flag --------------------

            k = 1
            do i = 1, ncol
               ! real parameter in [0,1]
               r = flag_rgb(k,1)
               g = flag_rgb(k,2)
               b = flag_rgb(k,3)
               call grscr(offset+ind(i), r, g, b)
               k = k + 1
               if( k == 5 ) k = 1
            enddo

      end select

   contains
   !____________________________________________________________________
   !
      function ind( i ) result( res )

         integer, intent(in) :: i
         integer             :: res
         !------ API end ------

         if( inverted ) then
            res = ncol - i + 1
         else
            res = i
         end if

      end function ind
   !____________________________________________________________________
   !
   end subroutine set_predefined_colormap
