! These routines decode an encoded string containing one or few attributes.
!
! If an attribute is not found, the returned integer code is always -127.
! The special value -127 is important, it is checked by many routines.
!
! Default values cannot be defined here, because they depend on the calling
! routine.

!_______________________________________________________________________
!
   subroutine decode_linespec( linespec,                                &
                               linecolor, linestyle, marker,            &
                               ier )

      character(len=*), intent(in)            :: linespec
      integer,          intent(out)           :: linecolor
      integer,          intent(out), optional :: linestyle, marker
      integer,          intent(out), optional :: ier
      !------ API end ------

      ! From a string containing encoded information, find:
      !  - a color
      !  - a style
      !  - a marker
      !
      ! ier=0 when all is correct.
      ! ier=1 when we find in linespec some unexpected attributes
      !       (e.g. found linestyle and marker codes, while the output
      !        argument is not present).

      integer :: i, n_opt, color_nb, style_nb, marker_nb
      integer :: color_number, linestyle0, marker0
      character :: c
      character(len=3) :: color_nb_str
      type(mf_win_info), pointer :: win
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( present(ier) ) then
         ier = 0
      end if

      linecolor  = -127 ! means: not found
      linestyle0 = -127 ! means: not found
      marker0    = -127 ! means: not found

      n_opt = len_trim(linespec)
      color_nb = 0
      style_nb = 0
      marker_nb = 0
      i = 1
      do
         if( i == n_opt + 1 ) exit
         c = linespec(i:i)
         i = i + 1
         if( c == " " ) cycle
         select case( to_lower(c) )

            ! --- line color ---
            case( "k" ) ! black
               if( BLACK_ON_WHITE == 1 ) then
                  linecolor = 1
               else
                  linecolor = MFPLOT_QUASI_BLACK
               end if
               color_nb = color_nb + 1
            case( "w" ) ! white
               if( BLACK_ON_WHITE == 1 ) then
                  linecolor = MFPLOT_QUASI_WHITE
               else
                  linecolor = 1
               end if
               color_nb = color_nb + 1
            case( "r" ) ! red
               linecolor = 2
               color_nb = color_nb + 1
            case( "g" ) ! green
               linecolor = 3
               color_nb = color_nb + 1
            case( "b" ) ! blue
               linecolor = 4
               color_nb = color_nb + 1
            case( "c" ) ! cyan
               linecolor = 5
               color_nb = color_nb + 1
            case( "m" ) ! magenta
               linecolor = 6
               color_nb = color_nb + 1
            case( "y" ) ! yellow
               linecolor = 7
               color_nb = color_nb + 1

            ! --- line style ---
            case( "-" ) ! multiple case
               if( i == n_opt + 1 ) then
                  linestyle0 = 1
               else
                  if( linespec(i:i) == "." ) then ! dashdot "-."
                     linestyle0 = 3
                     i = i + 1
                  else if( linespec(i:i) == "-" ) then ! dashed "--"
                     linestyle0 = 2
                     i = i + 1
                  else ! continuous "-"
                     linestyle0 = 1
                  end if
               end if
               style_nb = style_nb + 1
            case( ":" ) ! dotted
               linestyle0 = 4
               style_nb = style_nb + 1

            ! --- marker style (mainly) ---
            case( "." ) ! point
               if( i == n_opt + 1 ) then
                  marker0 = 1
                  marker_nb = marker_nb + 1
               else
                  if( linespec(i:i) == "." ) then ! dotted (".." alias for ":")
                     linestyle0 = 4
                     i = i + 1
                     style_nb = style_nb + 1
                  else if( linespec(i:i) == "-" ) then ! dashdot (".-" alias for "-.")
                     linestyle0 = 3
                     i = i + 1
                     style_nb = style_nb + 1
                  else
                     marker0 = 1
                     marker_nb = marker_nb + 1
                  end if
               end if
            case( "+" ) ! plus
               marker0 = 2
               marker_nb = marker_nb + 1
            case( "*" ) ! asterisk
               marker0 = 3
               marker_nb = marker_nb + 1
            case( "o" ) ! circle
               marker0 = 6
               marker_nb = marker_nb + 1
            case( "x" ) ! x-mark
               marker0 = 4
               marker_nb = marker_nb + 1
            case( "s" ) ! square
               marker0 = 7
               marker_nb = marker_nb + 1
            case( "^" ) ! triangle (up)
               marker0 = 10
               marker_nb = marker_nb + 1
            case( "d" ) ! diamond
               marker0 = 8
               marker_nb = marker_nb + 1
            case( "\" ) ! multiple subcase
               if( i+11 <= n_opt ) then
                  if( to_lower(linespec(i:i+11)) == "circlefilled" ) then
                     marker0 = 14
                     marker_nb = marker_nb + 1
                     i = i + 12
                     cycle
                  end if
               end if
               if( i+5 <= n_opt ) then
                  if( to_lower(linespec(i:i+5)) == "circle" ) then
                     marker0 = 6
                     marker_nb = marker_nb + 1
                     i = i + 6
                     cycle
                  end if
               end if
               if( i+11 <= n_opt ) then
                  if( to_lower(linespec(i:i+11)) == "squarefilled" ) then
                     marker0 = 15
                     marker_nb = marker_nb + 1
                     i = i + 12
                     cycle
                  end if
               end if
               if( i+5 <= n_opt ) then
                  if( to_lower(linespec(i:i+5)) == "square" ) then
                     marker0 = 7
                     marker_nb = marker_nb + 1
                     i = i + 6
                     cycle
                  end if
               end if
               if( i+12 <= n_opt ) then
                  if( to_lower(linespec(i:i+12)) == "diamondfilled" ) then
                     marker0 = 16
                     marker_nb = marker_nb + 1
                     i = i + 13
                     cycle
                  end if
               end if
               if( i+6 <= n_opt ) then
                  if( to_lower(linespec(i:i+6)) == "diamond" ) then
                     marker0 = 8
                     marker_nb = marker_nb + 1
                     i = i + 7
                     cycle
                  end if
               end if
               if( i+9 <= n_opt ) then
                  if( to_lower(linespec(i:i+9)) == "starfilled" ) then
                     marker0 = 17
                     marker_nb = marker_nb + 1
                     i = i + 10
                     cycle
                  end if
               end if
               if( i+3 <= n_opt ) then
                  if( to_lower(linespec(i:i+3)) == "star" ) then
                     marker0 = 9
                     marker_nb = marker_nb + 1
                     i = i + 4
                     cycle
                  end if
               end if
               if( i+15 <= n_opt ) then
                  if( to_lower(linespec(i:i+15)) == "triangleupfilled" ) then
                     marker0 = 18
                     marker_nb = marker_nb + 1
                     i = i + 16
                     cycle
                  end if
               end if
               if( i+9 <= n_opt ) then
                  if( to_lower(linespec(i:i+9)) == "triangleup" ) then
                     marker0 = 10
                     marker_nb = marker_nb + 1
                     i = i + 10
                     cycle
                  end if
               end if
               if( i+17 <= n_opt ) then
                  if( to_lower(linespec(i:i+17)) == "triangledownfilled" ) then
                     marker0 = 19
                     marker_nb = marker_nb + 1
                     i = i + 18
                     cycle
                  end if
               end if
               if( i+11 <= n_opt ) then
                  if( to_lower(linespec(i:i+11)) == "triangledown" ) then
                     marker0 = 11
                     marker_nb = marker_nb + 1
                     i = i + 12
                     cycle
                  end if
               end if
               if( i+17 <= n_opt ) then
                  if( to_lower(linespec(i:i+17)) == "triangleleftfilled" ) then
                     marker0 = 20
                     marker_nb = marker_nb + 1
                     i = i + 18
                     cycle
                  end if
               end if
               if( i+11 <= n_opt ) then
                  if( to_lower(linespec(i:i+11)) == "triangleleft" ) then
                     marker0 = 12
                     marker_nb = marker_nb + 1
                     i = i + 12
                     cycle
                  end if
               end if
               if( i+18 <= n_opt ) then
                  if( to_lower(linespec(i:i+18)) == "trianglerightfilled" ) then
                     marker0 = 21
                     marker_nb = marker_nb + 1
                     i = i + 19
                     cycle
                  end if
               end if
               if( i+12 <= n_opt ) then
                  if( to_lower(linespec(i:i+12)) == "triangleright" ) then
                     marker0 = 13
                     marker_nb = marker_nb + 1
                     i = i + 13
                     cycle
                  end if
               end if
               if( i+2 <= n_opt ) then
                  if( to_lower(linespec(i:i+2)) == "dot" ) then
                     marker0 = 1
                     marker_nb = marker_nb + 1
                     i = i + 3
                     cycle
                  end if
               end if
               if( i+3 <= n_opt ) then
                  if( to_lower(linespec(i:i+3)) == "plus" ) then
                     marker0 = 2
                     marker_nb = marker_nb + 1
                     i = i + 4
                     cycle
                  end if
               end if
               if( i+7 <= n_opt ) then
                  if( to_lower(linespec(i:i+7)) == "asterisk" ) then
                     marker0 = 3
                     marker_nb = marker_nb + 1
                     i = i + 8
                     cycle
                  end if
               end if
               if( i <= n_opt ) then
                  if( to_lower(linespec(i:i)) == "x" ) then
                     marker0 = 4
                     marker_nb = marker_nb + 1
                     i = i + 1
                     cycle
                  end if
               end if
               if( i+3 <= n_opt ) then
                  if( to_lower(linespec(i:i+3)) == "hash" ) then
                     marker0 = 5
                     marker_nb = marker_nb + 1
                     i = i + 4
                     cycle
                  end if
               end if
               if( i+15 <= n_opt ) then
                  if( to_lower(linespec(i:i+15)) == "heavyfourballoon" ) then
                     marker0 = 22
                     marker_nb = marker_nb + 1
                     i = i + 16
                     cycle
                  end if
               end if
               if( i+12 <= n_opt ) then
                  if( to_lower(linespec(i:i+12)) == "heavyteardrop" ) then
                     marker0 = 23
                     marker_nb = marker_nb + 1
                     i = i + 13
                     cycle
                  end if
               end if
               if( i+12 <= n_opt ) then
                  if( to_lower(linespec(i:i+12)) == "whiteflorette" ) then
                     marker0 = 24
                     marker_nb = marker_nb + 1
                     i = i + 13
                     cycle
                  end if
               end if
               if( i+8 <= n_opt ) then
                  if( to_lower(linespec(i:i+8)) == "snowflake" ) then
                     marker0 = 25
                     marker_nb = marker_nb + 1
                     i = i + 9
                     cycle
                  end if
               end if
               if( i+22 <= n_opt ) then
                  if( to_lower(linespec(i:i+22)) == "blackdiamondminuswhitex" ) then
                     marker0 = 26
                     marker_nb = marker_nb + 1
                     i = i + 23
                     cycle
                  end if
               end if

               ! match a color specification of the form \Cnn
               if( i+2 <= n_opt ) then
                  if( linespec(i:i) == "C" ) then
                     i = i + 1
                     read(linespec(i:i+1),*,iostat=status) color_number
                     if( status /= 0 ) then
                        call PrintMessage( "linespec", "E",             &
                                           "Cannot decode escaped sequence beginning by '\'!" )
                        return
                     end if
                     win => mf_win_db(CURRENT_WIN_ID)
                     select case( win%color_scheme )
                        case( 1 )
                           color_number = mod( color_number-1,          &
                                               size(COL_CYCLE_TAB_1) ) + 1
                           linecolor = COL_CYCLE_TAB_1(color_number)
                        case( 2 )
                           color_number = mod( color_number-1,          &
                                               size(COL_CYCLE_TAB_2) ) + 1
                           linecolor = COL_CYCLE_TAB_2(color_number)
                        case( 3 )
                           color_number = mod( color_number-1,          &
                                               size(COL_CYCLE_TAB_3) ) + 1
                           linecolor = COL_CYCLE_TAB_3(color_number)
                        case( 4 )
                           color_number = mod( color_number-1,          &
                                               size(COL_CYCLE_TAB_4) ) + 1
                           linecolor = COL_CYCLE_TAB_4(color_number)
                        case default
                           write(STDERR,*) "(FGL decode_linespec:) Internal error."
                           write(STDERR,*) "     Bad value for color_scheme."
                           pause "only for debugging purpose"
                           stop
                     end select
                     color_nb = color_nb + 1
                     i = i + 2
                     cycle
                  end if
               end if

               if( i+3 <= n_opt ) then
                  if( linespec(i:i+1) == "M-" ) then
                     i = i + 1
                     read(linespec(i:i+2),*) marker0
                     marker_nb = marker_nb + 1
                     i = i + 3
                     cycle
                  end if
               end if

               if( i+2 <= n_opt ) then
                  if( linespec(i:i) == "M" ) then
                     i = i + 1
                     read(linespec(i:i+1),*) marker0
                     marker_nb = marker_nb + 1
                     i = i + 2
                     cycle
                  end if
               end if

               call PrintMessage( "linespec", "E",                      &
                                  "Cannot decode escaped sequence beginning by '\'!" )

            case( " " )
               ! nothing

            ! error
            case default
               call PrintMessage( "linespec", "W",                      &
                                  "Bad character found: '" // c // "' (ignored)" )

         end select
      end do

      if( color_nb > 1 ) then
         write(color_nb_str,"(I0)") color_nb
         call PrintMessage( "linespec", "E",                            &
                            "String cannot contain more than one color!", &
                            "(found " // trim(color_nb_str) // " colors)" )
         return
      end if
      if( style_nb > 1 ) then
         call PrintMessage( "linespec", "E",                            &
                            "String cannot contain more than one style!" )
         return
      end if
      if( marker_nb > 1 ) then
         call PrintMessage( "linespec", "E",                            &
                            "String cannot contain more than one marker!" )
         return
      end if

      if( present(linestyle) ) then
         linestyle = linestyle0
      else
         if( linestyle0 /= -127 ) then
            if( present(ier) ) then
               ier = 1
            end if
         end if
      end if

      if( present(marker) ) then
         marker = marker0
      else
         if( marker0 /= -127 ) then
            if( present(ier) ) then
               ier = 1
            end if
         end if
      end if

   end subroutine decode_linespec
!_______________________________________________________________________
!
   subroutine decode_col_name( col_name, icol )

      character(len=*), intent(in)  :: col_name
      integer,          intent(out) :: icol
      !------ API end ------

      ! Wish to use a special color, given on input by its name from the
      ! rgb.txt database.
      !
      ! On output, the ICOL color index is returned.
      !
      ! How it works: try to get its R, G, B components via pgscrn/pgqcr,
      ! then call 'decode_col_rgb'.
      !
      ! Note: CI_HIGH_MAX is used as auxiliary color index.

      integer :: ier
      real(kind=MF_DOUBLE) :: r, g, b
      real(kind=MF_DOUBLE) :: col_rgb(3)

   !------ end of declarations -- execution starts hereafter  ------

      call pgscrn( CI_HIGH_MAX, col_name, ier )
      if( ier == 1 ) then
         call PrintMessage( "col_name", "W",                            &
                            'Color name "' // trim(col_name) // '" not found in RGB database!' )
         icol = -127 ! means: not found
         return
      end if

      call grqcr( CI_HIGH_MAX, r, g, b )
      col_rgb(:) = [ r, g, b ]
      call decode_col_rgb( col_rgb, icol )

   end subroutine decode_col_name
!_______________________________________________________________________
!
   subroutine decode_col_rgb( col_rgb, icol )

      real(kind=MF_DOUBLE), intent(in)  :: col_rgb(3)
      integer,              intent(out) :: icol
      !------ API end ------

      ! Wish to use a special color, given on input by its R, G, B components
      !
      ! On output, the ICOL color index is returned.
      !
      ! How it works: a search is done both in the lower and the upper part
      ! of the internal colors of MFPLOT:
      !
      ! - if it is a new color, it is registered
      !
      ! - if it is already registered, get its color index

      real(kind=MF_DOUBLE) :: r, g, b
      real(kind=MF_DOUBLE) :: ri, gi, bi
      integer :: i
      type(mf_win_info), pointer :: win

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(CURRENT_WIN_ID)

      ! must use, for testing below, an X11 color (as stored by the X11 server)
      r = col_rgb(1)
      g = col_rgb(2)
      b = col_rgb(3)
      if( X11_DEVICE ) then
         call correct_X11_color( r, g, b ) ! modify (r,g,b)
      else if( NULL_DEVICE ) then
         ! the NULL device has 256 levels for each color component
         ! (see opcodes 21 and 29 in the NULL driver)
         r = nint(r*255.0d0)/255.0d0
         g = nint(g*255.0d0)/255.0d0
         b = nint(b*255.0d0)/255.0d0
      end if

      ! is the requested (R,G,B) color in the 42 predefined colors?
      do i = 0, 41
         call grqcr(i,ri,gi,bi)
         if( r == ri .and. g == gi .and. b == bi ) then
            icol = i
            if( icol == 0 ) then
               ! 0 is always the background color, so it must be slightly
               ! altered (cf. transparency in X11 driver)
               if( BLACK_ON_WHITE == 1 ) then
                  icol = MFPLOT_QUASI_WHITE
               else ! WHITE on BLACK
                  icol = MFPLOT_QUASI_BLACK
               end if
            end if
            return
         end if
      end do

      if( NB_USER_COL == 0 ) then
         ! no special color already registered, so no search to do
         NB_USER_COL = 1
         icol = CI_HIGH_MAX - 1
      else
         ! scan the color table of MFPLOT to check whether the (R,G,B) color
         ! is present...
         do i = CI_HIGH_MAX-1, CI_HIGH_MAX-NB_USER_COL, -1
            call grqcr(i,ri,gi,bi)
            if( r == ri .and. g == gi .and. b == bi ) then
               icol = i
               return
            end if
         end do
         ! (R,G,B) not found
         NB_USER_COL = NB_USER_COL + 1
         icol = CI_HIGH_MAX - NB_USER_COL
         ! check that the 'user' colors don't overlap with any colormap
         if( win%colormap_used ) then
            if( icol <=  win%colormap_ci_high ) then
               NB_USER_COL = NB_USER_COL - 1
               call PrintMessage( "col_rgb", "W",                       &
                                  "nb of user-color too big!" )
               icol = -127 ! means: not found
               return
            end if
         end if

      end if

      ! registering the new color
      call grscr( icol, r, g, b )

   end subroutine decode_col_rgb
!_______________________________________________________________________
!
   subroutine decode_colorspec( colspec, icol )

      character(len=*), intent(in)  :: colspec
      integer,          intent(out) :: icol
      !------ API end ------

      ! From a string containing encoded information, find:
      !  - a color

      integer :: i, n_opt, color_nb
      integer :: color_number
      character :: c
      character(len=3) :: color_nb_str
      type(mf_win_info), pointer :: win

   !------ end of declarations -- execution starts hereafter  ------

      icol = -127 ! means: not found

      n_opt = len_trim(colspec)
      color_nb = 0
      i = 1
      do
         if( i == n_opt + 1 ) exit
         c = colspec(i:i)
         i = i + 1
         if( c == " " ) cycle
         select case( to_lower(c) )

            ! --- line color ---
            case( "k" ) ! black
               if( BLACK_ON_WHITE == 1 ) then
                  icol = 1
               else
                  icol = MFPLOT_QUASI_BLACK
               end if
               color_nb = color_nb + 1
            case( "w" ) ! white
               if( BLACK_ON_WHITE == 1 ) then
                  icol = MFPLOT_QUASI_WHITE
               else
                  icol = 1
               end if
               color_nb = color_nb + 1
            case( "r" ) ! red
               icol = 2
               color_nb = color_nb + 1
            case( "g" ) ! green
               icol = 3
               color_nb = color_nb + 1
            case( "b" ) ! blue
               icol = 4
               color_nb = color_nb + 1
            case( "c" ) ! cyan
               icol = 5
               color_nb = color_nb + 1
            case( "m" ) ! magenta
               icol = 6
               color_nb = color_nb + 1
            case( "y" ) ! yellow
               icol = 7
               color_nb = color_nb + 1

            case( "\" ) ! multiple subcase

               ! match a color specification of the form \Cnn
               if( i+2 <= n_opt ) then
                  if( colspec(i:i) == "C" ) then
                     i = i + 1
                     read(colspec(i:i+1),*) color_number
                     win => mf_win_db(CURRENT_WIN_ID)
                     select case( win%color_scheme )
                        case( 1 )
                           color_number = mod( color_number-1,          &
                                               size(COL_CYCLE_TAB_1) ) + 1
                           icol = COL_CYCLE_TAB_1(color_number)
                        case( 2 )
                           color_number = mod( color_number-1,          &
                                               size(COL_CYCLE_TAB_2) ) + 1
                           icol = COL_CYCLE_TAB_2(color_number)
                        case( 3 )
                           color_number = mod( color_number-1,          &
                                               size(COL_CYCLE_TAB_3) ) + 1
                           icol = COL_CYCLE_TAB_3(color_number)
                        case( 4 )
                           color_number = mod( color_number-1,          &
                                               size(COL_CYCLE_TAB_4) ) + 1
                           icol = COL_CYCLE_TAB_4(color_number)
                        case default
                           write(STDERR,*) "(FGL decode_colorspec:) Internal error."
                           write(STDERR,*) "     Bad value for color_scheme."
                           pause "only for debugging purpose"
                           stop
                     end select
                     color_nb = color_nb + 1
                     i = i + 2
                     cycle
                  end if
               end if

            ! error
            case default
               call PrintMessage( "colorspec", "W",                     &
                                  "Bad character found: '" // c // "' (ignored)" )

         end select

      end do

      if( color_nb > 1 ) then
         write(color_nb_str,"(I0)") color_nb
         call PrintMessage( "colorspec", "W",                           &
                            "String cannot contain more than one color!", &
                            "(found " // trim(color_nb_str) // " colors)", &
                            "-> last color has been selected." )
         return
      end if

      if( icol == -127 ) then
         call PrintMessage( "colorspec", "W",                           &
                            "String doesn't contain any color." )
      end if

   end subroutine decode_colorspec
