module mod_window

   ! Part of MUESLI Graphic Library
   ! Copyright É. Canot 2004-2025 -- IPR/CNRS

   use mod_mfarray, only: PrintMessage ! private when using FML

   use mod_pgplot

   use fgl_aux_2

   use fgl_aux

#ifndef _OPTIM
   use mod_fgl_mem_debug
#endif

#ifdef _SPREAD_MOD
!!   use mod_win_db

!!   use fml ! MUESLI Numerical Library
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

   private :: PrintMessage

!-----------------------------------------------------------------------
!                               public
!-----------------------------------------------------------------------

   public :: msClose, &
             mfFigure, msFigure, &
             msInitFgl, &
             msExitFgl, &
             mfGetColorOverflowPolicy, &
             msSetBackgroundColor, &
             msSetColorOverflowPolicy, &
             mfGetX11ColorDepth, &
             msSetCharEncoding, &
             mfGetCharEncoding, &
             msSetColorScheme, &
             mfGetColorScheme, &
             msSetColorInd, &
             mfGetColorInd, &
             msSetX11Device, &
             mfGetX11Device

   public :: MFPLOT_DEBUG

!-----------------------------------------------------------------------
!                             parameters
!-----------------------------------------------------------------------

   ! Position for "Cascade Window Layout" (28 was adapted to Ubuntu-18.04?)
   ! (used in 'msFigure' and 'read_mf_win_pos_db')
   integer, parameter :: MF_WIN_X_OFFSET = 32, MF_WIN_Y_OFFSET = 32

!-----------------------------------------------------------------------
!                             interfaces
!-----------------------------------------------------------------------

contains

#include "mf_win_aux.F90"

#include "mf_win_redraw.f90"

#include "Arrow_draw.f90"

#include "Bar_draw.f90"

#include "Contour_draw.f90"

#include "ContourF_draw.f90"

#include "ErrorBar_draw.f90"

#include "Image_draw.f90"

#include "Patch_draw.f90"

#include "Pcolor_draw.f90"

#include "Plot_draw.f90"

#include "PlotCubicBezier_draw.f90"

#include "PlotHist_draw.f90"

#include "PlotQuadrBezier_draw.f90"

#include "PlotVoronoi_draw.f90"

#include "Quiver_draw.f90"

#include "Spy_draw.f90"

#include "Streamline_draw.f90"

#include "Text_draw.f90"

#include "TriContour_draw.f90"

#include "TriContourF_draw.f90"

#include "TriFill_draw.f90"

#include "TriMesh_draw.f90"

#include "TriPcolor_draw.f90"

#include "TriQuiver_draw.f90"

#include "PlotPLdomain_draw.f90"

#include "Legend_aux.f90"

!-----------------------------------------------------------------------

#include "Close.f90"

#include "Figure.f90"

#include "InitFgl.f90"

#include "ExitFgl.f90"

#include "SetBackgroundColor.f90"

#include "GetX11Device.f90"

#include "SetX11Device.F90"

!_______________________________________________________________________
!
   subroutine msSetColorOverflowPolicy( low, high )

      character(len=*), optional :: low, high
      !------ API end ------

      character(len=*), parameter :: ROUTINE_NAME = "msSetColorOverflowPolicy"

   !------ end of declarations -- execution starts hereafter  ------

      if( present(low) ) then
         if( to_lower(low) == "signaled" ) then
            ! default
            COLOR_OVERFLOW_LOW_POLICY  = 0
         else if( to_lower(low) == "truncated" ) then
            COLOR_OVERFLOW_LOW_POLICY  = 1
         else
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "unknown keyword : '" // trim(low) // "'",&
                               "(should be 'signaled' or 'truncated')" )
         end if
      end if

      if( present(high) ) then
         if( to_lower(high) == "signaled" ) then
            ! default
            COLOR_OVERFLOW_HIGH_POLICY  = 0
         else if( to_lower(high) == "truncated" ) then
            COLOR_OVERFLOW_HIGH_POLICY  = 1
         else
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "unknown keyword : '" // trim(high) // "'",&
                               "(should be 'signaled' or 'truncated')" )
         end if
      end if

   end subroutine msSetColorOverflowPolicy
!_______________________________________________________________________
!
   function mfGetColorOverflowPolicy() result( policy )

      character(len=10) :: policy(2)
      !------ API end ------

      character(len=*), parameter :: ROUTINE_NAME = "mfGetColorOverflowPolicy"

   !------ end of declarations -- execution starts hereafter  ------

      if( COLOR_OVERFLOW_LOW_POLICY == 0 ) then
         policy(1) = "signaled"
      else if( COLOR_OVERFLOW_LOW_POLICY == 1 ) then
         policy(1) = "truncated"
      else
         write(STDERR,*) "(MUESLI " // trim(ROUTINE_NAME) // ":) internal error: #1"
         write(STDERR,*) "        Please report this bug to: Edouard.Canot@univ-rennes.fr"
         call msMuesliTrace( pause="yes" )
         stop
      end if

      if( COLOR_OVERFLOW_HIGH_POLICY == 0 ) then
         policy(2) = "signaled"
      else if( COLOR_OVERFLOW_HIGH_POLICY == 1 ) then
         policy(2) = "truncated"
      else
         write(STDERR,*) "(MUESLI " // trim(ROUTINE_NAME) // ":) internal error: #2"
         write(STDERR,*) "        Please report this bug to: Edouard.Canot@univ-rennes.fr"
         call msMuesliTrace( pause="yes" )
         stop
      end if

   end function mfGetColorOverflowPolicy
!_______________________________________________________________________
!
   function mfGetX11ColorDepth() result( depth )

      integer :: depth
      !------ API end ------

      character(len=*), parameter :: ROUTINE_NAME = "mfGetX11ColorDepth"

   !------ end of declarations -- execution starts hereafter  ------

      if( CURRENT_WIN_ID == 0 ) then
         depth = -1
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "no X11 window opened!" )
         return
      end if

      depth = Get_Color_Depth()

   end function mfGetX11ColorDepth
!_______________________________________________________________________
!
   subroutine msSetCharEncoding( encoding )

      character(len=*) :: encoding
      !------ API end ------

      character(len=*), parameter :: ROUTINE_NAME = "msSetCharEncoding"

   !------ end of declarations -- execution starts hereafter  ------

      if( to_lower(encoding) == "latin" .or.                            &
          to_lower(encoding) == "iso-8859-1"  ) then
         ! default
         UTF8_ENCODING = .false.
      else if( to_lower(encoding) == "utf-8" ) then
         UTF8_ENCODING = .true.
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "unknown keyword : '" // trim(encoding) // "'",&
                            "(should be 'Latin' or 'UTF-8')" )
      end if

   end subroutine msSetCharEncoding
!_______________________________________________________________________
!
   function mfGetCharEncoding() result( encoding )

      character(len=5) :: encoding
      !------ API end ------

   !------ end of declarations -- execution starts hereafter  ------

      if( UTF8_ENCODING ) then
         encoding = "UTF-8"
      else
         encoding = "Latin"
      end if

   end function mfGetCharEncoding
!_______________________________________________________________________
!
   subroutine msSetColorScheme( color_scheme )

      integer, intent(in) :: color_scheme
      !------ API end ------

      type(mf_win_info), pointer :: win

      character(len=*), parameter :: ROUTINE_NAME = "msSetColorScheme"

   !------ end of declarations -- execution starts hereafter  ------

      if( CURRENT_WIN_ID == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "no X11 window opened!" )
         return
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      if( 1 <= color_scheme .and. color_scheme <= 4 ) then
         win%color_scheme = color_scheme
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "argument must be in [1-4]!",               &
                            "(color scheme is left unchanged)")
      end if

   end subroutine msSetColorScheme
!_______________________________________________________________________
!
   function mfGetColorScheme( ) result( color_scheme )

      integer :: color_scheme
      !------ API end ------

      type(mf_win_info), pointer :: win

      character(len=*), parameter :: ROUTINE_NAME = "msSetColorScheme"

   !------ end of declarations -- execution starts hereafter  ------

      if( CURRENT_WIN_ID == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "no X11 window opened!" )
         color_scheme = -1
         return
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      color_scheme = win%color_scheme

   end function mfGetColorScheme
!_______________________________________________________________________
!
   subroutine msSetColorInd( ind )

      integer, intent(in) :: ind
      !------ API end ------

      type(mf_win_info), pointer :: win

      integer :: ind_max

      character(len=*), parameter :: ROUTINE_NAME = "msSetColorInd"

   !------ end of declarations -- execution starts hereafter  ------

      if( CURRENT_WIN_ID == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "no X11 window opened!" )
         return
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      ! Get number of colors in the current color scheme
      select case( win%color_scheme )
         case( 1 ) ! Old PGPLOT colors
            ind_max = 6
         case( 2 ) ! Old MATLAB colors (before the 2014a release)
            ind_max = 7
         case( 3 ) ! New MATLAB colors
            ind_max = 7
         case( 4 ) ! Breeze colortable
            ind_max = 12
      end select

      if( 1 <= ind .and. ind <= ind_max ) then
         win%ind_next_color = ind
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "for the current color scheme, 'ind' is out-of-range!")
      end if

   end subroutine msSetColorInd
!_______________________________________________________________________
!
   function mfGetColorInd( ) result( ind )

      integer :: ind
      !------ API end ------

      type(mf_win_info), pointer :: win

      character(len=*), parameter :: ROUTINE_NAME = "mfGetColorInd"

   !------ end of declarations -- execution starts hereafter  ------

      if( CURRENT_WIN_ID == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "no X11 window opened!" )
         ind = -1
         return
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      ind = win%ind_next_color

   end function mfGetColorInd
!_______________________________________________________________________
!
   subroutine msSetDefaultCapStyle( cap_style )

      character(len=*), intent(in) :: cap_style
      !------ API end ------

      character(len=*), parameter :: ROUTINE_NAME = "msSetDefaultCapStyle"

   !------ end of declarations -- execution starts hereafter  ------

      select case( to_lower(cap_style) )
         case( "capbutt" )
            MF_DEFAULT_CAP_STYLE = 0
         case( "capround" )
            MF_DEFAULT_CAP_STYLE = 1
         case( "capprojecting" )
            MF_DEFAULT_CAP_STYLE = 2
         case default
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "'cap_style' argument must be one of the following:", &
                               '"CapButt", "CapRound" or "CapProjecting"', &
                               "-> default value has not been changed!")
      end select

   end subroutine msSetDefaultCapStyle
!_______________________________________________________________________
!
   function mfGetDefaultCapStyle( ) result( out )

      integer :: out
      !------ API end ------

      character(len=*), parameter :: ROUTINE_NAME = "mfGetDefaultCapStyle"

   !------ end of declarations -- execution starts hereafter  ------

      out = MF_DEFAULT_CAP_STYLE

   end function mfGetDefaultCapStyle
!_______________________________________________________________________
!
   subroutine msSetDefaultJoinStyle( join_style )

      character(len=*), intent(in) :: join_style
      !------ API end ------

      character(len=*), parameter :: ROUTINE_NAME = "msSetDefaultJoinStyle"

   !------ end of declarations -- execution starts hereafter  ------

      select case( to_lower(join_style) )
         case( "joinmiter" )
            MF_DEFAULT_CAP_STYLE = 0
         case( "joinround" )
            MF_DEFAULT_CAP_STYLE = 1
         case( "joinbevel" )
            MF_DEFAULT_CAP_STYLE = 2
         case default
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "'join_style' argument must be one of the following:", &
                               '"JoinMiter", "JoinRound" or "JoinBevel"', &
                               "-> default value has not been changed!")
      end select

   end subroutine msSetDefaultJoinStyle
!_______________________________________________________________________
!
   function mfGetDefaultJoinStyle( ) result( out )

      integer :: out
      !------ API end ------

      character(len=*), parameter :: ROUTINE_NAME = "mfGetDefaultJoinStyle"

   !------ end of declarations -- execution starts hereafter  ------

      out = MF_DEFAULT_JOIN_STYLE

   end function mfGetDefaultJoinStyle
!_______________________________________________________________________
!
end module mod_window
