module mod_win_db

   ! Part of MUESLI Graphic Library
   ! Copyright É. Canot 2004-2025 -- IPR/CNRS

   use mod_mfaux

   use mod_ieee

   implicit none

#ifndef _DEVLP
   private
#endif

   !    ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
   !    ┃ In Fortran, all module variables which are initialized get ┃
   !    ┃ automatically the 'save' attribute.                        ┃
   !    ┃ Don't forget to add the 'save' attribute for the others.   ┃
   !    ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛

!-----------------------------------------------------------------------
!                              parameters
!-----------------------------------------------------------------------

   ! max. nb of windows (and also max. ID for a MUESLI window)
   !                    (must be equal to GRIMAX in 'mod_grplot.F90')
   integer, parameter :: MF_WIN_NB_MAX = 64

   character(len=*), parameter :: WIN_POS_DB_NAME = ".mf_win_pos_db"

   ! MF_WIN_WIDTH_DEF and MF_WIN_HEIGHT_DEF should match default values
   ! used in the MFPLOT X11 driver sources (i.e. x11_driver.c); these
   ! two values can be overwritten by user environment variables.
   ! Should match also the EPS and PDF drivers.
   integer, parameter :: MF_WIN_X_DEF = 200,                            &
                         MF_WIN_Y_DEF = 50,                             &
                         MF_WIN_WIDTH_DEF = 800,                        &
                         MF_WIN_HEIGHT_DEF = 600

   integer, parameter :: MIN_SIZE_WIN_HANDLES = 16

   integer, parameter :: NCOLORS = 42 ! predefined colors (see other drivers)
   integer, parameter :: DEFAULT_NB_COLORS = 256
   integer, parameter :: CI_HIGH_MAX = 4175 ! in MFPLOT X11 driver

   double precision, parameter :: MF_REALMAX_SINGLE = huge(1.0)

   real(kind=MF_DOUBLE), parameter :: DEFAULT_FONT_WIDTH = 15.0d0

   ! old MFPLOT colors
   integer, parameter :: COL_CYCLE_TAB_1(6) = [ 4, 2, 3, 5, 6, 7 ]

   ! old MATLAB colors
   integer, parameter :: COL_CYCLE_TAB_2(7) = [ 16, 17, 18, 19, 20, 21, 22 ]

   ! new MATLAB colors
   integer, parameter :: COL_CYCLE_TAB_3(7) = [ 23, 24, 25, 26, 27, 28, 29 ]

   ! BREEZE colors
   integer, parameter :: COL_CYCLE_TAB_4(12) = [ 30, 31, 32, 33, 34, 35, &
                                                 36, 37, 38, 39, 40, 41 ]

   ! cursor shapes:
   integer, parameter :: MF_LEFT_ARROW_CURSOR  = 0, &
                         MF_CROSSHAIR_CURSOR   = 1, &
                         MF_RESIZE_CURSOR      = 2, &
                         MF_WATCH_CURSOR       = 3, &
                         MF_OPENED_HAND_CURSOR = 4, &
                         MF_CLOSED_HAND_CURSOR = 5, &
                         MF_ZOOM_CURSOR        = 6, &
                         MF_TRANSPARENT_CURSOR = 7, &
                         MF_USER1_CURSOR       = 8, &
                         MF_USER2_CURSOR       = 9
   ! integer, parameter :: MF_MAX_CURSOR_INDEX = 9 - defined in PGPLOT module.

!-----------------------------------------------------------------------
!                        new types declarations
!-----------------------------------------------------------------------

   type :: grobj_struct

      character(len=40) :: cmd = ""

      integer :: hdle = 0 ! stores its own handle number

      ! used for complex objects
      real(kind=MF_DOUBLE) :: range(4) = [ MF_NAN, MF_NAN, MF_NAN, MF_NAN ]

      integer              :: npt = 0
      integer              :: npt2 = 0 ! (used in Pcolor, Hist, Contour)
      integer              :: npt3 = 0 ! (used in TriContour, TriContourF)
      integer,   pointer   :: ir(:) => null() ! (used in PcolorCoreSpySparse)
      integer,   pointer   :: jc(:) => null() ! (used in PcolorCoreSpySparse)
      real(kind=MF_DOUBLE), pointer :: abs_tab(:) => null()
      real(kind=MF_DOUBLE), pointer :: ord_tab(:) => null()
      real(kind=MF_DOUBLE), pointer :: tm1_tab(:) => null() ! ErrorBar (x_err)
      real(kind=MF_DOUBLE), pointer :: tm2_tab(:) => null() ! ErrorBar (y_err)
      real(kind=MF_DOUBLE), pointer :: lev_tab(:) => null() ! contour levels
      integer,              pointer :: col_tab(:) => null() ! colormap indexes
      real(kind=MF_DOUBLE), pointer :: val_mat(:,:) => null() ! pcolor matrix (val)
      real(kind=MF_DOUBLE), pointer :: val_mat_2(:,:) => null() ! for many purpose
      real(kind=MF_DOUBLE), pointer :: abs_mat(:,:) => null() ! pcolor matrix (abs)
      real(kind=MF_DOUBLE), pointer :: ord_mat(:,:) => null() ! pcolor matrix (ord)
      integer,              pointer :: tab_2d_1(:,:) => null() ! for many purpose
      integer,              pointer :: tab_2d_2(:,:) => null() ! for many purpose
      integer,              pointer :: int_tab(:) => null() ! for many purpose
      type(mf_Int_List),    pointer :: int_lst(:) => null() ! for many purpose
      real(kind=MF_DOUBLE) :: alpha_transp = 1.0d0 ! transparency (patch only)
      logical, pointer     :: bool_vec(:) => null() ! multi-purpose boolean vector
      logical              :: bool1 = .false. ! multi-purpose boolean fields,
      logical              :: bool2 = .false. ! for storing clipping,
      logical              :: bool3 = .false. ! draw_grid, pixvoffset, etc.
      integer              :: color = 1         ! default foreground color
      integer              :: linestyle = -127
      real(kind=MF_DOUBLE) :: linewidth = 1.0d0 ! 1/200 inch (MFPLOT)
      ! No default value pour cap and join styles. See global variables below.
      integer      :: cap_style = -1 ! 0: CapButt, 1: CapRound, 2:CapProjecting
      integer      :: join_style = -1 ! 0: JoinMiter, 1: JoinRound, 2:JoinBevel
      integer              :: marker = -127
      real(kind=MF_DOUBLE) :: markersize = 1.0d0
      logical              :: visible = .true.
      character, pointer   :: text(:) => null()
      real(kind=MF_DOUBLE) :: x_text = 0.0d0
      real(kind=MF_DOUBLE) :: y_text = 0.0d0
      real(kind=MF_DOUBLE) :: ang_text = 0.0d0
      real(kind=MF_DOUBLE) :: just_text = 0.0d0     ! horizontal justification
      real(kind=MF_DOUBLE) :: just_vert_text = 0.0d0 ! vertical justification
      real(kind=MF_DOUBLE) :: height_text = 1.0d0 ! MFPLOT default
      real(kind=MF_DOUBLE) :: height_symb = 1.0d0 ! MFPLOT default

      ! Specify that the current GrObj is tagged by a user comment
      ! in the EPS image
      logical :: add_user_comment_in_EPS = .false.
      character(len=80) :: user_comment_in_EPS = ""

      type(legend_struct), pointer :: legend => null()

      ! for PDF Optional Content
      logical            :: PDF_OC = .false.
      integer            :: PDF_OC_num = 0 ! valid numbers are > 0
      character(len=80)  :: PDF_OC_name = "" ! name appearing in the PDF viewer
      integer            :: PDF_OC_value = 1 ! 1="On", 0="Off"
      ! currently, only one Radio-Button group
      integer            :: PDF_OC_mutex = 0 ! mutually exclusive OCGs (1='yes')
      integer            :: PDF_OC_mutex_default = 0 ! default grobj in mutex
      character(len=80)  :: PDF_OC_RBGroups_name = "" ! in /Order, actually
      ! currently, only one Super-group
      integer            :: PDF_OC_sg = 0 ! group of OCGs (1='yes')
      character(len=80)  :: PDF_OC_sg_name = "" ! in /Order, actually

   end type ! grobj_struct

   !-----------------------------

   type :: legend_struct
      character(len=80), pointer :: legends(:) => null()
      integer,           pointer :: hdle(:) => null()
      integer                    :: legend_i_max ! index of the longer legend
      real(kind=MF_DOUBLE) :: x0 = 0.0d0, y0 = 0.0d0,                   &
                              DL_x = 0.0d0, DL_y = 0.0d0,               &
                              xmin = 0.0d0, xmax = 0.0d0,               &
                              ymin = 0.0d0, ymax = 0.0d0,               &
                              proto_width = 0.0d0
      logical :: outside = .false.
   end type ! legend_struct

   !-----------------------------

   type :: grobj_elem ! used in a double-linked list
      type(grobj_struct)        :: struct
      type(grobj_elem), pointer :: prev => null()
      type(grobj_elem), pointer :: next => null()
   end type ! grobj_elem

   !-----------------------------

   ! handle pour accéder directement à un maillon
   type :: grobj_handle
      type(grobj_elem), pointer :: ptr => null()
   end type ! grobj_handle

   !-----------------------------

   type :: mf_win_info

      integer :: mfplot_id = 0
      character(len=80) :: win_title ! for X11 and EPS title
      logical :: save_geometry = .false. ! if .true., the window geometry
                                         ! will be saved in a hidden file,
                                         ! when the windows will be closed.
      integer :: geometry(4) ! X11 or requested equivalent X11 geometry
      logical :: must_be_redrawn = .false.
      logical :: hold = .false.
      logical :: transparency_used = .false.
      logical :: blank = .true. ! page complètement blanche
                                ! (indép. du nbr d'objets graph.)
      logical :: empty = .true. ! no grobj
                                ! (devient 'false' si 'pgenv' appelé)
      logical :: axis_drawn = .false. ! to force call of mf_win_draw_box
                                      ! when Plot() is called after Text()

      integer :: nx = 0
      integer :: ny = 0
      real(kind=MF_DOUBLE), pointer :: x(:) => null()
      real(kind=MF_DOUBLE), pointer :: y(:) => null()
      real(kind=MF_DOUBLE), pointer :: z(:) => null()

      real(kind=MF_DOUBLE) :: current_axes(4) = 0.0d0 ! [ 0.0d0, 1.0d0, 0.0d0, 1.0d0 ]
                                                ! (keeep the zoom level)
      real(kind=MF_DOUBLE) :: abs_zoom_factor = 1.0d0
      real(kind=MF_DOUBLE) :: viewport(4) = 0.0d0 ! sert à sauvegarder le viewport
                                                  ! en cas de ClipBox, en dehors
                                                  ! d'un niveau de zoom

      character(len=8)     :: shading = "flat" ! "flat" | "interp"
      real(kind=MF_DOUBLE) :: color_axes(2) = [ 0.0d0, 1.0d0 ] ! min max for colormap
      logical              :: color_axis_set = .false.
      logical              :: color_axis_extensible = .true.

      integer              :: colorbar = 0 ! 0:no colorbar, 1: linear, 2: log
      character(len=2)     :: colorbar_pos = "" ! "ri": right, "bi": bottom
      character(len=80)    :: colorbar_label = ""

      ! selected color scheme for the window
      ! (1 to 4; the default 3 = new Matlab colors)
      integer :: color_scheme = 3

      logical :: axis_on     = .true.  ! set by: "on" (shown) | "off" (hidden)
      integer :: axis_manual_x = 0     ! 0 = "auto", 1 = "manual", 2 = "tight"
      integer :: axis_manual_y = 0     ! 0 = "auto", 1 = "manual", 2 = "tight"

      integer :: axis_scale_x = 0      ! 0 = undef., 1 = linear, 2 = log10
      integer :: axis_scale_y = 0      ! 0 = undef., 1 = linear, 2 = log10
      logical :: axis_equal  = .false. ! set by: "equal" | "auto" (non-equal)

      logical :: axis_mode_xy = .true. ! 'xy' mode (opposed to 'ij' mode)

      logical :: axis_time_x = .false. ! std, TRUE => time labelling
      logical :: axis_time_y = .false. ! std, TRUE => time labelling

      integer :: ind_next_color = 1 ! between 1 and 6=size(COL_CYCLE_TAB)

      logical :: grid_on = .false. ! dashed grey lines on major ticks
      logical :: minor_grid = .false. ! dashed grey lines on minor ticks

      ! for legend
      type(legend_struct), pointer :: legend => null()

      ! for animation
      logical :: mf_win_db_active = .true. ! FALSE: avoid redrawing

      real(kind=MF_DOUBLE) :: axis_line_width = 1.0d0 ! default line width for axis
      real(kind=MF_DOUBLE) :: axis_font_size = 1.0d0 ! default character height for axis
      real(kind=MF_DOUBLE) :: label_font_size = 1.0d0 ! default character height for labels
      real(kind=MF_DOUBLE) :: title_font_size = 2.0d0 ! default character height for title
      logical :: char_height_pixel = .false. ! default : unit for char height
                                             ! is 1/40 of view surface height
                                ! if true : unit is the same, but the three
                                ! previous item are modified to verify :
                                !           1 <-> 12 pixels
      real(kind=MF_DOUBLE) :: char_height_factor = -1.0d0 ! to force initialization
      real(kind=MF_DOUBLE) :: char_width_factor = -1.0d0 ! to force initialization

      ! there is usual grobjs of type "xlabel", "ylabel" and "title"
      type(grobj_struct), pointer :: xlabel_grobj => null()
      type(grobj_struct), pointer :: ylabel_grobj => null()
      type(grobj_struct), pointer :: title_grobj => null()

      logical :: xlabel_exist = .false.
      logical :: ylabel_exist = .false.
      logical :: title_exist = .false.

      ! accesses to the double linked list of graphic objects
      type(grobj_elem), pointer :: grobj_head => null()
      type(grobj_elem), pointer :: grobj_tail => null()

      ! sequential list of handles
      type(grobj_handle), pointer :: handles(:) => null()
      integer :: hdle_last_ind = 0
      ! associated list of free handle indices
      integer, pointer :: handles_free(:) => null()
      integer :: hdle_free_last_ind = 0

      ! gestion des contenus optionnels pour le PDF (Optional Content)
      integer :: pdf_ocg_nb = 0

      ! colormap
      integer :: colormap_ci_low = NCOLORS,                             &
                 colormap_ci_high = NCOLORS + DEFAULT_NB_COLORS - 1
      logical :: colormap_init = .false. ! true if colormap has been initialized.
      logical :: colormap_used = .false. ! true if really used
                                      ! (useful to know if the EPS/PDFdriver
                                      !  must initialize its own colormap)
      logical :: predefined_colormap = .true. ! false if user-defined
      character(len=20) :: colormap_name = "" ! empty if user-defined
      logical :: colormap_inverted = .false. ! not used if user-defined
      real(kind=MF_DOUBLE), allocatable :: user_colormap(:,:) ! shape=(:,3),
                                           ! allocated only if user-defined

      logical :: colormap_none_removed = .false.

      ! in EPS and PDF: to avoid redrawing the frame box at the end of the
      ! redrawing, in the case where at least one grobj is not clipped.
      logical :: at_least_one_grobj_not_clipped = .false.

      logical :: user_cursors_defined = .false.

   end type ! mf_win_info

   !-----------------------------

   type(mf_win_info), save, protected :: mf_win_info_empty

!-----------------------------------------------------------------------
!                         FGL global variables
!-----------------------------------------------------------------------

   ! keep track of initialization (or opening any window), to avoid disabling
   ! the X11 device during plotting. Only msExitFgl() reset this variable to 0.
   integer :: MF_INIT_FGL = 0

   character(len=1024) :: MUESLI_EXE_NAME = ""

   type(mf_win_info), save, target :: MF_WIN_DB(MF_WIN_NB_MAX)

   ! numéro de la fenêtre courante
   integer :: CURRENT_WIN_ID = 0

   logical :: FGL_MFPLOT_BUFFER = .true.

   integer :: MFPLOT_BUF_LEVEL = 0

   integer :: COLOR_OVERFLOW_LOW_POLICY  = 0 ! signaled via BACKG color
                                         ! 1 ! truncated to lowest color

   integer :: COLOR_OVERFLOW_HIGH_POLICY = 0 ! signaled via FOREG color
                                         ! 1 ! truncated to highest color

   ! Black on white -> BLACK_ON_WHITE = 1 [default value]
   ! White on black -> BLACK_ON_WHITE = 0
   ! Initial value (-1) allows to know whether MFPLOT has been already
   ! opened. (cf. test in msSetBackgroundColor)
   integer :: BLACK_ON_WHITE = -1

   ! nombre de couleurs 'user-defined' (par RGB ou par noms de rgb.txt).
   ! Ces indices partent de CI_HIGH_MAX-1 et remontent à l'envers le
   ! tableau. (cf. 'decode_col_name', 'decode_col_rgb')
   ! CI_HIGH_MAX est gardé comme indice tempo.
   integer :: NB_USER_COL = 0

#ifdef _NO_X11
   logical :: X11_DEVICE = .false.
   logical :: NULL_DEVICE = .true.
#else
   logical :: X11_DEVICE = .true.
   logical :: NULL_DEVICE = .false.
#endif

   ! registering the X11 color depth, to avoid requiring it from X11 for
   ! each call of PatchRectCorePX... (-1 means undefined)
   integer :: X11_COLOR_DEPTH = -1

   logical :: UTF8_ENCODING = .false. ! .true. for Latin-1 (Iso-8859-1)

   integer :: LAST_MF_WIN_X = 0, LAST_MF_WIN_Y = 0

   ! The default Cap Style for all lines
   !   0: CapButt, 1: CapRound, 2:CapProjecting
   ! (can be reset by msSetDefaultCapStyle)
   integer :: MF_DEFAULT_CAP_STYLE = 1

   ! The default Join Style for all lines
   !   0: JoinMiter, 1: JoinRound, 2:JoinBevel
   ! (can be reset by msSetDefaultJoinStyle)
   integer :: MF_DEFAULT_JOIN_STYLE = 1

contains
!_______________________________________________________________________
!
   subroutine create_grobj( win, grobj )

      type(mf_win_info), pointer :: win
      type(grobj_elem), pointer :: grobj
      !------ API end ------

      allocate( grobj )

      if( associated(win%grobj_tail) ) then
         grobj%prev => win%grobj_tail
         win%grobj_tail%next => grobj
         win%grobj_tail => grobj
      else
         ! first element in the double linked list
         win%grobj_head => grobj
         win%grobj_tail => grobj
      end if

   end subroutine create_grobj
!_______________________________________________________________________
!
end module mod_win_db
