! f90 include file
!_______________________________________________________________________
!
   subroutine update_mf_win_pos_db( win_id, win_pos )

      integer, intent(in) :: win_id, win_pos(4)
      !------ API end ------

      integer :: win_pos_db(MF_WIN_NB_MAX,4)
      integer :: id, tmp(4), iostat, unit
      logical :: exist
      character(len=512) :: filename
      character(len=9) :: action

   !------ end of declarations -- execution starts hereafter  ------

      ! find an available unit number
      call find_unit ( unit )
      if( unit == -1 ) then
         write(STDERR,*) "(MUESLI update_mf_win_pos_db:) info: no available IO unit"
         call msMuesliTrace( pause="no" )
         return
      end if

      win_pos_db(:,:) = 0

      ! check for the presence of the database file
      if( len_trim(MUESLI_EXE_NAME) /= 0 ) then
         filename = "." // trim(MUESLI_EXE_NAME) // WIN_POS_DB_NAME
      else
         filename = WIN_POS_DB_NAME
      end if
      inquire( file=trim(filename), exist=exist )

      if( exist ) then
         ! scanning the whole file
         open( unit=unit, file=trim(filename) )
         do
            read( unit=unit, fmt=*, iostat=iostat ) id, tmp(:)
            if( iostat < 0 ) exit
            if( iostat > 0 ) then
               write(STDERR,*) "(MUESLI update_mf_win_pos_db:) info: found incorrect values"
               write(STDERR,*) "                               in win_pos_db file for X11 window."
               call msMuesliTrace( pause="no" )
               exit
            end if
            if( id < 1 .or. MF_WIN_NB_MAX < id ) then
               write(STDERR,*) "(MUESLI update_mf_win_pos_db:) internal error!"
               call msMuesliTrace( pause="yes" )
               stop
            else
               win_pos_db( id, : ) = tmp(:)
            end if
         end do
         close(unit)
      end if

      ! update
      win_pos_db( win_id, : ) = win_pos(:)

      ! write
      open( unit=unit, file=trim(filename) )
      ! user may have protect an existing file to prevent the update of another size.
      inquire( file=trim(filename), action=action )
      if( action == "READWRITE" ) then
         do id = 1, MF_WIN_NB_MAX
            if( all( win_pos_db(id,3:4) /= 0 ) ) then
               write( unit=unit, fmt=* ) id, win_pos_db(id,:)
            end if
         end do
      end if
      close(unit)

   end subroutine update_mf_win_pos_db
!_______________________________________________________________________
!
   subroutine read_mf_win_pos_db( win_id, win_pos, win_shift )

      integer, intent(in)           :: win_id
      integer, intent(out)          :: win_pos(4)
      logical, intent(in), optional :: win_shift
      !------ API end ------

      integer :: win_pos_db(MF_WIN_NB_MAX,4)
      integer :: id, tmp(4), iostat, unit
      logical :: exist
      character(len=512) :: filename
      integer :: win_x, win_y, win_width, win_height

   !------ end of declarations -- execution starts hereafter  ------

      win_pos_db(:,:) = 0

      ! test de présence du fichier db
      if( len_trim(MUESLI_EXE_NAME) /= 0 ) then
         filename = "." // trim(MUESLI_EXE_NAME) // WIN_POS_DB_NAME
      else
         filename = WIN_POS_DB_NAME
      end if

      inquire( file=trim(filename), exist=exist )

      if( exist ) then
         ! on cherche un numéro d'unité disponible
         call find_unit ( unit )
         if( unit == -1 ) then
            write(STDERR,*) "(MUESLI read_mf_win_pos_db:) info: no available IO unit"
            call msMuesliTrace( pause="no" )
            return
         end if
         ! relecture complet du fichier (tous les id présents)
         open( unit=unit, file=trim(filename) )
         do
            read( unit=unit, fmt=*, iostat=iostat ) id, tmp(:)
            ! iostat = EOF or EOR is ok
            if( iostat < 0 ) exit
            if( iostat > 0 ) then
               write(STDERR,*) "(MUESLI read_mf_win_pos_db:) info: found incorrect values"
               write(STDERR,*) "                             in win_pos_db file for X11 window."
               call msMuesliTrace( pause="no" )
               exit
            end if
            if( id < 1 .or. MF_WIN_NB_MAX < id ) then
               write(STDERR,*) "(MUESLI read_mf_win_pos_db:) internal error!"
               call msMuesliTrace( pause="yes" )
               stop
            else
               win_pos_db( id, : ) = tmp(:)
            end if
         end do
         close(unit)
      end if

      ! extraction
      ! too small or too large values for 'size' are discarded
      if( all( 10 <= win_pos_db(win_id,3:4) ) .and.                     &
          all(       win_pos_db(win_id,3:4) <= 5000 ) ) then
         win_pos(:) = win_pos_db(win_id,:)
         ! negative values for 'position' are not yet supported
         if( any( win_pos(1:2) < 0 ) ) then
            win_pos(1:2) = [ 0, 0 ]
         end if
         ! too large values for 'position' are discarded
         if( any( win_pos(1:2) > 5000 ) ) then
            win_pos(1:2) = [ 0, 0 ]
         end if
      else
         call get_default_x11_size( win_width, win_height )
         ! default values for pos. and size of a window
!!         win_pos(:) = [ MF_WIN_X_DEF, MF_WIN_Y_DEF,                     &
!!                        win_width, win_height ]
         ! It is preferable to not use a constant (X,Y) position of the
         ! window (hide windows is not a good idea). Here, each window is
         ! shifted from the previous one by a constant... this is
         ! CASCADE WINDOWS.
         if( present(win_shift) ) then
            if( win_shift ) then
               win_x = LAST_MF_WIN_X + MF_WIN_X_OFFSET
               win_y = LAST_MF_WIN_Y + MF_WIN_Y_OFFSET
            else
               win_x = LAST_MF_WIN_X
               win_y = LAST_MF_WIN_Y
            end if
         else ! default is shift
            win_x = LAST_MF_WIN_X + MF_WIN_X_OFFSET
            win_y = LAST_MF_WIN_Y + MF_WIN_Y_OFFSET
         end if
         win_pos(:) = [ win_x, win_y,                                   &
                        win_width, win_height ]
         LAST_MF_WIN_X = win_x
         LAST_MF_WIN_Y = win_y
      end if

   end subroutine read_mf_win_pos_db
!_______________________________________________________________________
!
   subroutine get_default_x11_size( win_width, win_height )

      integer, intent(out) :: win_width, win_height
      !------ API end ------

      integer :: str_len, iostat
      character(len=8) :: string

   !------ end of declarations -- execution starts hereafter  ------

      ! check presence of non default size via environment variables
      call grgenv( "X11_WIDTH", string, str_len ) ! grgenv add "MFPLOT_"
      if( str_len == 0 ) then
         win_width = MF_WIN_WIDTH_DEF
      else
         read(string,*,iostat=iostat) win_width
         if( iostat /= 0 ) then
            ! something wrong happened during the read in string,
            ! fallback to default value
            win_width = MF_WIN_WIDTH_DEF
         end if
      end if
      call grgenv( "X11_HEIGHT", string, str_len ) ! grgenv add "MFPLOT_"
      if( str_len == 0 ) then
         win_height = MF_WIN_HEIGHT_DEF
      else
         read(string,*,iostat=iostat) win_height
         if( iostat /= 0 ) then
            ! something wrong happened during the read in string,
            ! fallback to default value
            win_height = MF_WIN_HEIGHT_DEF
         end if
      end if

   end subroutine get_default_x11_size
!_______________________________________________________________________
!
   subroutine erase_mf_win( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      ! Clear the window surface and remove all graphic objects in the window.

      type(mf_win_info), pointer :: win
      type(grobj_elem), pointer :: grobj

      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      ! clear the window surface
      call grepic
      call grbpic

      win => mf_win_db(win_id)

      win%blank = .true.
      win%empty = .true.
      win%axis_drawn = .false.
      win%transparency_used = .false.

      if( associated( win%legend ) ) then
         if( associated( win%legend%legends ) ) then
            deallocate( win%legend%legends )
         end if
         deallocate( win%legend )
         win%legend => null()
      end if

      ! erase_mf_win is internally called each time we have a new page
      ! (i.e. at each new drawing in "hold off" mode)

      ! most of global properties of the window are "sticky" (see at the
      ! end of this routine), because they must be kept even we have a
      ! new page...

      ! removing all memorised grobjs, only if they belong to the
      ! appropriate y_axis.

      grobj => win%grobj_head

      do

         if( .not. associated(grobj) ) then
            exit
         end if

         call delete_grobj_inside( grobj )

         if( associated(grobj%next) ) then
            grobj => grobj%next
            deallocate( grobj%prev )
         else
            deallocate( grobj )
            exit
         end if

      end do

      win%grobj_head => null()
      win%grobj_tail => null()

      if( associated( win%handles ) ) then
         win%hdle_last_ind = 0
      end if
      if( associated( win%handles_free ) ) then
         win%hdle_free_last_ind = 0
      end if

      win%ind_next_color = 1

      ! sinon, il y a un REDRAW !
      if( associated(win%xlabel_grobj) ) then
         deallocate( win%xlabel_grobj%text )
         deallocate( win%xlabel_grobj )
         win%xlabel_grobj => null()
      end if
      if( associated(win%ylabel_grobj) ) then
         deallocate( win%ylabel_grobj%text )
         deallocate( win%ylabel_grobj )
         win%ylabel_grobj => null()
      end if
      if( associated(win%title_grobj) ) then
         deallocate( win%title_grobj%text )
         deallocate( win%title_grobj )
         win%title_grobj => null()
      end if

   end subroutine erase_mf_win
!_______________________________________________________________________
!
   subroutine erase_viewport( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      ! Erase current axes (equivalent of 'cla' of Matlab).
      !
      ! Graphics objects are ignored. Usually, it is called in an
      ! animation, during which grobjs are not recorded.

   !------ end of declarations -- execution starts hereafter  ------

      ! Select background color : 0
      call grsci( 0 )

      ! Fill the rectangle exactly in the axes (with an offset of one
      ! pixel around the axes, to avoid erasing the box)
      call grrec0( grxmin(grcide)+1, grymin(grcide)+1,                  &
                   grxmax(grcide)-1, grymax(grcide)-1, filled=.true.  )

   end subroutine erase_viewport
!_______________________________________________________________________
!
   subroutine erase_axes_and_labels( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      ! Erase only axes and their numerical labels (not the viewport,
      ! neither the XLabel, YLabel and Title).

      double precision :: rbuf(1)
      integer :: ibuf(1), lchr
      character(len=1) :: chr

   !------ end of declarations -- execution starts hereafter  ------

      ! Select background color : 0
      call grsci( 0 )

      ! Fill the area between two rectangles
      ibuf(1) = 0
      call grexec( grgtyp, ERASE_LABEL_AREA_AROUND, rbuf, ibuf, chr, lchr )

   end subroutine erase_axes_and_labels
!_______________________________________________________________________
!
   subroutine dealloc_mf_win( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      type(mf_win_info), pointer :: win
      type(grobj_elem), pointer :: grobj

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      if( associated( win%x ) ) then
         deallocate( win%x )
      end if
      if( associated( win%y ) ) then
         deallocate( win%y )
      end if
      if( associated( win%z ) ) then
         deallocate( win%z )
      end if
      if( associated( win%legend ) ) then
         if( associated( win%legend%legends ) ) then
            deallocate( win%legend%legends )
         end if
         if( associated( win%legend%hdle ) ) then
            deallocate( win%legend%hdle )
         end if
         deallocate( win%legend )
      end if

      call dealloc_all_grobjs( win_id )

      if( associated( win%handles ) ) then
         deallocate( win%handles )
         win%hdle_last_ind = 0
      end if
      if( associated( win%handles_free ) ) then
         deallocate( win%handles_free )
         win%hdle_free_last_ind = 0
      end if

      if( associated(win%xlabel_grobj) ) then
         deallocate( win%xlabel_grobj%text )
         deallocate( win%xlabel_grobj )
         win%xlabel_grobj => null()
      end if
      if( associated(win%ylabel_grobj) ) then
         deallocate( win%ylabel_grobj%text )
         deallocate( win%ylabel_grobj )
         win%ylabel_grobj => null()
      end if
      if( associated(win%title_grobj) ) then
         deallocate( win%title_grobj%text )
         deallocate( win%title_grobj )
         win%title_grobj => null()
      end if

      ! more safe to return to default values for all fields in the
      ! mf_win_info structure !
      win = mf_win_info_empty

   end subroutine dealloc_mf_win
!_______________________________________________________________________
!
   subroutine dealloc_all_grobjs( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      type(mf_win_info), pointer :: win
      type(grobj_elem), pointer :: grobj

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      grobj => win%grobj_head

      do

         if( .not. associated(grobj) ) then
            exit
         end if

         call delete_grobj_inside( grobj )

         if( associated( grobj%next ) ) then
            grobj => grobj%next
            deallocate( grobj%prev )
         else
            deallocate( grobj )
            exit
         end if

      end do

      win%grobj_head => null()
      win%grobj_tail => null()

   end subroutine dealloc_all_grobjs
!_______________________________________________________________________
!
   subroutine delete_grobj_inside( grobj )

      type(grobj_elem) :: grobj
      !------ API end ------

   !------ end of declarations -- execution starts hereafter  ------

      if( associated( grobj%struct%ir ) ) then
         deallocate( grobj%struct%ir )
      end if
      if( associated( grobj%struct%jc ) ) then
         deallocate( grobj%struct%jc )
      end if
      if( associated( grobj%struct%abs_tab ) ) then
         deallocate( grobj%struct%abs_tab )
      end if
      if( associated( grobj%struct%ord_tab ) ) then
         deallocate( grobj%struct%ord_tab )
      end if
      if( associated( grobj%struct%tm1_tab ) ) then
         deallocate( grobj%struct%tm1_tab )
      end if
      if( associated( grobj%struct%tm2_tab ) ) then
         deallocate( grobj%struct%tm2_tab )
      end if
      if( associated( grobj%struct%lev_tab ) ) then
         deallocate( grobj%struct%lev_tab )
      end if
      if( associated( grobj%struct%col_tab ) ) then
         deallocate( grobj%struct%col_tab )
      end if
      if( associated( grobj%struct%val_mat ) ) then
         deallocate( grobj%struct%val_mat )
      end if
      if( associated( grobj%struct%val_mat_2 ) ) then
         deallocate( grobj%struct%val_mat_2 )
      end if
      if( associated( grobj%struct%abs_mat ) ) then
         deallocate( grobj%struct%abs_mat )
      end if
      if( associated( grobj%struct%ord_mat ) ) then
         deallocate( grobj%struct%ord_mat )
      end if
      if( associated( grobj%struct%tab_2d_1 ) ) then
         deallocate( grobj%struct%tab_2d_1 )
      end if
      if( associated( grobj%struct%tab_2d_2 ) ) then
         deallocate( grobj%struct%tab_2d_2 )
      end if
      if( associated( grobj%struct%bool_vec ) ) then
         deallocate( grobj%struct%bool_vec )
      end if
      if( associated( grobj%struct%int_tab ) ) then
         deallocate( grobj%struct%int_tab )
      end if
      if( associated( grobj%struct%int_lst ) ) then
         call msRelease( grobj%struct%int_lst )
      end if
      if( associated( grobj%struct%text ) ) then
         deallocate( grobj%struct%text )
      end if
      if( associated( grobj%struct%legend ) ) then
         if( associated( grobj%struct%legend%legends ) ) then
            deallocate( grobj%struct%legend%legends )
         end if
         if( associated( grobj%struct%legend%hdle ) ) then
            deallocate( grobj%struct%legend%hdle )
         end if
         deallocate( grobj%struct%legend )
      end if

   end subroutine delete_grobj_inside
!_______________________________________________________________________
!
   subroutine mf_win_handles_init( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      type(mf_win_info), pointer :: win

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      ! allocate the (dynamic) vector of handles for the window 'win_id'
      allocate( win%handles(MIN_SIZE_WIN_HANDLES) )

   end subroutine mf_win_handles_init
!_______________________________________________________________________
!
   subroutine mf_win_handles_free_init( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      type(mf_win_info), pointer :: win

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      ! allocate the (dynamic) vector of free handles for the window 'win_id'
      allocate( win%handles_free(MIN_SIZE_WIN_HANDLES) )

   end subroutine mf_win_handles_free_init
!_______________________________________________________________________
!
   subroutine mf_win_handles_increase_size( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      type(grobj_handle), dimension(:), pointer :: handles_tmp => null()

      type(mf_win_info), pointer :: win
      integer :: i, current_size, new_size

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      ! size of the handles' vector is multiplied by 2
      current_size = size(win%handles)
      new_size = 2*current_size

      allocate( handles_tmp(new_size) )

      do i = 1, current_size
         handles_tmp(i)%ptr => win%handles(i)%ptr
      end do

      deallocate( win%handles )

      allocate( win%handles(new_size) )

      do i = 1, current_size
         win%handles(i)%ptr => handles_tmp(i)%ptr
      end do

      deallocate( handles_tmp )

   end subroutine mf_win_handles_increase_size
!_______________________________________________________________________
!
   subroutine mf_win_handles_free_increase_size( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      integer, pointer :: handles_free_tmp(:)

      type(mf_win_info), pointer :: win
      integer :: i, current_size, new_size

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      ! size of the handles' free vector is multiplied by 2
      current_size = size(win%handles_free)
      new_size = 2*current_size

      allocate( handles_free_tmp(new_size) )

      do i = 1, current_size
         handles_free_tmp(i) = win%handles_free(i)
      end do

      deallocate( win%handles_free )

      allocate( win%handles_free(new_size) )

      do i = 1, current_size
         win%handles_free(i) = handles_free_tmp(i)
      end do

      deallocate( handles_free_tmp )

   end subroutine mf_win_handles_free_increase_size
!_______________________________________________________________________
!
   function mf_win_get_free_handle( win_id ) result( handle )

      integer, intent(in)  :: win_id
      integer              :: handle
      !------ API end ------

      ! returns a free index in the handles' vector

      type(mf_win_info), pointer :: win
      integer :: current_size

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      if( .not. associated(win%handles) ) then
         call mf_win_handles_init(CURRENT_WIN_ID)
      end if

      if( win%hdle_free_last_ind == 0 ) then
         win%hdle_last_ind = win%hdle_last_ind + 1
         if( win%hdle_last_ind > size(win%handles) ) then
            call mf_win_handles_increase_size( CURRENT_WIN_ID )
         end if
         handle = win%hdle_last_ind
      else
         handle = win%handles_free(win%hdle_free_last_ind)
         win%hdle_free_last_ind = win%hdle_free_last_ind - 1
      end if

   end function mf_win_get_free_handle
!_______________________________________________________________________
!
   subroutine mf_win_redraw_pre_PS( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      ! called only by print_eps, in case where transparency is used.
      type(mf_win_info), pointer :: win
      real(kind=MF_DOUBLE) :: x_min, x_max, y_min, y_max
      integer :: i, j, len_text, just, colbar, xlabel_pos
      character(len=96) :: xlabel_tmp, ylabel_tmp
      character(len=128) :: title_tmp
      real(kind=MF_DOUBLE) :: disp

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      x_min = win%current_axes(1)
      x_max = win%current_axes(2)
      y_min = win%current_axes(3)
      y_max = win%current_axes(4)

      call pgsch(1.0d0) ! reset to unity to have std labelling margins

      if( win%colorbar == 0 ) then
         colbar = 0
      else
         if( win%colorbar_pos == "ri" ) then
            colbar = 1
         else ! colorbar_pos = "bi"
            colbar = 2
         end if
      end if
      if( win%axis_equal ) then
         ! scaled axis
         just = 1
      else
         ! automatic axis
         just = 0
      end if
      if( win%axis_mode_xy ) then
         xlabel_pos = 1
      else
         xlabel_pos = 2
      end if
      call pgenv( x_min, x_max, y_min, y_max, just, colbar, xlabel_pos )

      call char_height_factor_update( win_id )

      if( associated(win%xlabel_grobj) ) then
         if( associated(win%xlabel_grobj%text) ) then
            len_text = size(win%xlabel_grobj%text)
         else
            len_text = 0
         end if
         if( len_text /= 0 ) then
            do j = 1, len_text
               xlabel_tmp(j:j) = win%xlabel_grobj%text(j)
            end do
         else
            xlabel_tmp = ""
         end if
      else
         xlabel_tmp = ""
      end if

      if( associated(win%ylabel_grobj) ) then
         if( associated(win%ylabel_grobj%text) ) then
            len_text = size(win%ylabel_grobj%text)
         else
            len_text = 0
         end if
         if( len_text /= 0 ) then
            do j = 1, len_text
               ylabel_tmp(j:j) = win%ylabel_grobj%text(j)
            end do
         else
            ylabel_tmp = ""
         end if
      else
         ylabel_tmp = ""
      end if

      if( associated(win%title_grobj) ) then
         if( associated(win%title_grobj%text) ) then
            len_text = size(win%title_grobj%text)
         else
            len_text = 0
         end if
         if( len_text /= 0 ) then
            do j = 1, len_text
               title_tmp(j:j) = win%title_grobj%text(j)
            end do
         else
            title_tmp = ""
         end if
      else
         title_tmp = ""
      end if

      gr_pdf_color_intent = 3 ! Stroke & Fill
      call grsci(1) ! default foreground color
      call pgsch( win%title_font_size * win%char_height_factor ) ! character height
      call pglab( "", xlabel_pos, "", title_tmp )
      if( win%axis_on ) then
         call pgsch( win%label_font_size * win%char_height_factor ) ! character height
         call pglab( xlabel_tmp, xlabel_pos, ylabel_tmp, "" )
      end if

      if( win%colorbar /= 0 ) then ! colorbar present
         ! Shift in y to encompass the ticks and the numerical labels
         ! (unit is axis_font_size); depends also on 'xy'/'ij' axis mode.
         if( win%colorbar_pos(1:1) == "b" ) then
            if( win%axis_mode_xy ) then
               disp = 2.75d0
               if( win%xlabel_exist ) then
                  ! additional shift to encompass the descriptive label
                  disp = disp + 1.5d0*(win%label_font_size/win%axis_font_size)
               end if
            else ! 'ij'
               disp = 2.0d0
            end if
         else ! "vert"
            disp = 2.25d0
         end if
         ! Very important: the axis font size must be selected just before
         !                 calling pgwedg.
         call pgsch( win%axis_font_size * win%char_height_factor )
         call pgwedg( win%colorbar_pos, disp, 3.5d0,                    &
                      win%color_axes(1), win%color_axes(2),             &
                      win%colorbar_label, win%colorbar )
      end if

   end subroutine mf_win_redraw_pre_PS
!_______________________________________________________________________
!
   subroutine mf_win_draw_box( win_id, draw_labels, draw_grid )

      integer, intent(in)           :: win_id
      logical, intent(in), optional :: draw_labels, draw_grid
      !------ API end ------

      ! device independant !

      type(mf_win_info), pointer :: win
      real(kind=MF_DOUBLE) :: x_min, x_max, y_min, y_max
      character(len=8) :: xopt, yopt
      logical :: xopt_log, yopt_log
      logical :: draw_grid_requested, draw_labels_requested
      integer :: icol_gray
      integer :: just, colbar, xlabel_pos

      double precision :: rbuf(1)
      integer :: ibuf(1), lchr
      character(len=1) :: chr

   !------ end of declarations -- execution starts hereafter  ------

      if( present(draw_labels) ) then
         draw_labels_requested = draw_labels
      else
         draw_labels_requested = .true.
      end if
      if( present(draw_grid) ) then
         draw_grid_requested = draw_grid
      else
         draw_grid_requested = .true.
      end if

      win => mf_win_db(win_id)

      if( all( win%current_axes(:) == 0.0d0 ) ) then
         ! par défaut, on choisit des échelles linéaires.
         win%axis_scale_x = 1
         win%axis_scale_y = 1
         win%current_axes(:) = [ 0.0d0, 1.0d0, 0.0d0, 1.0d0 ]
      end if

      if( win%blank ) then
         x_min = win%current_axes(1)
         x_max = win%current_axes(2)
         y_min = win%current_axes(3)
         y_max = win%current_axes(4)

         if( win%axis_scale_x == 2 ) then
            x_min = log10( x_min )
            x_max = log10( x_max )
         end if
         if( win%axis_scale_y == 2 ) then
            y_min = log10( y_min )
            y_max = log10( y_max )
         end if

         if( win%char_height_factor < 0.0d0 ) then
            ! must be defined
            call char_height_factor_update( win_id )
         end if
         call pgsch( win%axis_font_size * win%char_height_factor )

         if( win%colorbar == 0 ) then
            colbar = 0
         else
            if( win%colorbar_pos == "ri" ) then
               colbar = 1
            else ! colorbar_pos = "bi"
               colbar = 2
            end if
         end if
         if( win%axis_equal ) then
            ! scaled axis
            just = 1
         else
            ! automatic axis
            just = 0
         end if
         if( win%axis_mode_xy ) then
            xlabel_pos = 1
         else
            xlabel_pos = 2
         end if
         call pgenv( x_min, x_max, y_min, y_max, just, colbar, xlabel_pos )
      end if

      if( win%axis_on ) then

         ! Ici, on pratique l'économie : on ne change les choses, temporairement,
         ! que si c'est nécessaire
         if( CLIPPING_IN_AXES ) then
            ibuf(1) = 0 ! no clipping
            call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr ) ! set clipping
         end if

         gr_pdf_color_intent = 3 ! Stroke & Fill
         call grsci(1) ! default foreground color
         call grslw(win%axis_line_width) ! default line width
         call grsls( 1 ) ! default line style
         if( win%char_height_factor < 0.0d0 ) then
            ! must be defined
            call char_height_factor_update( win_id )
         end if
         call pgsch( win%axis_font_size * win%char_height_factor ) ! character height
         ! options pour le tracé du cadre :
         !   b = bottom (X) or left (Y)
         !   c = top (X) or right (Y)
         !   n = numeric labels in the conventional location
         !       e.g. bottom for (X)
         !   m = numeric labels in the unconventional location
         !       e.g. top for (X)
         !   s = minor tick marks (subticks)
         !   t = major tick marks
         !   i = invert the tick marks
         !   l = label axis logarithmically

         if( win%axis_mode_xy ) then
            if( draw_labels_requested ) then
               xopt = "bcnsti"
            else
               xopt = "bcsti"
            end if
         else ! axis mode = "ij"
            if( draw_labels_requested ) then
               xopt = "bcmsti"
            else
               xopt = "bcsti"
            end if
         end if
         if( draw_labels_requested ) then
            yopt = "bcnsti"
         else
            yopt = "bcsti"
         end if
         if( win%axis_scale_x == 2 ) then
            xopt = trim(xopt) // "l"
            xopt_log = .true.
         else
            xopt_log = .false.
         end if
         if( win%axis_scale_y == 2 ) then
            yopt = trim(yopt) // "l"
            yopt_log = .true.
         else
            yopt_log = .false.
         end if

         if( mf_win_db(CURRENT_WIN_ID)%axis_time_x ) then
            xopt = trim(xopt) // "zhfo"
         end if

         if( mf_win_db(CURRENT_WIN_ID)%axis_time_y ) then
            yopt = trim(yopt) // "zhfo"
         end if

         if( win%grid_on ) then
            ! Plotting grid
            if( draw_grid_requested ) then
               if( BLACK_ON_WHITE == 1 ) then
                  icol_gray = MFPLOT_LIGHT_GREY
               else
                  icol_gray = MFPLOT_DARK_GREY
               end if
               call pggrid_ec( xopt_log, yopt_log, win%minor_grid, icol_gray )
               ! reset to default line style (grid uses dashed lines)
               call grsls(1)
            end if
         end if

         ! PGTBOX has the same arguments as PGBOX but has the additional
         ! feature in labelling TIME in (DD) HH MM SS.S format.
         call pgtbox( trim(xopt), MF_SMART_XTICK(pgid), MF_SMART_NXSUB(pgid), &
                      trim(yopt), MF_SMART_YTICK(pgid), MF_SMART_NYSUB(pgid) )

         win%blank = .false.
         win%axis_drawn = .true.

         if( CLIPPING_IN_AXES ) then
            ibuf(1) = 1 ! clipping at viewport
            call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr ) ! set clipping
         end if

      end if

   end subroutine mf_win_draw_box
!_______________________________________________________________________
!
   subroutine mf_redraw_axis_lines( win_id )

      integer, intent(in)           :: win_id
      !------ API end ------

      type(mf_win_info), pointer :: win
      real(kind=MF_DOUBLE) :: xblc, xtrc, yblc, ytrc
      logical :: clipping_disabled

      double precision :: rbuf(1)
      integer :: ibuf(1), lchr
      character(len=1) :: chr

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      if( win%axis_on ) then

         ! Ici, on pratique l'économie : on ne change les choses, temporairement,
         ! que si c'est nécessaire
         clipping_disabled = .false.
         if( CLIPPING_IN_AXES ) then
            ibuf(1) = 0 ! no clipping
            call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr ) ! set clipping
            CLIPPING_IN_AXES = .false.
            clipping_disabled = .true.
         end if

         call grsci(1) ! default foreground color
         call grslw(win%axis_line_width) ! default line width
         call grsls(1) ! default line style
         call pgqwin( xblc, xtrc, yblc, ytrc )

         call pgbbuf
         call grrect( xblc, yblc, xtrc, ytrc, filled=.false. )
         call pgebuf

         if( clipping_disabled ) then
            ibuf(1) = 1 ! clipping at viewport
            call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr ) ! set clipping
            CLIPPING_IN_AXES = .true.
         end if

      end if

   end subroutine mf_redraw_axis_lines
!_______________________________________________________________________
!
   subroutine mf_prepare_axes( win_id, range )

      integer,              intent(in) :: win_id
      real(kind=MF_DOUBLE), intent(in) :: range(4)
      !------ API end ------

      real(kind=MF_DOUBLE) :: x_min, x_max, y_min, y_max
      type(mf_win_info), pointer :: win
      logical :: changed_x, changed_y, must_be_redraw
      integer :: just, colbar, xlabel_pos
      real(kind=MF_DOUBLE) :: v(2)

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      if( .not. win%hold .and. .not. win%blank ) then
         call erase_mf_win( CURRENT_WIN_ID ) ! not dealloc_mf_win !
      end if

      select case( win%axis_manual_x )
         case( 0, 2 ) ! "auto" or "tight"
            x_min = range(1)
            x_max = range(2)
         case( 1 ) ! "manual"
            x_min = win%current_axes(1)
            x_max = win%current_axes(2)
      end select
      select case( win%axis_manual_y )
         case( 0, 2 ) ! "auto" or "tight"
            y_min = range(3)
            y_max = range(4)
         case( 1 ) ! "manual"
            y_min = win%current_axes(3)
            y_max = win%current_axes(4)
      end select

      if( win%axis_scale_x == 0 ) then
         win%axis_scale_x = 1
      end if
      if( win%axis_scale_y == 0 ) then
         win%axis_scale_y = 1
      end if

      must_be_redraw = .false.

      if( win%blank ) then
         ! to avoid invalid axes in MFPLOT, and also to avoid
         ! floating-point absorption.
!### TODO 1: ok pour log-log ?
         call fix_min_max_for_axes( x_min, x_max )
         call fix_min_max_for_axes( y_min, y_max )
         win%current_axes(:) = [ x_min, x_max, y_min, y_max ]
      else

         changed_x = .true.
         if( win%hold .and. (win%axis_manual_x/=1) ) then
            changed_x = .false.
            if( x_min <= x_max ) then
               if( x_min < win%current_axes(1) .or.                      &
                   x_max > win%current_axes(2)      ) then
                  changed_x = .true.
                  x_min = min( x_min, win%current_axes(1) )
                  x_max = max( x_max, win%current_axes(2) )
!### TODO 1: is this good?
!    I didn't remember why I introduced that
!!if( win%axis_scale_x /= 2 ) then
!!   x_min = x_min - 1.0d-4*(x_max-x_min)
!!   x_max = x_max + 1.0d-4*(x_max-x_min)
!!end if
               end if
            else ! axe inversé
               if( x_min > win%current_axes(2) .or.                      &
                   x_max < win%current_axes(1)      ) then
                  changed_x = .true.
                  x_min = max( x_min, win%current_axes(2) )
                  x_max = min( x_max, win%current_axes(1) )
               end if
            end if

            if( changed_x ) then
               win%current_axes(1:2) = [ x_min, x_max ]
               must_be_redraw = .true.
            end if
         end if

         changed_y = .true.
         if( win%hold .and. (win%axis_manual_y/=1) ) then
            changed_y = .false.
            if( y_min <= y_max ) then
               if( y_min < win%current_axes(3) .or.                      &
                   y_max > win%current_axes(4)      ) then
                  changed_y = .true.
                  y_min = min( y_min, win%current_axes(3) )
                  y_max = max( y_max, win%current_axes(4) )
!### TODO 1: is this good?
!    I didn't remember why I introduced that
!!if( win%axis_scale_y /= 2 ) then
!!   y_min = y_min - 1.0d-4*(y_max-y_min)
!!   y_max = y_max + 1.0d-4*(y_max-y_min)
!!end if
               end if
            else ! axe inversé
               if( y_min > win%current_axes(4) .or.                      &
                   y_max < win%current_axes(3)      ) then
                  changed_y = .true.
                  y_min = max( y_min, win%current_axes(4) )
                  y_max = min( y_max, win%current_axes(3) )
               end if
            end if

            if( changed_y ) then
               win%current_axes(3:4) = [ y_min, y_max ]
               must_be_redraw = .true.
            end if
         end if

      end if

      if( win%axis_manual_x == 0 ) then
         ! mode "auto"
         v = smart_max( win%current_axes(1:2), "x" )
         x_min = v(1)
         x_max = v(2)
         win%current_axes(1:2) = [ x_min, x_max ]
      end if
      if( win%axis_manual_y == 0 ) then
         ! mode "auto"
         v = smart_max( win%current_axes(3:4), "y" )
         y_min = v(1)
         y_max = v(2)
         win%current_axes(3:4) = [ y_min, y_max ]
      end if

      if( must_be_redraw ) then
         call mf_win_redraw( CURRENT_WIN_ID )
      end if

      if( .not. win%hold .or. win%blank ) then

         if( win%axis_scale_x == 2 ) then
            x_min = log10( x_min )
            x_max = log10( x_max )
         end if
         if( win%axis_scale_y == 2 ) then
            y_min = log10( y_min )
            y_max = log10( y_max )
         end if

         call pgsch(1.0d0) ! reset to unity to have std labelling margins

         if( win%colorbar == 0 ) then
            colbar = 0
         else
            if( win%colorbar_pos == "ri" ) then
               colbar = 1
            else ! colorbar_pos = "bi"
               colbar = 2
            end if
         end if
         if( win%axis_equal ) then
            ! scaled axis
            just = 1
         else
            ! automatic axis
            just = 0
         end if
         if( win%axis_mode_xy ) then
            xlabel_pos = 1
         else
            xlabel_pos = 2
         end if
         call pgenv( x_min, x_max, y_min, y_max, just, colbar, xlabel_pos )

         call char_height_factor_update( win_id )

         call mf_win_draw_box( win_id )

      end if

      ! here, we force the box to be drawn if Plot() is called after Text()
      ! [Nota: the usual order is Plot() first, then Text()...]
      if( win%axis_on .and. .not. win%axis_drawn ) then

         call mf_win_draw_box( win_id )

      end if

   end subroutine mf_prepare_axes
!_______________________________________________________________________
!
   subroutine char_height_factor_update( win_id )

      integer, intent(in) :: win_id
      !------ API end ------

      ! MFPLOT initialisation (by PGENV) must be done before calling
      ! the current routine.
      !
      ! NULL device: 96 dpi must be the resolution used in 'null_driver'.

      type(mf_win_info), pointer :: win
      real(kind=MF_DOUBLE) :: x1, x2, y1, y2
      real(kind=MF_DOUBLE) :: ref_length_pixels, ref_length_inches

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      ! on récupère la taille de la 'view surface' en inches
      if( X11_DEVICE ) then
         call pgqvsz( 1, x1, x2, y1, y2 ) ! 1: inches
      else if( NULL_DEVICE ) then
         ! from msFigure()
         x2 = win%geometry(3) ! pixels
         y2 = win%geometry(4) ! pixels
         ! assuming a screen resolution about 96 dpi...
         x2 = x2/96.0d0       ! inches
         y2 = y2/96.0d0       ! inches
         ! substract margins (1/4 inch)
         x2 = x2 - 2*0.25d0
         y2 = y2 - 2*0.25d0
      else
         write(STDERR,*) '(MUESLI:) bug in char_height_factor_update()'
         write(STDERR,*) '          both X11_DEVICE and NULL_DEVICE are FALSE!'
         stop
      end if

      ! in MFPLOT, unit is 1/40th the height of the view surface
      if( x2 > y2 ) then ! landscape
         ref_length_inches = y2/40.0d0
      else ! portrait
         ref_length_inches = x2/40.0d0
      end if

      if( win%char_height_pixel ) then

         ! on récupère la taille de la 'view surface' en pixels.
         if( X11_DEVICE ) then
            call pgqvsz( 3, x1, x2, y1, y2 ) ! 3: pixels
         else if( NULL_DEVICE ) then
            ! from msFigure()
            x2 = win%geometry(3) ! pixels
            y2 = win%geometry(4) ! pixels
            x2 = x2 - 2*0.25d0*96.0d0
            y2 = y2 - 2*0.25d0*96.0d0
         end if

         ! in MFPLOT, unit is 1/40 the height of the view surface
         if( x2 > y2 ) then ! landscape
            ref_length_pixels = y2/40.0d0
         else ! portrait
            ref_length_pixels = x2/40.0d0
         end if

         ! on veut : height = 1 => char_height_in_pix = 12
         win%char_height_factor = 12.0d0/ref_length_pixels

      else

         win%char_height_factor = 1.0d0

      end if

      pg_char_height_factor(pgid) = win%char_height_factor

      win%char_width_factor = win%char_height_factor                    &
                            * DEFAULT_FONT_WIDTH * ref_length_inches

   end subroutine char_height_factor_update
!_______________________________________________________________________
!
   function get_true_min_max( win_id ) result( range )

      integer, intent(in) :: win_id
      real(kind=MF_DOUBLE) :: range(4)
      !------ API end ------

      ! called by 'Axis', 'Zoom' and consort...
      !
      ! (when called by 'Zoom', it cannot come from 'Spy';
      !  so, the 'cmd' variable below must never be equal to
      !  'pcolor_spy' or 'pcolor_spy_sparse')

      ! Warning: this routine may return [0,0,0,0]; in such a case,
      !          the calling unit must take an appropriate decision.

      type(mf_win_info), pointer :: win
      type(grobj_elem), pointer :: grobj

      integer :: j, ni, nj
      real(kind=MF_DOUBLE) :: tr(6), xx(4), yy(4), x, y, d, h_min, h_max, xp, yp
      real(kind=MF_DOUBLE) :: angle, hjust, vjust, cosa, sina, bbox(4), xbox(4), ybox(4)
      real(kind=MF_DOUBLE) :: relative_height
      character :: c

      integer :: len_text
      character(len=1024) :: text

      character(len=20) :: cmd

   !------ end of declarations -- execution starts hereafter  ------

      win => mf_win_db(win_id)

      ! In the following, don't use the 'msMuesliTrace' routine when
      ! encountering warnings: this will badly interact with X11 events...

      grobj => win%grobj_head

      if( win%empty .or. .not. associated(grobj) ) then
         range(:) = 0.0d0
         return
      else
         range(1) = +huge(1.0)
         range(2) = -huge(1.0)
         range(3) = +huge(1.0)
         range(4) = -huge(1.0)
      end if

      do
         if( .not. associated(grobj) ) then
            exit
         end if

         cmd = grobj%struct%cmd

         select case( cmd )
         case( "tri_fill", "histogram", "tri_pcolor", "tri_contour",    &
               "bar_groups_stacks", "quiver", "streamline" )
            range(1) = min( range(1), grobj%struct%range(1) )
            range(2) = max( range(2), grobj%struct%range(2) )
            range(3) = min( range(3), grobj%struct%range(3) )
            range(4) = max( range(4), grobj%struct%range(4) )
         case( "contour", "contour_filled", "pcolor", "pcolor_proj",    &
               "pcolor_mat_proj" )
            ! assuming that the grobj is alone (y-axis may be inverted)
            range(:) = grobj%struct%range
         case( "image" )
            ! assuming that the grobj is alone (y-axis may be inverted)
            range(1) = 1
            range(2) = size(grobj%struct%val_mat,2)
            range(3) = size(grobj%struct%val_mat,1)
            range(4) = 1
         case( "pcolor_spy" )
            ! assuming that the grobj is alone (y-axis may be inverted)
            range(1) = 0.45d0
            range(2) = grobj%struct%npt2 + 0.55d0
            range(3) = grobj%struct%npt  + 0.55d0
            range(4) = 0.45d0
         case( "pcolor_spy_sparse" )
            ! assuming that the grobj is alone (y-axis may be inverted)
            range(1) = 0.45d0
            range(2) = grobj%struct%npt2 + 0.55d0
            range(3) = grobj%struct%npt  + 0.55d0
            range(4) = 0.45d0
         case( "arrow" )
            xx(1:4) = [ grobj%struct%x_text, grobj%struct%y_text,       &
                        grobj%struct%ang_text, grobj%struct%just_text ]
            range(1) = min( range(1), minval(xx) )
            range(2) = max( range(2), maxval(xx) )
            range(3) = min( range(3), minval(xx) )
            range(4) = max( range(4), maxval(xx) )
         case( "text" )
            if( associated(grobj%struct%text) ) then
               len_text = size(grobj%struct%text)
            else
               len_text = 0
            end if
            text = ""
            do j = 1, len_text
               c = grobj%struct%text(j)
               text(j:j) = c
            end do
            if( len_text > 0 ) then
               relative_height = grobj%struct%height_text
               call pgsch( relative_height * win%char_height_factor ) ! character height
               call grparse_ps_font( text(1:len_text), d, h_min, h_max, &
                                     draw=.false. )
               x = grobj%struct%x_text
               y = grobj%struct%y_text
               ! Convert to device coordinates
               xp = pgxorg(pgid) + x*pgxscl(pgid)
               yp = pgyorg(pgid) + y*pgyscl(pgid)
               angle = grobj%struct%ang_text
               hjust = grobj%struct%just_text
               if( hjust /= 0.0d0 ) then
                  if( angle == 0.0d0 ) then
                     xp = xp - d*hjust
                  else
                     xp = xp - d*hjust*cos(angle/rad_to_deg)
                     yp = yp - d*hjust*sin(angle/rad_to_deg)
                  end if
               end if
               vjust = grobj%struct%just_vert_text
               if( vjust /= 0.0d0 ) then
                  write(STDERR,*) "(MUESLI get_true_min_max:) warning:"
                  write(STDERR,*) "        in 'text' processing: vertical justification is not yet implemented!"
               end if
               bbox(1) = 0.0d0
               bbox(2) = h_min
               bbox(3) = d
               bbox(4) = h_max
               if( angle /= 0.0d0 ) then
                  ! Rotate bounding box.
                  cosa = cos(angle/rad_to_deg)
                  sina = sin(angle/rad_to_deg)
                  xbox(1) = xp + (cosa*bbox(1) - sina*bbox(2))
                  ybox(1) = yp + (sina*bbox(1) + cosa*bbox(2))
                  xbox(2) = xp + (cosa*bbox(1) - sina*bbox(4))
                  ybox(2) = yp + (sina*bbox(1) + cosa*bbox(4))
                  xbox(3) = xp + (cosa*bbox(3) - sina*bbox(4))
                  ybox(3) = yp + (sina*bbox(3) + cosa*bbox(4))
                  xbox(4) = xp + (cosa*bbox(3) - sina*bbox(2))
                  ybox(4) = yp + (sina*bbox(3) + cosa*bbox(2))
               else ! angle == 0
                  xbox(1) = xp + bbox(1)
                  ybox(1) = yp + bbox(2)
                  xbox(2) = xp + bbox(1)
                  ybox(2) = yp + bbox(4)
                  xbox(3) = xp + bbox(3)
                  ybox(3) = yp + bbox(4)
                  xbox(4) = xp + bbox(3)
                  ybox(4) = yp + bbox(2)
               end if
               bbox(1) = minval(xbox)
               bbox(2) = maxval(xbox)
               bbox(3) = minval(ybox)
               bbox(4) = maxval(ybox)
               ! Convert bounding box to world coordinates.
               bbox(1) = (bbox(1)-pgxorg(pgid))/pgxscl(pgid)
               bbox(2) = (bbox(2)-pgxorg(pgid))/pgxscl(pgid)
               bbox(3) = (bbox(3)-pgyorg(pgid))/pgyscl(pgid)
               bbox(4) = (bbox(4)-pgyorg(pgid))/pgyscl(pgid)
               range(1) = min( range(1), bbox(1) )
               range(2) = max( range(2), bbox(2) )
               range(3) = min( range(3), bbox(3) )
               range(4) = max( range(4), bbox(4) )
            end if
         case( "plot_spy_sparse" )
            ! when y-axis is inverted, assuming that the grobj is alone...
            range(1) = 0.45d0
            range(2) = grobj%struct%npt2 + 0.55d0
            range(3) = grobj%struct%npt  + 0.55d0
            range(4) = 0.45d0
         case( "bar",                                                   &
               "errorbar_x_line", "errorbar_y_line", "errorbar_xy_line", &
               "errorbar_x_line+pt", "errorbar_y_line+pt", "errorbar_xy_line+pt", &
               "errorbar_x_pt", "errorbar_y_pt", "errorbar_xy_pt",      &
               "patch", "line", "line+point", "point",                  &
               "cubic_bezier", "polygon" )
            ! All these cases stores coordinates in {abs_tab,ord_tab} vectors
            range(1) = min( range(1), minval(grobj%struct%abs_tab) )
            range(2) = max( range(2), maxval(grobj%struct%abs_tab) )
            range(3) = min( range(3), minval(grobj%struct%ord_tab) )
            range(4) = max( range(4), maxval(grobj%struct%ord_tab) )
         case default
            print *, "(MUESLI get_true_min_max:) internal error:"
            print *, "        cmd not implemented: ", trim(cmd)
            pause "for debugging purpose only..."
            stop
         end select

         grobj => grobj%next

      end do

   end function get_true_min_max
!_______________________________________________________________________
!
   subroutine fix_min_max_for_axes( x1, x2, warning )

      real(kind=MF_DOUBLE), intent(in out)        :: x1, x2
      logical,              intent(out), optional :: warning
      !------ API end ------

      ! Avoid invalid axes in MFPLOT, and also to avoid floating-point
      ! absorption.
      ! (do not choose a more tiny interval: 500 seems to be a
      !  reasonable choice)
      !
      ! Avoid also NaN values

      real(kind=MF_DOUBLE), parameter :: factor = 500.0d0
      real(kind=MF_DOUBLE) :: x_mid, max_all_abs, s, threshold
      logical :: warn

   !------ end of declarations -- execution starts hereafter  ------

      if( isnan(x1) ) then
         if( isnan(x2) ) then
            ! both values are NaN
            x1 = 0.0d0
            x2 = 1.0d0
         else
            ! only x1 is NaN
            x1 = x2 - 1.0d0
         end if
      else
         if( isnan(x2) ) then
            ! only x2 is NaN
            x2 = x1 + 1.0d0
         end if
      end if

      max_all_abs = max( abs(x1), abs(x2) )
      if( max_all_abs == 0.0d0 ) then
         x1 = -1.0d-150
         x2 =  1.0d-150
         warn = .true.
      else
         ! Though double precision is used in FGL, MF_EPS (i.e. 1e-15)
         ! is too stringent for labelling. Using 1e-7 instead
         threshold = factor*1.0d-7*max_all_abs
         if( abs(x1-x2) < threshold ) then
            x_mid = (x1+x2)/2.0d0
            s = sign(1.0d0,x2-x1)
            x1 = x_mid - s*0.5d0*threshold
            x2 = x_mid + s*0.5d0*threshold
            warn = .true.
         else
            warn = .false.
         end if
      end if
      if( present(warning) ) then
         warning = warn
      end if

   end subroutine fix_min_max_for_axes
!_______________________________________________________________________
!
   subroutine count_PDF_objects( win_id, nb_bitmaps, nb_tri_grad,       &
                                 nb_quad_grad, cmap_used, markers_used )

      integer, intent(in) :: win_id
      integer, intent(out) :: nb_bitmaps, nb_tri_grad, nb_quad_grad
      logical, intent(out) :: cmap_used, markers_used(26)
      !------ API end ------

      ! called only for the PDF driver.
      !
      ! - count bitmap images;
      ! - determine if the current colormap must be inserted;
      ! - determine if fonts other than the default one (/fn = Helvetica)
      !   are used via labels (XLabel, YLabel, Title) or via Text;
      ! - determine which markers (among the 26) are effectively used.

      type(mf_win_info), pointer :: win
      type(grobj_elem), pointer :: grobj, grobj_next
      character(len=20) :: cmd
      integer :: marker, ni, nj

   !------ end of declarations -- execution starts hereafter  ------

      nb_bitmaps = 0
      nb_tri_grad = 0
      nb_quad_grad = 0
      cmap_used = .false.
      markers_used(:) = .false.

      win => mf_win_db(win_id)

      ! here, we have to travel the entire linked list of grobj to count
      ! the number of some types of objects.
      grobj => win%grobj_head
      do
         if( .not. associated(grobj) ) then
            exit
         end if
         cmd = grobj%struct%cmd
         select case( cmd )
            case( "image" )
               nb_bitmaps = nb_bitmaps + 1
            case( "patch" )
               if( will_be_drawn_in_pdf( grobj ) ) then
                  if( grobj%struct%npt == 3 ) then
                     if( win%shading == "interp" ) then
                        nb_tri_grad = nb_tri_grad + 1
                        cmap_used = .true.
                     end if
                  else if( grobj%struct%npt == 4 ) then
                     if( win%shading == "interp" ) then
                        nb_quad_grad = nb_quad_grad + 1
                        cmap_used = .true.
                     end if
                  else ! npt > 4
                     if( win%shading == "interp" ) then
                        nb_tri_grad = nb_tri_grad + (grobj%struct%npt-2)
                        cmap_used = .true.
                     end if
                  end if
               end if
            case( "pcolor" )
               if( win%shading == "interp" ) then
                  nb_quad_grad = nb_quad_grad + 1
                  cmap_used = .true.
               end if
            case( "pcolor_proj" )
               if( win%shading == "interp" ) then
                  if( grobj%struct%marker == 1 ) then ! "XZ"
                     ni = size(grobj%struct%val_mat,1)
                     nb_tri_grad = nb_tri_grad + (ni-1)
                  else if( grobj%struct%marker == 2 ) then ! "YZ"
                     nj = size(grobj%struct%val_mat,2)
                     nb_tri_grad = nb_tri_grad + (nj-1)
                  end if
                  cmap_used = .true.
               end if
            case( "tri_pcolor", "pcolor_mat_proj" )
               if( win%shading == "interp" ) then
                  nb_tri_grad = nb_tri_grad + 1
                  cmap_used = .true.
               end if
            case( "point", "line+point", "plot_spy_sparse",             &
                  "errorbar_x_pt", "errorbar_y_pt", "errorbar_xy_pt",   &
                  "errorbar_x_line+pt", "errorbar_y_line+pt",           &
                  "errorbar_xy_line+pt", "voronoi" )
               marker = abs(grobj%struct%marker)
               markers_used(marker) = .true.
            case( "PLdomain" )
!### TODO:
!### Hou là... je les ai codé en dur... en attendant de faire mieux !
               markers_used(14) = .true.
               markers_used(23) = .true.
         end select
         grobj => grobj%next
      end do

      if( win%colorbar == 1 .or. win%colorbar == 2 ) then
         nb_bitmaps = nb_bitmaps + 1
      end if

!!print "(A)", " *** End of 'count_PDF_objects' ***"
!!print "(2(A,I0),A,L)", "   nb_tri_grad = ", nb_tri_grad,       &
!!                       "  nb_quad_grad = ", nb_quad_grad,      &
!!                       "  cmap_used = ", cmap_used

   end subroutine count_PDF_objects
!_______________________________________________________________________
!
   function encode_handle( win_id, hdle ) result( handle )

      integer, intent(in) :: win_id, hdle
      integer             :: handle
      !------ API end ------

   !------ end of declarations -- execution starts hereafter  ------

      ! multiplying by 64 give a shift (on left) of 6 bits
      handle = win_id + hdle*64

      ! circular shift about 20 positions (in order to give a large number)
      handle = ishftc( handle, 20 )

   end function encode_handle
!_______________________________________________________________________
!
   subroutine decode_handle( handle, win_id, hdle )

      integer, intent(in)  :: handle
      integer, intent(out) :: win_id, hdle
      !------ API end ------

      integer :: handle_0

   !------ end of declarations -- execution starts hereafter  ------

      handle_0 = ishftc( handle, -20 )
      hdle = handle_0/64
      win_id = handle_0 - hdle*64

   end subroutine decode_handle
!_______________________________________________________________________
!
   function smart_max( v, axis ) result( out )

      real(kind=MF_DOUBLE), intent(in) :: v(2)
      character,            intent(in) :: axis
      real(kind=MF_DOUBLE)             :: out(2)
      !------ API end ------

      ! Returns a smart range for more aesthetic axes in FGL graphics

      ! v is a vector containing the two extremal values; it doesn't need
      ! to be ordered.

      ! A check is done to see if extremal values are very close to a ticks.
      ! See explanation in the 'tiny_shift_range' routine of pgbox1.f90.
      ! The most important is that the single precision epsilon machine must
      ! be used, and not the double precision value.

      ! The current routine should not be called many times with updated
      ! values for the axis ranges: in such a case, intervals may become
      ! larger and larger (no convergence). This is why the interval width
      ! INTV and the number of subticks NSUB are saved in global variables,
      ! and passed to PGBOX (or PGTBOX).

      real(kind=MF_DOUBLE) :: a, b, t, intv, a_t, b_t
      real(kind=MF_DOUBLE) :: log_a, log_b, xval, yval, prev
      logical :: inverted, found, do_remains
      integer :: i, j, nsub, i1, i2
      type(mf_win_info), pointer :: win

      ! Table of logarithms 1..10 (log10)
      double precision :: tab(10), utab(10)
      DATA TAB / 0.000000000000000d0, 0.301029995663981d0,              &
                 0.477121254719662d0, 0.602059991327962d0,              &
                 0.698970004336019d0, 0.778151250383644d0,              &
                 0.845098040014257d0, 0.903089986991944d0,              &
                 0.954242509439325d0, 1.000000000000000d0 /

   !------ end of declarations -- execution starts hereafter  ------

      if( v(1) < v(2) ) then
         inverted = .false.
         a = v(1)
         b = v(2)
      else if( v(1) > v(2) ) then
         inverted = .true.
         a = v(2)
         b = v(1)
      else ! v(1) == v(2)
         out = v
         return
      end if

      win => mf_win_db(CURRENT_WIN_ID)

      if( pgxlen(pgid) == 0.0d0 ) then
         ! This value cannot be zero (else a NaN value will be found
         ! in the 'intv' computation -- see below).
         ! Set to an approximate value, because we have to avoid a
         ! circular definition (PGENV vs SMART_MAX): choose the width
         ! of the current window
         pgxlen(pgid) = grxmxa(grcide)
         call pgsch( win%axis_font_size * 1.0d0 ) ! character height
      end if

      if( pgylen(pgid) == 0.0d0 ) then
         ! This value cannot be zero (else a NaN value will be found
         ! in the 'intv' computation -- see below).
         ! Set to an approximate value, because we have to avoid a
         ! circular definition (PGENV vs SMART_MAX): choose the width
         ! of the current window
         pgylen(pgid) = grymxa(grcide)
      end if

      if( win%axis_scale_x == 0 ) win%axis_scale_x = 1
      if( win%axis_scale_y == 0 ) win%axis_scale_y = 1

      do_remains = .false.

      ! Get smallest tick length as in PGBOX
      if( axis == "x" ) then
         if( win%axis_scale_x == 1 ) then
            ! linear scaling
            intv = max(0.05d0, min(7.0d0*pgxsp(pgid)/pgxlen(pgid), 0.2d0))*(b-a)
            intv = pgrnd(intv,nsub)
            t = intv/nsub
            MF_SMART_XTICK(pgid) = intv
            MF_SMART_NXSUB(pgid) = nsub
            do_remains = .true.
         else
            ! logarithmic scaling
            ! round 'a' if its log is very close to an integer value
            log_a = log10(a)
            i = nint(log_a)
            if( abs( dble(i) - log_a ) <= 10.0d0*epsilon(1.0)*abs(log_a) ) then
               a = 10.0d0**i
               log_a = i
            end if
            ! round 'b' if its log is very close to an integer value
            log_b = log10(b)
            i = nint(log_b)
            if( abs( dble(i) - log_b ) <= 10.0d0*epsilon(1.0)*abs(log_b) ) then
               b = 10.0d0**i
               log_b = i
            end if
            call pgbox1( log_a, log_b, 1.0d0, i1, i2)

            ! searck for an outside tick nearest the left value
            prev = -MF_INF
            found = .false.
            loop1: do i = i1-1, i2+1
               do j = 1, 9
                  xval = i + tab(j)
                  if( 10.0d0**xval > a ) then
                     found = .true.
                     a = 10.0d0**prev
                     exit loop1
                  end if
                  prev = xval
               end do
            end do loop1
            if( .not. found ) then
               print *, "smart_max: (LOG X) internal error: xval not found in loop1"
               pause "for debugging purpose only"
               stop
            end if

            ! searck for an outside tick nearest the right value
            prev = +MF_INF
            found = .false.
            loop2: do i = i2+1, i1-1, -1
               do j = 9, 1, -1
                  xval = i + tab(j)
                  if( 10.0d0**xval < b ) then
                     found = .true.
                     b = 10.0d0**prev
                     exit loop2
                  end if
                  prev = xval
               end do
            end do loop2
            if( .not. found ) then
               print *, "smart_max: (LOG X) internal error: xval not found in loop2"
               pause "for debugging purpose only"
               stop
            end if

            out = [ a, b ]

         end if
      else ! axis == "y"
         if( win%axis_scale_y == 1 ) then
            ! linear scaling
            intv = max(0.05d0, min(7.0d0*pgxsp(pgid)/pgylen(pgid), 0.2d0))*(b-a)
            intv = pgrnd(intv,nsub)
            t = intv/nsub
            MF_SMART_YTICK(pgid) = intv
            MF_SMART_NYSUB(pgid) = nsub
            do_remains = .true.
         else
            ! logarithmic scaling
            ! round 'a' if its log is very close to an integer value
            log_a = log10(a)
            i = nint(log_a)
            if( abs( dble(i) - log_a ) <= 10.0d0*epsilon(1.0)*abs(log_a) ) then
               a = 10.0d0**i
               log_a = i
            end if
            ! round 'b' if its log is very close to an integer value
            log_b = log10(b)
            i = nint(log_b)
            if( abs( dble(i) - log_b ) <= 10.0d0*epsilon(1.0)*abs(log_b) ) then
               b = 10.0d0**i
               log_b = i
            end if
            call pgbox1( log_a, log_b, 1.0d0, i1, i2)

            ! searck for an outside tick nearest the left value
            prev = -MF_INF
            found = .false.
            loop3: do i = i1-1, i2+1
               do j = 1, 9
                  yval = i + tab(j)
                  if( 10.0d0**yval > a ) then
                     found = .true.
                     a = 10.0d0**prev
                     exit loop3
                  end if
                  prev = yval
               end do
            end do loop3
            if( .not. found ) then
               print *, "smart_max: (LOG Y) internal error: yval not found in loop3"
               pause "for debugging purpose only"
               stop
            end if

            ! searck for an outside tick nearest the right value
            prev = +MF_INF
            found = .false.
            loop4: do i = i2+1, i1-1, -1
               do j = 9, 1, -1
                  yval = i + tab(j)
                  if( 10.0d0**yval < b ) then
                     found = .true.
                     b = 10.0d0**prev
                     exit loop4
                  end if
                  prev = yval
               end do
            end do loop4
            if( .not. found ) then
               print *, "smart_max: (LOG Y) internal error: yval not found in loop4"
               pause "for debugging purpose only"
               stop
            end if

            out = [ a, b ]

         end if
      end if

      if( do_remains ) then

         a_t = a/t
         i = nint(a_t)
         if( abs( dble(i) - a_t ) <= 1.5d0*epsilon(1.0)*abs(a_t) ) then
            out(1) = i*t
         else
            out(1) = floor(a_t)*t
         end if

         b_t = b/t
         i = nint(b_t)
         if( abs( dble(i) - b_t ) <= 1.5d0*epsilon(1.0)*abs(b_t) ) then
            out(2) = i*t
         else
            out(2) = ceiling(b_t)*t
         end if

      end if

      if( inverted ) then
         t = out(1)
         out(1) = out(2)
         out(2) = t
      end if

   end function smart_max
!_______________________________________________________________________
!
   subroutine unset_smart_ticks_x()
      !------ API end ------

   !------ end of declarations -- execution starts hereafter  ------

      MF_SMART_XTICK(pgid) = 0.0d0
      MF_SMART_NXSUB(pgid) = 0

   end subroutine unset_smart_ticks_x
!_______________________________________________________________________
!
   subroutine unset_smart_ticks_y()
      !------ API end ------

   !------ end of declarations -- execution starts hereafter  ------

      MF_SMART_YTICK(pgid) = 0.0d0
      MF_SMART_NYSUB(pgid) = 0

   end subroutine unset_smart_ticks_y
