! auxiliary routines used by FGL but not linked to graphics

module fgl_aux

   use fml ! MUESLI Numerical Library

   use mod_mfarray, only: PrintMessage ! private when using FML

#ifndef _OPTIM
   use mod_fgl_mem_debug
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

   private :: PrintMessage

contains
!_______________________________________________________________________
!
   subroutine msPointer_real_only( A, f90_ptr, status_alloc )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), pointer :: f90_ptr(:,:)
      integer, intent(out) :: status_alloc
      !------ API end ------

      ! specific version to FGL
      ! 'msFreePointer' must be used, sometimes, to finalize
      ! (cf. status_alloc)
      !
      ! Allow the pointing to a real array, even if the original mfArray
      ! contains complex values.
      !
      ! The status_alloc argument allows the calling routine to know if
      ! data has been copied (true allocation) and therefore to deallocate
      ! f90_ptr (status_alloc=1), or if data has been pointing only
      ! (status_alloc=0).
      ! Moreover, the status_alloc flag is used to know if we have to
      ! call 'msFreePointer' in the calling program.

      ! used only to point to data
      real(kind=MF_DOUBLE), pointer :: r_ptr(:,:) => null()
      complex(kind=MF_DOUBLE), pointer :: c_ptr(:,:) => null()

      ! used to obtain a non-temporary copy of A, in case where this last
      ! mfArray is temporary. (it's required for 'msPointer')
      type(mfArray) :: A_copy

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      ! no check : this should be done in the calling FGL routine.

      if( A%status_temporary ) then

         A_copy = A
         allocate( f90_ptr( size(A,1), size(A,2) ) )

         if( mfIsReal(A) ) then
            ! this is a copy: no need to keep data CRC
            call msPointer( A_copy, r_ptr, no_crc=.true. )
            f90_ptr(:,:) = r_ptr(:,:)
            call msFreePointer( A_copy, r_ptr ) ! r_ptr => null()
         else if( mfIsComplex(A) ) then
            call msPointer( A_copy, c_ptr, no_crc=.true. )
            f90_ptr(:,:) = c_ptr(:,:)
            call msFreePointer( A_copy, c_ptr ) ! c_ptr => null()
         end if
         call msSilentRelease( A_copy )
         status_alloc = 1

      else

         if( mfIsReal(A) ) then
            ! in FGL, data are read-only: no need to keep data CRC
            call msPointer( A, f90_ptr, no_crc=.true., intern_call=.true. )
            status_alloc = 0
         else if( mfIsComplex(A) ) then
            ! in FGL, data are read-only: no need to keep data CRC
            call msPointer( A, c_ptr, no_crc=.true. )
            allocate( f90_ptr( size(A,1), size(A,2) ) )

            f90_ptr(:,:) = c_ptr(:,:)
            call msFreePointer( A, c_ptr ) ! c_ptr => null()
            status_alloc = 1
         end if

      end if

   end subroutine msPointer_real_only
!_______________________________________________________________________
!
   subroutine msPointer_imag_only( A, f90_ptr )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), pointer :: f90_ptr(:,:)
      !------ API end ------

      ! specific version to FGL
      ! 'msFreePointer' must not be used to finalize
      !
      ! Allow the pointing to a real array, even if the original mfArray
      ! contains complex values.
      !
      ! The calling program must deallocate f90_ptr, because there is
      ! always an allocation for f90_ptr.

      ! used only to point to data
      complex(kind=MF_DOUBLE), pointer :: c_ptr(:,:) => null()

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      ! no check : this should be done in the calling FGL routine.

      ! particular case : A may be 'tempo' because c_ptr will be freed
      ! before A is deallocated.
      ! in FGL, data are read-only: no need to keep data CRC
      call msPointer( A, c_ptr, no_crc=.true. )
      allocate( f90_ptr( size(a,1), size(a,2) ) )

      f90_ptr(:,:) = aimag( c_ptr(:,:) )
      call msFreePointer( A, c_ptr ) ! c_ptr => null()

   end subroutine msPointer_imag_only
!_______________________________________________________________________
!
   subroutine extract_vector_from_mfArray( a, v, status, string )

      type(mfArray), intent(in) :: a
      double precision, intent(out) :: v(:)
      integer, intent(out) :: status
      character(len=*), intent(out) :: string
      !------ API end ------

      ! Extraction from 'a' of a vector v(:) of specified size
      ! (v is already allocated)

      ! pointer for manipulating mfArray out of fml module
      real(kind=MF_DOUBLE), pointer :: r_ptr(:,:)
      complex(kind=MF_DOUBLE), pointer :: c_ptr(:,:)

      type(mfArray) :: A_copy
      integer :: mf_message_level_save

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      status = 0

      if( mfIsReal(a) ) then

         if( a%status_temporary ) then
            mf_message_level_save = mf_message_level
            mf_message_level = 0
            A_copy = a
            mf_message_level = mf_message_level_save
            call msPointer( A_copy, r_ptr, no_crc=.true. )
         else
            call msPointer( a, r_ptr, no_crc=.true. )
         end if

         ! checking that 'a' is allocated
         if( .not. associated( r_ptr ) ) then
            string = " arg not initialized!"
            status = 1
            return
         end if

         ! checking that 'a' is a 1-D array
         if( size(r_ptr,1)/=1 .and. size(r_ptr,2)/=1 ) then
            string = " arg must be a 1-D array!"
            status = 2
            return
         end if

         n = size(v)

         if( size(r_ptr,1)==1 ) then
            if( size(r_ptr,2) >= n ) then
               v(:) = r_ptr(1,1:n)
            else
               string = " arg with size too small!"
               status = 3
               return
            end if
         else
            if( size(r_ptr,1) >= n ) then
               v(:) = r_ptr(1:n,1)
            else
               string = " arg with size too small!"
               status = 3
               return
            end if
         end if

         if( a%status_temporary ) then
            call msFreePointer( A_copy, r_ptr )
            call msSilentRelease( A_copy )
         else
            call msFreePointer( a, r_ptr )
         end if

      else if( mfIsComplex(a) ) then

         string = "complex"
         status = -1

         if( a%status_temporary ) then
            A_copy = a
            call msPointer( A_copy, c_ptr, no_crc=.true. )
         else
            call msPointer( a, c_ptr, no_crc=.true. )
         end if

         ! checking that 'a' is allocated
         if( .not. associated( c_ptr ) ) then
            string = " arg not initialized!"
            status = 1
            return
         end if

         ! checking that 'a' is a 1-D array
         if( size(c_ptr,1)/=1 .and. size(c_ptr,2)/=1 ) then
            string = " arg must be a 1-D array!"
            status = 2
            return
         end if

         n = size(v)

         if( size(c_ptr,1)==1 ) then
            if( size(c_ptr,2) >= n ) then
               v(:) = real( c_ptr(1,1:n), kind=MF_DOUBLE )
            else
               string = " arg with size too small!"
               status = 3
               return
            end if
         else
            if( size(c_ptr,1) >= n ) then
               v(:) = real( c_ptr(1:n,1), kind=MF_DOUBLE )
            else
               string = " arg with size too small!"
               status = 3
               return
            end if
         end if

         if( a%status_temporary ) then
            call msFreePointer( A_copy, c_ptr )
            call msSilentRelease( A_copy )
         else
            call msFreePointer( a, c_ptr )
         end if

      else if( mfIsLogical(a) ) then

         ! up to now, the current routine is called only be :
         !   - Axis
         !   - ColorAxis
         !   - Figure
         !   - Text
         !
         ! so, booleans are not accepted!
         string = " arg data type is boolean!"
         status = 4

      end if

      call msFreeArgs( a )

   end subroutine extract_vector_from_mfArray
!_______________________________________________________________________
!
   function quad_is_rect( x_pg, y_pg ) result( bool )

      real(kind=MF_DOUBLE), intent(in) :: x_pg(:), y_pg(:)
      logical :: bool
      !------ API end ------

   !------ end of declarations -- execution starts hereafter  ------

      if( size(x_pg) /= 4 ) then
         bool = .false.
         return
      end if

      ! in order to plot with efficiency, not only the quadrilateral must
      ! be rectangular, but also its sides must be parallel to the axis.

      if( y_pg(1)-y_pg(2) == 0 .and.  x_pg(2)-x_pg(3) == 0 .and.        &
          y_pg(3)-y_pg(4) == 0 .and.  x_pg(4)-x_pg(1) == 0 ) then
         bool = .true.
      else if( x_pg(1)-x_pg(2) == 0 .and.  y_pg(2)-y_pg(3) == 0 .and.   &
               x_pg(3)-x_pg(4) == 0 .and.  y_pg(4)-y_pg(1) == 0 ) then
         bool = .true.
      else
         bool = .false.
      end if

   end function quad_is_rect
!_______________________________________________________________________
!
   function quad_is_tri( x_pg, y_pg, c_pg ) result( bool )

      real(kind=MF_DOUBLE), intent(in out) :: x_pg(:), y_pg(:)
      integer,              intent(in out) :: c_pg(:)
      logical :: bool
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      if( size(x_pg) /= 4 ) then
         write(STDERR,*) "(MUESLI quad_is_tri:) internal error: input polygon"
         write(STDERR,*) "                      must have 4 vertices!"
         write(STDERR,*) "                      Please report this bug to: Edouard.Canot@univ-rennes.fr"
         call msMuesliTrace( pause ="no" )
         stop
      end if

      ! checks if the quadrangle degenerates in a triangle;
      ! this implies 6 tests

      ! In case where the check is true, the triangle is returned in the
      ! first three elements of 'x' and 'y'.

      ! If the degeneration is greater (i.e. n=2 or n=1) an error is returned

      n = 4

      if( x_pg(1) == x_pg(2) .and.  y_pg(1) == y_pg(2) ) then
         n = 3
         x_pg(2) = x_pg(3)
         y_pg(2) = y_pg(3)
         c_pg(2) = c_pg(3)
         x_pg(3) = x_pg(4)
         y_pg(3) = y_pg(4)
         c_pg(3) = c_pg(4)
      else if( x_pg(1) == x_pg(3) .and.  y_pg(1) == y_pg(3) ) then
         n = 3
         x_pg(3) = x_pg(4)
         y_pg(3) = y_pg(4)
         c_pg(3) = c_pg(4)
      else if( x_pg(1) == x_pg(4) .and.  y_pg(1) == y_pg(4) ) then
         n = 3
      else if( x_pg(2) == x_pg(3) .and.  y_pg(2) == y_pg(3) ) then
         n = 3
         x_pg(3) = x_pg(4)
         y_pg(3) = y_pg(4)
         c_pg(3) = c_pg(4)
      else if( x_pg(2) == x_pg(4) .and.  y_pg(2) == y_pg(4) ) then
         n = 3
      else if( x_pg(3) == x_pg(4) .and.  y_pg(3) == y_pg(4) ) then
         n = 3
      end if

      if( n == 4 ) then
         bool = .false.
         return
      end if

      bool = .true.

      if( ( x_pg(1) == x_pg(2) .and.  y_pg(1) == y_pg(2) ) .or.         &
          ( x_pg(1) == x_pg(3) .and.  y_pg(1) == y_pg(3) ) .or.         &
          ( x_pg(2) == x_pg(3) .and.  y_pg(2) == y_pg(3) )      ) then
         write(STDERR,*) "(MUESLI quad_is_tri:) internal error: input quadrangle"
         write(STDERR,*) "                      have only two different vertices!"
         write(STDERR,*) "                      Please report this bug to: Edouard.Canot@univ-rennes.fr"
         call msMuesliTrace( pause="yes" )
         stop
! --> error handle in another way !?
      end if

   end function quad_is_tri
!_______________________________________________________________________
!
   function quad_is_tri_simple( x_pg, y_pg ) result( bool )

      real(kind=MF_DOUBLE), intent(in) :: x_pg(:), y_pg(:)
      logical :: bool
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      if( size(x_pg) /= 4 ) then
         write(STDERR,*) "(MUESLI quad_is_tri:) internal error: input polygon"
         write(STDERR,*) "                      must have 4 vertices!"
         write(STDERR,*) "                      Please report this bug to: Edouard.Canot@univ-rennes.fr"
         call msMuesliTrace( pause ="no" )
         stop
      end if

      ! colors are not present ! coords are not modified !

      ! checks if the quadrangle degenerates in a triangle;
      ! this implies 6 tests

      ! In case where the check is true, the triangle is returned in the
      ! first three elements of 'x' and 'y'.

      ! If the degeneration is greater (i.e. n=2 or n=1) an error is returned

      n = 4

      if( x_pg(1) == x_pg(2) .and.  y_pg(1) == y_pg(2) ) then
         n = 3
      else if( x_pg(1) == x_pg(3) .and.  y_pg(1) == y_pg(3) ) then
         n = 3
      else if( x_pg(1) == x_pg(4) .and.  y_pg(1) == y_pg(4) ) then
         n = 3
      else if( x_pg(2) == x_pg(3) .and.  y_pg(2) == y_pg(3) ) then
         n = 3
      else if( x_pg(2) == x_pg(4) .and.  y_pg(2) == y_pg(4) ) then
         n = 3
      else if( x_pg(3) == x_pg(4) .and.  y_pg(3) == y_pg(4) ) then
         n = 3
      end if

      if( n == 4 ) then
         bool = .false.
         return
      end if

      bool = .true.

   end function quad_is_tri_simple
!_______________________________________________________________________
!
end module fgl_aux
