! f90 include file

! Writing integers as 16-bit numbers may appear as too small resolution...
! but if the clipping is done correctly, it will be ok (manual clipping
! is done in the calling routine: mf_pcolor_draw).
! The only exception is for non rectangular grids, for which the manual
! clipping is not sufficient, due to the requirement of writing a
! continuous part of a row of cells; in this case only, a test must be
! done to see whether 16-bit is sufficient, or 32-bit is required to
! avoid integer overflow. Unfortunately, the header is already written
! in the EPS or PDF file, before calling the current routine, so it is
! not easy to adapt the precision of integer during the print... The
! choice which is done here is to emit an error and to warn the user
! that he must set an optional global for increasing the integer
! resolution.
!
! Adaptive way for 16- or 32-bit integers writing, but only for the EPS...
!
! This routine is called by 'mf_pcolor_draw' in 'Pcolor_draw.f90'

!_______________________________________________________________________
!
   subroutine PatchQuadMesh_PS_PDF_shad( x, y, c, type_of_patch,        &
                                         status, finalize )

      real(kind=MF_DOUBLE), intent(in)  :: x(:,:), y(:,:)
      integer,              intent(in)  :: c(:,:), type_of_patch
      integer,              intent(out) :: status
      logical,              intent(in), optional :: finalize
      !------ API end ------

      ! Structured Mesh (not always rectangular, but quadrangles
      !                  are never self-intersecting)
      !
      ! This routine draw a complete line of cells attached to each other.

      ! Direct drawing in EPS or PDF (using shading).

      ! 'type_of_patch' indicates the line position in the matrix:
      !   1 : first line    (shading header is written)
      !   0 : middle line   (nothing else is written)
      !   2 : last line     (shading trailer is written)
      !
      ! 'status' is an output flag (used only in DEBUG mode) indicating
      ! whether integer overflows may occur:
      !    0 => no overflow
      !   -1 => integer overflow may occur with 16-bit integers
      !   -2 => integer overflow may occur with 32-bit integers
      !
      ! Note that for a simple mesh who have only one line of cells, we
      ! must have 'type_of_patch' = 3 (i.e. the sum), in order to write
      ! all things.

      real(kind=MF_DOUBLE) :: xorg, yorg, xscale, yscale
      real(kind=MF_DOUBLE) :: xx(4), yy(4)
      integer :: icol(4), i, j, n, job, ier, L
      logical :: first, last
      real(kind=MF_DOUBLE) :: par_val(4), one_third(2), two_third(2)
      character(len=80) :: inline
      character(len=10) :: str1
      integer :: max_xx_abs, max_yy_abs, max_data
      logical :: finalization

      ! deflate_stream_to_hex support (zlib)
      integer, parameter :: len_in_max = 65536
      character(len=len_in_max), save :: string_in, string_out
      integer, save :: len_in, len_out
      interface
         subroutine deflate_stream_to_hex( in_bin_str, len_in_bin,      &
                                           out_hex_str, len_out_hex,    &
                                           job, ier )
            character(len=*) :: in_bin_str, out_hex_str
            integer          :: len_in_bin, len_out_hex, job, ier
         end subroutine
         subroutine deflate_stream_to_a85( filename, len_filename,      &
                                           buffer, len_buffer,          &
                                           job, ier )
            character(len=*) :: filename
            integer          :: len_filename
            character(len=*) :: buffer
            integer          :: len_buffer, job, ier
         end subroutine
      end interface

      character(len=256), save :: fnameC
      integer, save :: lfnameC
      type(mf_win_info), pointer :: win

      integer :: BPC, full_range, coord_shift
      integer, parameter :: nb_try_BPC = 2
      integer, parameter :: BitsPerCoordinate(nb_try_BPC) = [ 16, 24 ]
      character(len=4), parameter :: HexForm(nb_try_BPC) = [ "Z4.4", "Z6.6" ]
      real(kind=MF_DOUBLE) :: range_check(4), rdummy
      character(len=25) :: BitsPerCoordinate_str
      character(len=96) :: Decode_str
      character(len=32) :: hexa_format
      logical :: alert
      logical, save :: alert_already_printed = .false.

   !------ end of declarations -- execution starts hereafter  ------

      status = 0

      win => mf_win_db(CURRENT_WIN_ID)

      xorg = pgxorg(pgid)
      yorg = pgyorg(pgid)
      xscale = pgxscl(pgid)
      yscale = pgyscl(pgid)

   if( PRINTING_EPS ) then
      range_check(:) = [ minval(x), minval(y), maxval(x), maxval(y) ]
      ! convert to device coords
      range_check(1:3:2) = xorg + xscale*range_check(1:3:2)
      range_check(2:4:2) = yorg + yscale*range_check(2:4:2)
      ! Check numerical range of these coordinates to see whether they
      ! can be written in hexadecimal using 16 or 24 bits.
      alert = .true.
      do i = 1, nb_try_BPC
         BPC = BitsPerCoordinate(i)
         full_range = 2**BPC - 1
         coord_shift = 2**(BPC-1) - 1 ! half range - 1
         if( all(nint(range_check(1:2))+coord_shift > 0) .and.          &
             all(nint(range_check(3:4))+coord_shift < full_range) ) then
            alert = .false.
            write(BitsPerCoordinate_str,"(I0)") BPC
            ! write, e.g. "  /BitsPerCoordinate 16"
            BitsPerCoordinate_str = "  /BitsPerCoordinate "             &
                                    // trim(BitsPerCoordinate_str)
            ! write, e.g. "  /Decode [ -32767 32768 -32767 32768 0 1 ]"
            write(Decode_str,"('  /Decode [ ',4(I0,1X),'0 1 ]')" )      &
                                    -coord_shift, coord_shift+1,        &
                                    -coord_shift, coord_shift+1!!print *, 'Decode_str = "', trim(Decode_str), '"'
            ! write, e.g. "(6(Z4.4))"
            hexa_format = "(6(" // HexForm(i) // "))"
            exit
         end if
      end do

      if( alert .and. .not. alert_already_printed ) then
         print *, char(27) // "[38;5;202m" ! to switch to orange printing
         print *, "(MUESLI msPatchQuad:) Warning: 'out-of-range' pixel values!"
         print *, "     The algorithm to print a quadrangle mesh in EPS or PDF is not yet"
         print *, "     reliable when using 'shading'='interp' AND zooming strongly in the"
         print *, "     viewport. Hexadecimal overflows prevent to obtain valid printed image,"
         print *, "     therefore this part of your image is skipped."
         print *
         print *, "  -> The only way to avoid this is to revert to 'shading'='flat' OR to"
         print *, "     unzoom the axis before printing (anyway, most of EPS or PDF viewers"
         print *, "     are able to zoom inside the image)."
         print *
         print *, "  -> Note that this warning is printed only one time. The other patches"
         print *, "     of the current image may have or not the same problem!"
         print *, char(27) // "[0m" ! to revert to normal printing
         alert_already_printed = .true.
         return
      end if
   end if

      if( present(finalize) ) then
         finalization = finalize
      else
         finalization = .false.
      end if

      if( finalization ) then
         last = .true.
         go to 10 ! we don't want to enter in the loop
      end if

      first = mod(type_of_patch,2) == 1
      last = type_of_patch/2 == 1

      n = size(x,2)

      do i = 1, n-1

         if( mod(i,2) == 0 ) then
            ! device coords
            xx(1) = nint( xorg + xscale*x(1,i) )
            yy(1) = nint( yorg + yscale*y(1,i) )
            icol(1) = c(1,i)
            xx(2) = nint( xorg + xscale*x(2,i) )     !    1 +---------+ 4
            yy(2) = nint( yorg + yscale*y(2,i) )     !      |         |
            icol(2) = c(2,i)                         !      |         |
            xx(3) = nint( xorg + xscale*x(2,i+1) )   !      |         |
            yy(3) = nint( yorg + yscale*y(2,i+1) )   !      |         |
            icol(3) = c(2,i+1)                       !    2 +---------+ 3
            xx(4) = nint( xorg + xscale*x(1,i+1) )
            yy(4) = nint( yorg + yscale*y(1,i+1) )
            icol(4) = c(1,i+1)
         else
            ! device coords
            xx(1) = nint( xorg + xscale*x(2,i) )
            yy(1) = nint( yorg + yscale*y(2,i) )
            icol(1) = c(2,i)
            xx(2) = nint( xorg + xscale*x(1,i) )     !    2 +---------+ 3
            yy(2) = nint( yorg + yscale*y(1,i) )     !      |         |
            icol(2) = c(1,i)                         !      |         |
            xx(3) = nint( xorg + xscale*x(1,i+1) )   !      |         |
            yy(3) = nint( yorg + yscale*y(1,i+1) )   !      |         |
            icol(3) = c(1,i+1)                       !    1 +---------+ 4
            xx(4) = nint( xorg + xscale*x(2,i+1) )
            yy(4) = nint( yorg + yscale*y(2,i+1) )
            icol(4) = c(2,i+1)
         end if
         where( icol < win%colormap_ci_low )
            icol = win%colormap_ci_low
         else where( win%colormap_ci_high < icol )
            icol = win%colormap_ci_high
         end where
         ! parametric value must be in [0.0, 1.0] as specified in the
         ! shading dictionnary (no need to call pgqcr to get the RGB components)
         par_val(:) = dble(icol(:)-win%colormap_ci_low)                 &
                      / (win%colormap_ci_high-win%colormap_ci_low)

#ifndef _OPTIM
         if( .not. PRINTING_EPS ) then
            ! Once the coordinates have been obtained, it can be checked
            !  whetherthey exceed the current integer limit (16- or 32-bit
            ! limit).
            ! PRINTING_PDF
            max_xx_abs = maxval( abs(xx) )
            max_yy_abs = maxval( abs(yy) )
            max_data = max( max_xx_abs, max_yy_abs )
            if( max_data > 32767 - 1 ) status = -1
            if( max_data > 2147483647 - 1 ) status = -2
         end if
#endif

         if( i == 1 .and. first ) then
            if( PRINTING_EPS ) then

               call gresc("<<")
               call gresc("  /ShadingType 6")
               call gresc("  /ColorSpace /DeviceRGB")
               call gresc("  /BitsPerFlag 8")
               call gresc( trim(BitsPerCoordinate_str) )
               call gresc("  /BitsPerComponent 8")
               call gresc( trim(Decode_str) )
               call gresc("  /DataSource <")

            else ! PRINTING_PDF
               ! preparing compression, only for PDF
               job = 1
               if( MF_DEFLATE_TO_A85 ) then
                  fnameC = trim(PDF_NAME) // char(0)
                  lfnameC = len_trim(PDF_NAME) + 1
                  close( MF_PDF_UNIT )
                  ! the following routine writes directly in the PDF file.
                  call deflate_stream_to_a85( fnameC, lfnameC,          &
                                              string_in, 0, job, ier )
               else
                  ! à l'initialisation, on a besoin de 'len_in' pour allouer
                  ! une fois pour toute le tableau de compression 'compr_str'
                  len_in = len_in_max
                  string_out = ""
                  len_out = len(string_out)
                  call deflate_stream_to_hex( string_in, len_in,        &
                                              string_out, len_out,      &
                                              job, ier )
               end if
               len_in = 0

            end if

         end if

         if( PRINTING_PDF ) then
            ! Below, the value 106 comes from the greater number of chars
            ! added to strin_in, each time the routine is called.
            if( len_in+106 > len_in_max ) then
               ! compress string_in and write the output in the PDF file
               job = 2
               if( MF_DEFLATE_TO_A85 ) then
                  call deflate_stream_to_a85( "", 0,                    &
                                              string_in, len_in, job, ier )
               else
                  call deflate_stream_to_hex( string_in, len_in,        &
                                              string_out, len_out,      &
                                              job, ier )
               end if
               if( ier /= 0 ) then
                  call PrintMessage( "PatchQuadMesh_PS_PDF_shad", "E",  &
                                     "while deflating data" )
                  return
               end if

               if( .not. MF_DEFLATE_TO_A85 ) then
                  L = 0
                  ! on écrit la chaîne résultante de ZLIB par ligne de
                  ! 132 caractères
                  ! (c'est la largeur utilisée dans MFPLOT pour les EPS/PDF)
                  do
                     if( L >= len_out ) exit
                     if( L+132 >= len_out ) then
                        call gresc( string_out(L+1:len_out) )
                        exit
                     else
                        call gresc( string_out(L+1:L+132) )
                        L = L + 132
                     end if
                  end do
               end if
               len_in = 0
            end if
         end if

         ! quadrangle 1-2-3-4: Warning, position of control points is very
         ! important to obtain a strictly bilinear colour interpolation inside
         ! the rectangle. Indeed, the mapping of the boundary parameter must
         ! be linear itself: therefore, control points must be located at the
         ! thirds (1/3 and 2/3) of each side.
         if( i == 1 ) then

            if( PRINTING_EPS ) then
               call gresc("00") ! flag
            else ! PRINTING_PDF
               string_in(len_in+1:len_in+2) = "00" ! flag
               len_in = len_in + 2
            end if
            do j = 1, 3
               one_third = [ xx(j)*2./3.+xx(j+1)*1./3., yy(j)*2./3.+yy(j+1)*1./3. ]
               two_third = [ xx(j)*1./3.+xx(j+1)*2./3., yy(j)*1./3.+yy(j+1)*2./3. ]
               if( PRINTING_EPS ) then
                  write(inline,hexa_format) nint(xx(j))+coord_shift,    &
                                            nint(yy(j))+coord_shift,    &
                                            nint(one_third)+coord_shift, &
                                            nint(two_third)+coord_shift
                  call gresc( trim(inline) )
               else ! PRINTING_PDF
                  write(string_in(len_in+1:len_in+24),110)              &
                           nint(xx(j))+32767, nint(yy(j))+32767,        &
                           nint(one_third)+32767, nint(two_third)+32767
                  len_in = len_in + 24
               end if
            end do
            one_third = [ xx(4)*2./3.+xx(1)*1./3., yy(4)*2./3.+yy(1)*1./3. ]
            two_third = [ xx(4)*1./3.+xx(1)*2./3., yy(4)*1./3.+yy(1)*2./3. ]
            if( PRINTING_EPS ) then
               write(inline,hexa_format) nint(xx(4))+coord_shift,       &
                                         nint(yy(4))+coord_shift,       &
                                         nint(one_third)+coord_shift,   &
                                         nint(two_third)+coord_shift
               call gresc( trim(inline) )
               write(inline,200) nint( par_val(:)*255 )
               call gresc( trim(inline) )
            else ! PRINTING_PDF
               write(string_in(len_in+1:len_in+24),110)                 &
                                 nint(xx(4))+32767, nint(yy(4))+32767,  &
                                 nint(one_third)+32767, nint(two_third)+32767
               len_in = len_in + 24
               write(string_in(len_in+1:len_in+8),210) nint( par_val(:)*255 )
               len_in = len_in + 8
            end if

         else

            if( PRINTING_EPS ) then
               call gresc("02") ! flag
            else ! PRINTING_PDF
               string_in(len_in+1:len_in+2) = "02" ! flag
               len_in = len_in + 2
            end if
            one_third = [ xx(2)*2./3.+xx(3)*1./3., yy(2)*2./3.+yy(3)*1./3. ]
            two_third = [ xx(2)*1./3.+xx(3)*2./3., yy(2)*1./3.+yy(3)*2./3. ]
            if( PRINTING_EPS ) then
               write(inline,hexa_format) nint(one_third)+coord_shift,   &
                                         nint(two_third)+coord_shift
               call gresc( trim(inline) )
            else ! PRINTING_PDF
               write(string_in(len_in+1:len_in+16),110) nint(one_third)+32767, &
                                                        nint(two_third)+32767
               len_in = len_in + 16
            end if
            one_third = [ xx(3)*2./3.+xx(4)*1./3., yy(3)*2./3.+yy(4)*1./3. ]
            two_third = [ xx(3)*1./3.+xx(4)*2./3., yy(3)*1./3.+yy(4)*2./3. ]
            if( PRINTING_EPS ) then
               write(inline,hexa_format) nint(xx(3))+coord_shift,       &
                                         nint(yy(3))+coord_shift,       &
                                         nint(one_third)+coord_shift,   &
                                         nint(two_third)+coord_shift
               call gresc( trim(inline) )
            else ! PRINTING_PDF
               write(string_in(len_in+1:len_in+24),110)                 &
                                 nint(xx(3))+32767, nint(yy(3))+32767,  &
                                 nint(one_third)+32767, nint(two_third)+32767
               len_in = len_in + 24
            end if
            one_third = [ xx(4)*2./3.+xx(1)*1./3., yy(4)*2./3.+yy(1)*1./3. ]
            two_third = [ xx(4)*1./3.+xx(1)*2./3., yy(4)*1./3.+yy(1)*2./3. ]
            if( PRINTING_EPS ) then
               write(inline,hexa_format) nint(xx(4))+coord_shift,       &
                                         nint(yy(4))+coord_shift,       &
                                         nint(one_third)+coord_shift,   &
                                         nint(two_third)+coord_shift
               call gresc( trim(inline) )
               write(inline,200) nint( par_val(3:4)*255 )
               call gresc( trim(inline) )
            else ! PRINTING_PDF
               write(string_in(len_in+1:len_in+24),110)                 &
                                 nint(xx(4))+32767, nint(yy(4))+32767,  &
                                 nint(one_third)+32767, nint(two_third)+32767
               len_in = len_in + 24
               write(string_in(len_in+1:len_in+4),210) nint( par_val(3:4)*255 )
               len_in = len_in + 4
            end if

         end if
         ! format requirements for PRINTING_EPS
 100     format(6(Z4.4))
 200     format(4(Z2.2))
         ! format requirements for PRINTING_PDF
 110     format(6(Z4.4))
 210     format(4(Z2.2))

! DEBUG: the two following lines may be used to write no data
!!len_in = 1
!!string_in = " "

      end do

 10   continue
      if( last ) then
         if( PRINTING_EPS ) then
            call gresc("  >")
            call gresc("  colormap")
            call gresc(">>")
            call gresc("shfill")
         else ! PRINTING_PDF
            if( len_in /= 0 ) then
               ! compress string_in and write the output in the PDF file
               job = 3
               if( MF_DEFLATE_TO_A85 ) then
                  call deflate_stream_to_a85( "", 0,                 &
                                              string_in, len_in, job, ier )
                  ! reopen the file, close at step 1.
                  open( MF_PDF_UNIT, file=trim(PDF_NAME), position="append" )
               else
                  call deflate_stream_to_hex( string_in, len_in,     &
                                              string_out, len_out,   &
                                              job, ier )
               end if
               if( ier /= 0 ) then
                  call PrintMessage( "PatchQuadMesh_PS_PDF_shad", "E", &
                                     "while deflating data" )
               end if
               if( .not. MF_DEFLATE_TO_A85 ) then
                  L = 0
                  ! on écrit la chaîne résultante de ZLIB par ligne de
                  ! 132 caractères
                  ! (c'est la largeur utilisée dans MFPLOT pour les EPS/PDF)
                  do
                     if( L >= len_out ) exit
                     if( L+132 >= len_out ) then
                        call gresc( string_out(L+1:len_out) )
                        exit
                     else
                        call gresc( string_out(L+1:L+132) )
                        L = L + 132
                     end if
                  end do
               end if
            end if

            if( MF_DEFLATE_TO_A85 ) then
               call gresc("~>")
            else
               call gresc(">")
            end if

         end if
      end if

   end subroutine PatchQuadMesh_PS_PDF_shad
