/*
 **GR_REAL_USER -- get user full name (POSIX)
 *+
 *     subroutine GR_REAL_USER( USERNAME, FULLNAME, L )
 *     character(len=*), intent(in)  :: USERNAME
 *     character(len=*), intent(out) :: FULLNAME
 *     integer,          intent(out) :: L
 *
 * Get the full name of the user running the program.
 *
 * Arguments:
 *  USERNAME : the unix user name
 *  FULLNAME : receives user full name, truncated or extended with
 *             blanks as necessary.
 *  L        : receives the number of characters in FULLNAME, excluding
 *             trailing blanks.
 *--
 * 01-Aug-2018 (EC) Added the full user name in parenthesis
 *-----------------------------------------------------------------------
 */

#include <sys/types.h>
#include <pwd.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>

#ifdef UNDERSCORE
#  define GR_REAL_USER gr_real_user_
#else
#  define GR_REAL_USER gr_real_user
#endif

/* Get full name of a user, given their username. Return 0 for not found,
   -1 for error, or 1 for success. Copy name to `fullname`, but only up
   to maxlen-1 chars (maxlen includes trailing '\0'). Note that if the GECOS
   field contains commas, only up to to (but not including) the first comma
   is copied, since the commas are a convention to add more than just the
   name into the field, e.g., room number, phone number, etc. */
int GR_REAL_USER( const char *username, char *fullname, int *length,
                  int max_len_in, int max_len_out ){
    struct passwd *p;
    int n;

    errno = 0;
    p = getpwnam(username);
    if( p == NULL && errno == 0 )
        return 0;
    if( p == NULL )
        return -1;
    if( max_len_out == 0 )
        return 1;
    n = strcspn(p->pw_gecos, ",");
    if( n > max_len_out - 1 )
        n = max_len_out - 1;
    memcpy(fullname, p->pw_gecos, n);
    fullname[n] = '\0';
    *length = n;
    /* Pad to the end of the output string with spaces. */
    for( ; n<max_len_out; n++ )
    fullname[n] = ' ';
    return 1;
}
