/*
 * Part of material in this file comes from:
 *
 *      libbmeps - Bitmap to EPS conversion library
 *      Copyright (C) 2000 - Dirk Krause
 *
 *      http://sourceforge.net/projects/bmeps
 *      license: LGPL
 *
 *      replaced by: bmpp
 *      http://dktools.sourceforge.net/bmpp.html
 */

#include <stdio.h>
#include "zlib.h"
#include <string.h>
#include <stdlib.h>

#define CHECK_ERR_MUESLI(err, msg, routine) { \
  if( err != Z_OK ) { \
    fprintf( stderr, "\e[31m" ); \
    fprintf( stderr, "\n(MUESLI:) internal error in deflate_stream_to_hex ***\n" ); \
    fprintf( stderr, "          %s: %s error = %d\n", routine, msg, err ); \
    fprintf( stderr, "\e[0m" ); \
    fprintf( stderr, "\n *** PAUSE for debugging purpose only ***\n" ); \
    fprintf( stderr, "     (Return to resume...)    \n" ); \
    getchar(); \
    exit(1); \
  } \
}

static char hexdigits[] = {
  "0123456789ABCDEF"
};

z_stream c_stream;
static int out_pos;

static Byte * compr_str;
static uLong len_compr_str;

/**********************************************************************/
void my_deflate_init() {
  int err;

  c_stream.zalloc = (alloc_func)0;
  c_stream.zfree = (free_func)0;
  c_stream.opaque = (voidpf)0;

  out_pos = 0;

  err = deflateInit(&c_stream, 9);
  CHECK_ERR_MUESLI( err, "deflateInit", "my_deflate_init" );
}

/**********************************************************************/
void my_deflate( unsigned char *in_string, int len_in_string,
                 Byte *compr, uLong comprLen, int *out_len ) {
  int err;

  c_stream.next_in  = (Bytef*)in_string;
  c_stream.next_out = compr;

  c_stream.avail_in = (uLong)len_in_string;
  c_stream.avail_out = comprLen;
  err = deflate(&c_stream, Z_FULL_FLUSH);
  CHECK_ERR_MUESLI( err, "deflate", "my_deflate" );

  *out_len = c_stream.total_out - out_pos;
  out_pos = c_stream.total_out;
}

/**********************************************************************/
void my_deflate_end( unsigned char *in_string, int len_in_string,
                     Byte *compr, uLong comprLen, int *out_len ) {
  int err;

  c_stream.next_in  = (Bytef*)in_string;
  c_stream.next_out = compr;

  c_stream.avail_in = (uLong)len_in_string;
  c_stream.avail_out = comprLen;
  err = deflate(&c_stream, Z_FINISH);
  if (err != Z_STREAM_END) {
    CHECK_ERR_MUESLI( err, "deflate", "my_deflate_end" );
  }

  err = deflateEnd(&c_stream);
  CHECK_ERR_MUESLI( err, "deflateEnd", "my_deflate_end" );

  *out_len = c_stream.total_out - out_pos;
}

/**********************************************************************/
/* C Fortran interface */
#ifdef UNDERSCORE
#  define DEFLATE_STREAM_TO_HEX deflate_stream_to_hex_
#else
#  define DEFLATE_STREAM_TO_HEX deflate_stream_to_hex
#endif
void DEFLATE_STREAM_TO_HEX(
                 unsigned char * in_bin_str, int * len_in_bin,
                 unsigned char * out_hex_str, int * len_out_hex,
                 int * job, int * ier )
/*
  job == 1 : deflate_init
         2 : deflate
         3 : deflate_end

  The output in an HEXA string (2 chars per byte).
  It is intended to embed compressed bitmap images in EPS and PDF files.
  Two consecutive decoding filters must be used: /ASCIIHexDecode then
    /FlateDecode.

  Allocation of 'compr_str' is made only one time... this suppose that all
  calls to the current routine use input hexa strings of same length, which
  is true for lines of a bitmap image (all have the same width).
 */
{
  int out_len;
  int i, j;

  *ier = 0;

  if( *job == 1 ) {

    my_deflate_init();

    len_compr_str = 2*(*len_in_bin);
    compr_str = (Byte*)calloc((uInt)len_compr_str, 1);
    if( compr_str == Z_NULL ) {
      fprintf( stderr, "(MUESLI:) deflate_stream_to_hex: out of memory\n" );
      fprintf( stderr, "          => tried to allocate : %lu bytes !\n", len_compr_str );
      exit(1);
    }

    return;

  }

  if( *job == 2 ) {
    my_deflate( in_bin_str, *len_in_bin, compr_str, len_compr_str, &out_len );
  } else {
    my_deflate_end( in_bin_str, *len_in_bin, compr_str, len_compr_str, &out_len );
  }

  if( 2*out_len > *len_out_hex ) {
    printf("(MUESLI:) deflate_stream_to_hex: out_hex_str is too short !\n");
    printf("          current length : %d\n",*len_out_hex);
    printf("          required length : %d\n",2*out_len);
    *ier = 1;
    return;
  }
  *len_out_hex = 2*out_len;
  j = 0;
  for( i=0;i<out_len;i++ ) {
    out_hex_str[j]   = hexdigits[ (unsigned int)compr_str[i] / 16 ];
    out_hex_str[j+1] = hexdigits[ (unsigned int)compr_str[i] % 16 ];
    j = j + 2;
  }

  if( *job == 3 ) {
    free(compr_str);
  }
}
