/*
 * Part of material in this file comes from:
 *
 *      libbmeps - Bitmap to EPS conversion library
 *      Copyright (C) 2000 - Dirk Krause
 *
 *      http://sourceforge.net/projects/bmeps
 *      license: LGPL
 *
 *      replaced by: bmpp
 *      http://dktools.sourceforge.net/bmpp.html
 *
 * Modifications made by É. Canot on July 2018 for the Muesli library.
 * Copyright É. Canot 2003-2025 -- IPR/CNRS
 * Edouard.Canot@univ-rennes.fr
 *
 */

#include <stdio.h>
#include <zlib.h>
#include <ctype.h>
#include <stdlib.h>

/* Compressed stream (ASCII85) is written over 132 columns in the EPS/PDF
 * (like HEXA compressed bitmaps) in Muesli */
#define COL_WIDTH 132

#define FINALOUTPUT(c) fputc((c),o->out)

/* Error management:
 *   oe_err = 0 : ok
 *   oe_err = 1 : Error: Cannot initialize Zlib.
 *   oe_err = 2 : Error during 'deflate' (Zlib).
 *   oe_err = 3 : Error: Cannot finalize Zlib.
 */
static int oe_err;

typedef struct {
  FILE *out;
  int textpos;
  unsigned long a85_value;
  int      a85_consumed;
  unsigned long a85_4;
  unsigned long a85_3;
  unsigned long a85_2;
  unsigned long a85_1;
  unsigned long a85_0;
  z_stream flate_stream;
  Bytef *fl_i_buffer;
  Bytef *fl_o_buffer;
  uLong  fl_i_size;
  uLong  fl_o_size;
  uLong  fl_i_used;
} Output_Encoder;

static void oe_init(Output_Encoder *o, FILE *out,
                    Bytef *flib, size_t flis, Bytef *flob, size_t flos ) {
  oe_err = 0;
  o->out = out;
  o->textpos = 0;
  o->a85_value = 0; o->a85_consumed = 0;
  o->a85_0 = 1;
  o->a85_1 = 85;
  o->a85_2 = 85 * 85;
  o->a85_3 = 85 * o->a85_2;
  o->a85_4 = 85 * o->a85_3;
  if(flib && flis && flob && flos) {
    (o->flate_stream).zfree = (free_func)0;
    (o->flate_stream).zalloc = (alloc_func)0;
    (o->flate_stream).opaque = (voidpf)0;
    if(deflateInit((&(o->flate_stream)),9) != Z_OK) {
      oe_err = 1;
    }
  }
  o->fl_i_buffer = flib;
  o->fl_o_buffer = flob;
  o->fl_i_size   = flis;
  o->fl_o_size   = flos;
  o->fl_i_used   = 0;
}

static char ascii85_char(int x) {
  x += 33;
  return (char)x;
}

static void ascii85_output(Output_Encoder *o) {
  int i;
  char buffer[6], c, *ptr;
  unsigned long value;
  value = o->a85_value;
  buffer[0] = ascii85_char(value / (o->a85_4));
  value = value % (o->a85_4);
  buffer[1] = ascii85_char(value / (o->a85_3));
  value = value % (o->a85_3);
  buffer[2] = ascii85_char(value / (o->a85_2));
  value = value % (o->a85_2);
  buffer[3] = ascii85_char(value / (o->a85_1));
  value = value % (o->a85_1);
  buffer[4] = ascii85_char(value);
  buffer[5] = '\0';
  i = o->a85_consumed + 1;
  ptr = buffer;
  o->textpos = o->textpos + i;
  while(i--) {
    c = *(ptr++);
    FINALOUTPUT(c) ;
  }
  if(o->textpos >= COL_WIDTH) {
    FINALOUTPUT('\n') ;
    o->textpos = 0;
  }
}

static void ascii85_add(Output_Encoder *o, int b) {
  if( b < 0 ) b += 256;
  o->a85_value = 256 * (o->a85_value) + b;
  o->a85_consumed = o->a85_consumed + 1;
  if(o->a85_consumed >= 4) {
    ascii85_output(o);
    o->a85_value = 0;
    o->a85_consumed = 0;
  }
}

static void ascii85_flush(Output_Encoder *o) {
  if(o->a85_consumed > 0) {
    int i = o->a85_consumed;
    while(i < 4) {
      o->a85_value = 256 * o->a85_value;
      i++;
    }
    ascii85_output(o);
    o->a85_value = 0;
    o->a85_consumed = 0;
  }
  if(o->textpos > 0) {
    FINALOUTPUT('\n') ;
    o->textpos = 0;
  }
}

static void do_flate_flush(Output_Encoder *o, int final) {
  Bytef *iptr, *optr, *xptr;
  uLong  is, os, xs;
  int err, must_continue;

  iptr = o->fl_i_buffer; optr = o->fl_o_buffer;
  is = o->fl_i_size; os = o->fl_o_size;

  if(iptr && optr && is && os) {
    is = o->fl_i_used;
    if(is) {
      (o->flate_stream).next_in = iptr;
      (o->flate_stream).avail_in = is;
      if(final) {
        must_continue = 1;
        while(must_continue) {
          (o->flate_stream).next_out = optr;
          (o->flate_stream).avail_out = os;
          must_continue = 0;
          err = deflate(&(o->flate_stream), Z_FINISH);
          switch(err) {
            case Z_STREAM_END: {
              xptr = optr;
              xs = os - ((o->flate_stream).avail_out);
              while(xs--) {
                ascii85_add(o, (*(xptr++)));
              }
            } break;
            case Z_OK : {
              must_continue = 1;
              xptr = optr;
              xs = os - ((o->flate_stream).avail_out);
              while(xs--) {
                ascii85_add(o, (*(xptr++)));
              }
            } break;
            default : {
              oe_err = 2;
            } break;
          }
        }
      } else {
        must_continue = 1;
        while(must_continue) {
          must_continue = 0;
          (o->flate_stream).avail_out = os; (o->flate_stream).next_out = optr;
          err = deflate(&(o->flate_stream), 0);
          switch(err) {
            case Z_OK: {
              if((o->flate_stream).avail_in) {
                must_continue = 1;
              }
              xptr = optr;
              xs = os - ((o->flate_stream).avail_out);
              while(xs--) {
                ascii85_add(o, (*(xptr++)));
              }
            } break;
            default : {
              oe_err = 2;
            } break;
          }
        }
      }
    }
  }
}

static void flate_add(Output_Encoder *o, int b) {
  Byte bt;
  Bytef *btptr;

  btptr = o->fl_i_buffer;
  if(btptr) {
    bt = (Byte)b;
    btptr[(o->fl_i_used)] = bt;
    o->fl_i_used += 1;
    if(o->fl_i_used >= o->fl_i_size) {
      do_flate_flush(o, 0);
      o->fl_i_used = 0;
    }
  }
}

static void flate_flush(Output_Encoder *o) {
  do_flate_flush(o,1);
  if( deflateEnd(&(o->flate_stream)) != Z_OK ) {
    oe_err = 3;
  }
  ascii85_flush(o);
}

/* C Fortran interface */
#ifdef UNDERSCORE
#  define DEFLATE_STREAM_TO_A85 deflate_stream_to_a85_
#else
#  define DEFLATE_STREAM_TO_A85 deflate_stream_to_a85
#endif
void DEFLATE_STREAM_TO_A85(
           char * filename, int * len_filename,
           unsigned char * buffer, int * len_buffer,
           int * job, int * ier ) {
/*
  job == 1 : deflate_init
         2 : deflate
         3 : deflate_end

  The output in an ASCII85 string (1 printable chars per byte).
  It is intended to embed compressed bitmap images in EPS and PDF files.
  Two consecutive decoding filters must be used: /ASCII85Decode then
    /FlateDecode.

  The algorithm uses stream, this means that no allocation/deallocation
  are required during the process. Moreover, the ASCII85 output is written
  directly in the EPS or PDF file (filename argument) in an append mode,
  without copy in a string.

  'ier' is a returned code for error. See the values of this variable at
  the beginning of this file.
 */
  static Output_Encoder o;

  static Byte flate_in_buffer[16384];
  static Byte flate_out_buffer[17408];

  static FILE * file;

  if( *job == 1 ) {

    /* Fortran string must be terminated by the NULL character */
    file = fopen( filename, "a" );

    oe_init( &o, file,
             flate_in_buffer, sizeof(flate_in_buffer),
             flate_out_buffer, sizeof(flate_out_buffer) );

    *ier = oe_err;
  }

  if( (*job == 2) | (*job == 3) ) {
    oe_err = 0;
    if( *len_buffer > 0 ) {
      char *ptr = (char*)buffer;
      int i;
      for( i = 0; i < *len_buffer; i++ ) {
        int c = *ptr;
        flate_add( &o, c );
        ptr++;
      }
    }
    *ier = oe_err;
  }

  if( *job == 3 ) {
    oe_err = 0;
    flate_flush( &o );
    fclose( file );
    *ier = oe_err;
  }

}

