! Primitive routine to draw Graph Markers  with error bars at each points.
! The markers are drawn using some attributes (see below).
!
! Arguments:
!  N      (input)  : number of points to mark.
!  XPTS   (input)  : world x-coordinates of the points.
!  YPTS   (input)  : world y-coordinates of the points.
!  X_ERR, Y_ERR  (input)  : world x- or y-errors.
!  SYMBOL (input)  : code number of the symbol to be drawn at each point:
!
!                     1..26   : new marker symbols (size = current
!                               character-height)
!                    -1..-26  : same markers (using abs(SYMBOL)) but
!                               with a white border around the symbol.
!
!                    any other: a single dot (diameter = current
!                               line-width).
!
!  LW     (input)  : linewidth (of the error bars)
!--
!  5-Sep-2017 : Creation.
! 29-Feb-2020 - Use now double precision instead of single precision.
!  3-Apr-2020 - Update explanation in the header; fix the code for
!               testing the number symbol. Generalized polygons are
!               removed (see grmker).
!  6-Apr-2020 - Calling syntax to GRMKER updated.
!  8-Apr-2020 - Remove call to GRDOT1 (never used).
! 26-Aug-2020 - Fix the length of the ticks.
! 29-Oct-2021 - Added 5 new symbols. Total number is now 26 instead of 21.
!-----------------------------------------------------------------------

subroutine PGPT_ERRORBARX( n, xpts, ypts, x_err, symbol, lw )

   integer :: n
   double precision :: xpts(*), ypts(*), x_err(*), lw
   integer :: symbol
   !------ API end ------

   integer :: symb, i
   double precision :: x_lo, x_hi, y_lo, y_hi, ratio

   if( n < 1 ) return

   symb = abs(symbol)

   call pgbbuf()

   if( 1 <= symb .and. symb <= 26 ) then
      ! second argument tells grmker to use world coordinates
      call grmker( symbol, n, xpts, ypts )
   else
      call grwarn('PGPT_ERRORBARX - invalid symbol number')
   end if

   ratio = pgyscl(pgid) / pgxscl(pgid)

   ! set line width
   call grslw(lw)
   do i = 1, n
      ! adding the error bars along the x-axis
      ! 1) horizontal part
      x_lo = xpts(i) - x_err(i)
      call grmova( x_lo, ypts(i) )
      x_hi = xpts(i) + x_err(i)
      call grlina( x_hi, ypts(i) )
      ! 2) the ticks
      y_lo = ypts(i) - 0.2d0*x_err(i)/ratio
      y_hi = ypts(i) + 0.2d0*x_err(i)/ratio
      call grmova( x_lo, y_lo )
      call grlina( x_lo, y_hi )
      call grmova( x_hi, y_lo )
      call grlina( x_hi, y_hi )
   end do

   call pgebuf()

end subroutine pgpt_errorbarx
!_______________________________________________________________________
!
subroutine PGPT_ERRORBARY( n, xpts, ypts, y_err, symbol, lw )

   integer :: n
   double precision :: xpts(*), ypts(*), y_err(*), lw
   integer :: symbol
   !------ API end ------

   integer :: symb, i
   double precision :: x_lo, x_hi, y_lo, y_hi, ratio

   if( n < 1 ) return

   symb = abs(symbol)

   call pgbbuf()

   if( 1 <= symb .and. symb <= 26 ) then
      ! second argument tells grmker to use world coordinates
      call grmker( symbol, n, xpts, ypts )
   else
      call grwarn('PGPT_ERRORBARY - invalid symbol number')
   end if

   ratio = pgyscl(pgid) / pgxscl(pgid)

   ! set line width
   call grslw(lw)
   do i = 1, n
      ! adding the error bars along the y-axis
      ! 1) vertical part
      y_lo = ypts(i) - y_err(i)
      call grmova( xpts(i), y_lo )
      y_hi = ypts(i) + y_err(i)
      call grlina( xpts(i), y_hi )
      ! 2) the ticks
      x_lo = xpts(i) - 0.2d0*y_err(i)*ratio
      x_hi = xpts(i) + 0.2d0*y_err(i)*ratio
      call grmova( x_lo, y_lo )
      call grlina( x_hi, y_lo )
      call grmova( x_lo, y_hi )
      call grlina( x_hi, y_hi )
   end do

   call pgebuf()

end subroutine pgpt_errorbary
!_______________________________________________________________________
!
subroutine PGPT_ERRORBARXY( n, xpts, ypts, x_err, y_err, symbol, lw )

   integer :: n
   double precision :: xpts(*), ypts(*), x_err(*), y_err(*), lw
   integer :: symbol
   !------ API end ------

   integer :: symb, i
   double precision :: x_lo, x_hi, y_lo, y_hi, ratio

   if( n < 1 ) return

   symb = abs(symbol)

   call pgbbuf()

   if( 1 <= symb .and. symb <= 26 ) then
      ! second argument tells grmker to use world coordinates
      call grmker( symbol, n, xpts, ypts )
   else
      call grwarn('PGPT_ERRORBARXY - invalid symbol number')
   end if

   ratio = pgyscl(pgid) / pgxscl(pgid)

   ! set line width
   call grslw(lw)
   do i = 1, n
      ! adding the error bars along the x-axis
      ! 1) horizontal part
      x_lo = xpts(i) - x_err(i)
      call grmova( x_lo, ypts(i) )
      x_hi = xpts(i) + x_err(i)
      call grlina( x_hi, ypts(i) )
      ! 2) the ticks
      y_lo = ypts(i) - 0.2d0*x_err(i)/ratio
      y_hi = ypts(i) + 0.2d0*x_err(i)/ratio
      call grmova( x_lo, y_lo )
      call grlina( x_lo, y_hi )
      call grmova( x_hi, y_lo )
      call grlina( x_hi, y_hi )
      ! adding the error bars along the y-axis
      ! 1) vertical part
      y_lo = ypts(i) - y_err(i)
      call grmova( xpts(i), y_lo )
      y_hi = ypts(i) + y_err(i)
      call grlina( xpts(i), y_hi )
      ! 2) the ticks
      x_lo = xpts(i) - 0.2d0*y_err(i)*ratio
      x_hi = xpts(i) + 0.2d0*y_err(i)*ratio
      call grmova( x_lo, y_lo )
      call grlina( x_hi, y_lo )
      call grmova( x_lo, y_hi )
      call grlina( x_hi, y_hi )
   end do

   call pgebuf()

end subroutine pgpt_errorbarxy
