! Primitive routine to draw a Polyline with x-error bars at each points.
! A polyline is one or more connected straight-line segments. The polyline
! is drawn using the current setting of attributes color-index, line-style,
! and line-width. The polyline is clipped at the edge of the window.
!
! Arguments:
!  N      (input)  : number of points defining the line; the line
!                    consists of (N-1) straight-line segments.
!                    N should be greater than 1 (if it is 1 or less,
!                    nothing will be drawn).
!  XPTS   (input)  : world x-coordinates of the points.
!  YPTS   (input)  : world y-coordinates of the points.
!  X_ERR, Y_ERR  (input)  : world x- errors.
!
! The dimension of arrays X and Y must be greater than or equal to N.
! The "pen position" is changed to (X(N),Y(N)) in world coordinates
! (if N > 1).
!--
!  5-Sep-2017 - Creation [EC].
! 26-Aug-2020 - Fix the length of the ticks [EC].
!-----------------------------------------------------------------------

subroutine PGLINE_ERRORBARX( n, xpts, ypts, x_err )

   integer :: n
   double precision :: xpts(*), ypts(*), x_err(*)
   !------ API end ------

   integer :: i

   logical :: move_ok
   logical, external :: grisnan
   double precision :: x_lo, x_hi, y_lo, y_hi, ratio

   if( n < 2 ) return

   call pgbbuf()

   ratio = pgyscl(pgid) / pgxscl(pgid)

   move_ok = .false.
   do i = 1, n
      if( grisnan(xpts(i)) .or. grisnan(ypts(i)) ) then
         move_ok = .false.
      else
         if( move_ok ) then
            call grlina( xpts(i), ypts(i) )
         else
            call grmova( xpts(i), ypts(i) )
            move_ok = .true.
         end if
         ! adding the error bars along the x-axis
         ! 1) horizontal part
         x_lo = xpts(i) - x_err(i)
         call grmova(x_lo,ypts(i))
         x_hi = xpts(i) + x_err(i)
         call grlina(x_hi,ypts(i))
         ! 2) the ticks
         y_lo = ypts(i) - 0.2*x_err(i)/ratio
         y_hi = ypts(i) + 0.2*x_err(i)/ratio
         call grmova( x_lo, y_lo )
         call grlina( x_lo, y_hi )
         call grmova( x_hi, y_lo )
         call grlina( x_hi, y_hi )
         ! important for next line
         call grmova( xpts(i), ypts(i) )
      end if
   end do

   call pgebuf()

end subroutine pgline_errorbarx
!_______________________________________________________________________
!
subroutine pgline_errorbary( n, xpts, ypts, y_err )

   integer :: n
   double precision :: xpts(*), ypts(*), y_err(*)
   !------ API end ------

   integer :: i

   logical :: move_ok
   logical, external :: grisnan
   double precision :: x_lo, x_hi, y_lo, y_hi, ratio

   if( n < 2 ) return

   call pgbbuf()

   ratio = pgyscl(pgid) / pgxscl(pgid)

   move_ok = .false.
   do i = 1, n
      if( grisnan(xpts(i)) .or. grisnan(ypts(i)) ) then
         move_ok = .false.
      else
         if( move_ok ) then
            call grlina( xpts(i), ypts(i) )
         else
            call grmova( xpts(i), ypts(i) )
            move_ok = .true.
         end if
         ! adding the error bars along the y-axis
         ! 1) vertical part
         y_lo = ypts(i) - y_err(i)
         call grmova( xpts(i), y_lo )
         y_hi = ypts(i) + y_err(i)
         call grlina( xpts(i), y_hi )
         ! 2) the ticks
         x_lo = xpts(i) - 0.2*y_err(i)*ratio
         x_hi = xpts(i) + 0.2*y_err(i)*ratio
         call grmova( x_lo, y_lo )
         call grlina( x_hi, y_lo )
         call grmova( x_lo, y_hi )
         call grlina( x_hi, y_hi )
         ! important for next line
         call grmova( xpts(i), ypts(i) )
      end if
   end do

   call pgebuf()

end subroutine pgline_errorbary
!_______________________________________________________________________
!
subroutine pgline_errorbarxy( n, xpts, ypts, x_err, y_err )

   integer :: n
   double precision :: xpts(*), ypts(*), x_err(*), y_err(*)
   !------ API end ------

   integer :: i

   logical :: move_ok
   logical, external :: grisnan
   double precision :: x_lo, x_hi, y_lo, y_hi, ratio

   if( n < 2 ) return

   call pgbbuf()

   ratio = pgyscl(pgid) / pgxscl(pgid)

   move_ok = .false.
   do i = 1, n
      if( grisnan(xpts(i)) .or. grisnan(ypts(i)) ) then
         move_ok = .false.
      else
         if( move_ok ) then
            call grlina( xpts(i), ypts(i) )
         else
            call grmova( xpts(i), ypts(i) )
            move_ok = .true.
         end if
         ! adding the error bars along the x-axis
         ! 1) horizontal part
         x_lo = xpts(i) - x_err(i)
         call grmova( x_lo, ypts(i) )
         x_hi = xpts(i) + x_err(i)
         call grlina( x_hi, ypts(i) )
         ! 2) the ticks
         y_lo = ypts(i) - 0.2*x_err(i)/ratio
         y_hi = ypts(i) + 0.2*x_err(i)/ratio
         call grmova( x_lo, y_lo )
         call grlina( x_lo, y_hi )
         call grmova( x_hi, y_lo )
         call grlina( x_hi, y_hi )
         ! adding the error bars along the y-axis
         ! 1) vertical part
         y_lo = ypts(i) - y_err(i)
         call grmova( xpts(i), y_lo )
         y_hi = ypts(i) + y_err(i)
         call grlina( xpts(i), y_hi )
         ! 2) the ticks
         x_lo = xpts(i) - 0.2*y_err(i)*ratio
         x_hi = xpts(i) + 0.2*y_err(i)*ratio
         call grmova( x_lo, y_lo )
         call grlina( x_hi, y_lo )
         call grmova( x_lo, y_hi )
         call grlina( x_hi, y_hi )
         ! important for next line
         call grmova( xpts(i), ypts(i) )
      end if
   end do

   call pgebuf()

end subroutine pgline_errorbarxy
