! PGCNSC_EC -- Auxiliary routine used by PGCONT_EC and PGCONTXY_EC (modified EC)

subroutine PGCNSC_EC( z, mx, my, z0, order )

   integer,          intent(in) :: mx, my, order
   double precision, intent(in) :: z(mx,*)
   double precision, intent(in) :: z0

   ! MFPLOT (internal routine): Draw a single contour.
   ! This routine is called by PGCONT_EC.
   !
   ! Arguments:
   !
   ! Z      : the array of function values.
   ! MX, MY : actual declared dimension of Z(*,*).
   ! Z0     : the contour level sought.
   !
   !--
   ! 17-Sep-1989 - Completely rewritten [TJP]. The algorithm is my own,
   !               but it is probably not original. It could probably be
   !               coded more briefly, if not as clearly.
   !  1-May-1994 - Modified to draw contours anticlockwise about maxima,
   !               to prevent contours at different levels from
   !               crossing in degenerate cells [TJP].
   !  6-Jun-2006 - Convert the statement function 'XRANGE' in an internal
   !               procedure (after 'contains') [valid only in F90]
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !  7-Oct-2021 - Removed the 'plot' argument (routine), according the
   !               new redesign of the way of computing the contours from
   !               FGL (The Muesli Graphic Library).
   ! 13-Dec-2021 - Removed ranges for index I and J of Z. Anyway, to contour
   !               a sub-array, the calling program may use slices indices.
   !-----------------------------------------------------------------------
   integer, parameter :: up = 1, down = 2, left = 3, right = 4

   logical :: flags(mx,my,2)

   integer :: i, j, dir
   double precision :: z1, z2, z3

   !------ end of declarations -- execution starts hereafter  ------

   ! Initialize the flags. The first flag for a gridpoint is set if the
   ! contour crosses the line segment to the right of the gridpoint
   ! (joining [I,J] to [I+1,J]); the second flag is set if it crosses the
   ! line segment above the gridpoint (joining [I,J] to [I,J+1]). The top
   ! and right edges require special treatment. (For purposes of description
   ! only, we assume I increases horizontally to the right and J increases
   ! vertically upwards.)
   do i = 1, mx
      do j = 1, my
         z1 = z(i,j)
         flags(i,j,1) = .false.
         flags(i,j,2) = .false.
         if( i < mx ) then
            z2 = z(i+1,j)
            if( xrange(z0,z1,z2) ) flags(i,j,1) = .true.
         end if
         if( j < my ) then
            z3 = z(i,j+1)
            if( xrange(z0,z1,z3) ) flags(i,j,2) = .true.
         end if
      end do
   end do

   ! Search the edges of the array for the start of an unclosed contour.
   ! Note that (if the algorithm is implemented correctly) all unclosed
   ! contours must begin and end at the edge of the array. When one is
   ! found, call PGCN01_EC to draw the contour, telling it the correct
   ! starting direction so that it follows the contour into the array
   ! instead of out of it. A contour is only started if the higher
   ! ground lies to the left: this is to enforce the direction convention
   ! that contours are drawn anticlockwise around maxima. If the high
   ! ground lies to the right, we will find the other end of the contour
   ! and start there.

   ! Bottom edge.
   j = 1
   do i = 1, mx-1
      if( flags(i,j,1) .and. (z(i,j)>z(i+1,j)) ) then
          call pgcn01_ec( z, mx, my, z0, flags, i, j, up, order )
      end if
   end do

   ! Right edge.
   i = mx
   do j = 1, my-1
      if( flags(i,j,2) .and. (z(i,j)>z(i,j+1)) ) then
          call pgcn01_ec( z, mx, my, z0, flags, i, j, left, order )
      end if
   end do

   ! Top edge.
   j = my
   do i = mx-1, 1, -1
      if( flags(i,j,1) .and. (z(i+1,j)>z(i,j)) ) then
          call pgcn01_ec( z, mx, my, z0, flags, i, j, down, order )
      end if
   end do

   ! Left edge.
   i = 1
   do j = my-1, 1, -1
      if( flags(i,j,2)  .and. (z(i,j+1)>z(i,j)) ) then
          call pgcn01_ec( z, mx, my, z0, flags, i, j, right, order )
      end if
   end do

   ! Now search the interior of the array for a crossing point, which will
   ! lie on a closed contour (because all unclosed contours have been
   ! eliminated). It is sufficient to search just the horizontal crossings
   ! (or the vertical ones); any closed contour must cross a horizontal
   ! and a vertical gridline. PGCN01 assumes that when it cannot proceed
   ! any further, it has reached the end of a closed contour. Thus all
   ! unclosed contours must be eliminated first.
   do i = 2, mx-1
      do j = 2, my-1
         if( flags(i,j,1) ) then
            if( z(i+1,j) > z(i,j) ) then
               dir = down
            else
               dir = up
            end if
            call pgcn01_ec( z, mx, my, z0, flags, i, j, dir, order )
         end if
      end do
   end do

   ! We didn't find any more crossing points: we've finished.

   contains

      logical function xrange( p, p1, p2 )

         ! The function XRANGE decides whether a contour at level P
         ! crosses the line between two gridpoints with values P1 and P2.
         ! It is important that a contour cannot cross a line with equal
         ! endpoints.

         double precision, intent(in) :: p, p1, p2
         xrange = ( p > min(p1,p2) ) .and. ( p <= max(p1,p2) )          &
                    .and. ( p1 /= p2 )
      end function

end subroutine
