! PGBAND_DYN -- read cursor position, and change cursor dynamically
! PGBAND_DYN_QUAD -- idem PGBAND_DYN, for a quadrilateral polygon
! PGBAND_DYN_PLUS_EXCLUDE -- idem PGBAND_DYN, with an exclusion zone
! PGBAND_DYN_MULTISPOT -- idem PGBAND_DYN, using a multispot zone
!
! idem PGBAND but without POSN and MODE(=0)
!
integer function PGBAND_DYN( cursor, xmin, xmax, ymin, ymax, x, y, ch,  &
                             axis_scale_x, axis_scale_y )

   integer :: cursor
   double precision :: xmin, xmax, ymin, ymax, x, y
   character(len=*) :: ch
   integer, optional :: axis_scale_x, axis_scale_y
   !------ API end ------

   ! This modified version of PGBAND presents the cursor dynamically.
   !
   ! Arguments:
   !  CURSOR (input)  : specify the cursor shape (see 'mod_win_db' module)
   !  XMIN, XMAX, YMIN, YMAX (input) : the BBOX world-coordinates inside
   !                    which the cursor shape must change.
   !  X      (output) : the world x-coordinate of the cursor.
   !  Y      (output) : the world y-coordinate of the cursor.
   !  CH     (output) : the character typed by the user; some predefined
   !                    letters are returned when mouse buttons are used.
   !  AXIS_SCALE_X, AXIS_SCALE_Y : 1=lin, 2=log
   !
   ! Returns:
   !  PGBAND_DYN      : 1 if the call was successful; 0 if the device
   !                    has no cursor or some other error occurs.
   !                    Be aware that the routine returns only if an event
   !                    (button click or key pressed) has been got inside
   !                    the required BBOX
   !
   !--
   ! 20-Sep-2008 - Creation from PGBAND [EC].
   !--------------------------------------------------------------------

   double precision :: rbuf(1)
   integer :: ibuf(4), lchr
   character(len=16) :: chr

   integer :: imin, imax, jmin, jmax, i, j
   integer :: px_win_height, px_xoff, px_yoff
   !------ end of declarations -- execution starts hereafter  ------

   if( cursor < 0 .or. MF_MAX_CURSOR_INDEX < cursor ) then
      call grwarn( "Invalid CURSOR argument in PGBAND_DYN" )
   end if

   ! The following offsets: {px_xoff, px_yoff} correspond to the 1/4"
   ! margins of the X11 window (see 'geom.xmargin' in x11_driver.c).

   px_xoff = nint(0.25d0*pgxpin(pgid))
   px_yoff = nint(0.25d0*pgypin(pgid))
   call grexec( grgtyp, GET_WIN_SIZE_PIX, rbuf, ibuf, chr, lchr )
   px_win_height = ibuf(4)
   imin = nint(pgxorg(pgid) + xmin*pgxscl(pgid)) + px_xoff
   imax = nint(pgxorg(pgid) + xmax*pgxscl(pgid)) + px_xoff
   jmax = px_win_height                                                 &
          - ( nint(pgyorg(pgid) + ymin*pgyscl(pgid)) + px_yoff )
   jmin = px_win_height                                                 &
          - ( nint(pgyorg(pgid) + ymax*pgyscl(pgid)) + px_yoff )
   if( present(axis_scale_x) ) then
      pgband_dyn = grcurs_dyn( cursor, imin, imax, jmin, jmax, i, j, ch, &
                        pgxorg(pgid), pgyorg(pgid), pgxscl(pgid), pgyscl(pgid),  &
                        axis_scale_x, axis_scale_y )
   else
      pgband_dyn = grcurs_dyn( cursor, imin, imax, jmin, jmax, i, j, ch, &
                        pgxorg(pgid), pgyorg(pgid), pgxscl(pgid), pgyscl(pgid) )
   end if
   x = (i - pgxorg(pgid))/pgxscl(pgid)
   y = (j - pgyorg(pgid))/pgyscl(pgid)

end function pgband_dyn
!_______________________________________________________________________
!
integer function PGBAND_DYN_QUAD( cursor, xbox, ybox, x, y, ch )

   integer :: cursor
   double precision :: xbox(4), ybox(4), x, y
   character(len=*) :: ch
   !------ API end ------

   ! This modified version of PGBAND presents the cursor dynamically.
   !
   ! Arguments:
   !  CURSOR (input)  : specify the cursor shape (see 'mod_win_db' module)
   !  XBOX(4), YBOX(4) (input) : the world-coordinates of the quadrilateral,
   !                    inside which the cursor shape must change.
   !  X      (output) : the world x-coordinate of the cursor.
   !  Y      (output) : the world y-coordinate of the cursor.
   !  CH     (output) : the character typed by the user; some predefined
   !                    letters are returned when mouse buttons are used.
   !
   ! Returns:
   !  PGBAND_DYN_QUAD : 1 if the call was successful; 0 if the device
   !                    has no cursor or some other error occurs.
   !                    Be aware that the routine returns only if an event
   !                    (button click or key pressed) has been get inside
   !                    the required BBOX
   !
   !--
   ! 20-Feb-2021 - Creation from PGBAND_DYN [EC].
   !--------------------------------------------------------------------

   double precision :: rbuf(1)
   integer :: ibuf(4), lchr
   character(len=16) :: chr

   integer :: ibox(4), jbox(4), i, j
   integer :: px_win_height, px_xoff, px_yoff
   !------ end of declarations -- execution starts hereafter  ------

   if( cursor < 0 .or. MF_MAX_CURSOR_INDEX < cursor ) then
      call grwarn( "Invalid CURSOR argument in PGBAND_DYN_QUAD" )
   end if

   ! The following offsets: {px_xoff, px_yoff} correspond to the 1/4"
   ! margins of the X11 window (see 'geom.xmargin' in xwdriv.c).

   px_xoff = nint(0.25d0*pgxpin(pgid))
   px_yoff = nint(0.25d0*pgypin(pgid))
   call grexec( grgtyp, GET_WIN_SIZE_PIX, rbuf, ibuf, chr, lchr )
   px_win_height = ibuf(4)
   ibox(:) = nint(pgxorg(pgid) + xbox(:)*pgxscl(pgid)) + px_xoff
   jbox(:) = px_win_height                                              &
          - ( nint(pgyorg(pgid) + ybox(:)*pgyscl(pgid)) + px_yoff )
   pgband_dyn_quad = grcurs_dyn_quad( cursor, ibox, jbox, i, j, ch )
   x = (i - pgxorg(pgid))/pgxscl(pgid)
   y = (j - pgyorg(pgid))/pgyscl(pgid)

end function pgband_dyn_quad
!_______________________________________________________________________
!
integer function PGBAND_DYN_PLUS_EXCLUDE( cursor,                       &
                            xmin, xmax, ymin, ymax,                     &
                            not_imin, not_imax, not_jmin, not_jmax,     &
                            x, y, ch, axis_scale_x, axis_scale_y )

   integer,          intent(in)  :: cursor
   double precision, intent(in)  :: xmin, xmax, ymin, ymax
   integer,          intent(in)  :: not_imin, not_imax, not_jmin, not_jmax
   double precision, intent(out) :: x, y
   character(len=*), intent(out) :: ch
   integer, optional :: axis_scale_x, axis_scale_y
   !------ API end ------

   ! This modified version of PGBAND presents the cursor dynamically.
   !
   ! Arguments:
   !  CURSOR (input)  : specify the cursor shape (see 'mod_win_db' module)
   !  XMIN, XMAX, YMIN, YMAX (input) : the BBOX (world-coordinates) inside
   !                    which the cursor shape must change.
   !  NOT_IMIN, NOT_IMAX,
   !  NOT_JMIN, NOT_JMAX (input) : additional BBOX (device coordinates)
   !                    for which an exclusion is required.
   !  X      (output) : the world x-coordinate of the cursor.
   !  Y      (output) : the world y-coordinate of the cursor.
   !  CH     (output) : the character typed by the user; some predefined
   !                    letters are returned when mouse buttons are used.
   !  AXIS_SCALE_X, AXIS_SCALE_Y : 1=lin, 2=log
   !
   ! Returns:
   !  PGBAND_DYN      : 1 if the call was successful; 0 if the device
   !                    has no cursor or some other error occurs.
   !                    Be aware that the routine returns only if an event
   !                    (button click or key pressed) has been get inside
   !                    the required BBOX
   !
   !--
   ! 20-Sep-2008 - Creation from PGBAND.
   ! 20-Jun-2016 - Creation from PGBAND_DYN: take into consideration an
   !               exclusion box.
   ! 31-Mar-2020 - The pointer must keep its usual shape over a legend box:
   !               change arguments type for the additional BBOX.
   !  6-Jul-2021 - Added four parameter for calling pgband_dyn_plus_exclude.
   !--------------------------------------------------------------------

   double precision :: rbuf(1)
   integer :: ibuf(4), lchr
   character(len=16) :: chr

   integer :: imin, imax, jmin, jmax, i, j
   integer :: px_win_height, px_xoff, px_yoff
   !------ end of declarations -- execution starts hereafter  ------

   if( cursor < 0 .or. MF_MAX_CURSOR_INDEX < cursor ) then
      call grwarn( "Invalid CURSOR argument in PGBAND_DYN_PLUS_EXCLUDE" )
   end if

   ! The following offsets: {px_xoff, px_yoff} correspond to the 1/4"
   ! margins of the X11 window (see 'geom.xmargin' in xwdriv.c).

   px_xoff = nint(0.25d0*pgxpin(pgid))
   px_yoff = nint(0.25d0*pgypin(pgid))
   call grexec( grgtyp, GET_WIN_SIZE_PIX, rbuf, ibuf, chr, lchr )
   px_win_height = ibuf(4)
   imin = nint(pgxorg(pgid) + xmin*pgxscl(pgid)) + px_xoff
   imax = nint(pgxorg(pgid) + xmax*pgxscl(pgid)) + px_xoff
   jmax = px_win_height                                                 &
          - ( nint(pgyorg(pgid) + ymin*pgyscl(pgid)) + px_yoff )
   jmin = px_win_height                                                 &
          - ( nint(pgyorg(pgid) + ymax*pgyscl(pgid)) + px_yoff )
   if( present(axis_scale_x) ) then
      pgband_dyn_plus_exclude = grcurs_dyn_plus_exclude( cursor,        &
               imin, imax, jmin, jmax,                                  &
               not_imin, not_imax, not_jmin, not_jmax,                  &
               i, j, ch,                                                &
               pgxorg(pgid), pgyorg(pgid), pgxscl(pgid), pgyscl(pgid),  &
               axis_scale_x, axis_scale_y )
   else
      pgband_dyn_plus_exclude = grcurs_dyn_plus_exclude( cursor,        &
               imin, imax, jmin, jmax,                                  &
               not_imin, not_imax, not_jmin, not_jmax,                  &
               i, j, ch,                                                &
               pgxorg(pgid), pgyorg(pgid), pgxscl(pgid), pgyscl(pgid) )
   end if
   x = (i - pgxorg(pgid))/pgxscl(pgid)
   y = (j - pgyorg(pgid))/pgyscl(pgid)

end function pgband_dyn_plus_exclude
!_______________________________________________________________________
!
integer function PGBAND_DYN_MULTISPOT( cursor,                          &
                                       xspot, yspot, nbspot, sel_spot,  &
                                       x, y, ch )

   integer :: cursor, nbspot, sel_spot
   double precision :: xspot(*), yspot(*), x, y
   character(len=*) :: ch
   !------ API end ------

   ! This modified version of PGBAND presents the cursor dynamically.
   !
   ! Arguments:
   !  CURSOR (input)  : specify the cursor shape (see 'mod_win_db' module)
   !  XSPOT, YSPOT (input) : the multispot zones coords inside
   !                    which the cursor shape must change.
   !  NBSPOT (input)  : the number of zones.
   !  X      (output) : the world x-coordinate of the cursor.
   !  Y      (output) : the world y-coordinate of the cursor.
   !  CH     (output) : the character typed by the user; some predefined
   !                    letters are returned when mouse buttons are used.
   !
   ! Returns:
   !  PGBAND_DYN_MULTISPOT: 1 if the call was successful; 0 if the device
   !                    has no cursor or some other error occurs.
   !                    Be aware that the routine returns only if an event
   !                    (button click or key pressed) has been get inside
   !                    the required BBOX
   !
   !--
   ! 14-Sep-2018 - new routine [É. Canot] (comes from PGBAND_DYN)
   !--------------------------------------------------------------------

   double precision :: rbuf(1)
   integer :: ibuf(4), lchr
   character(len=16) :: chr

   integer :: ispot(5), jspot(5), i, j, k
   integer :: px_win_height, px_xoff, px_yoff
   !------ end of declarations -- execution starts hereafter  ------

   if( cursor < 0 .or. MF_MAX_CURSOR_INDEX < cursor ) then
      call grwarn( "Invalid CURSOR argument in PGBAND_DYN_MULTISPOT" )
   end if

   if( nbspot > 5 ) then
      call grwarn( "Invalid NBSPOT argument in PGBAND_DYN_MULTISPOT" )
      print "(A,I0,A)", "  -> nbspot must not be larger than 5! (actual value is ", &
                        nbspot, ")"
   end if

   ! The following offsets: {px_xoff, px_yoff} correspond to the 1/4"
   ! margins of the X11 window (see 'geom.xmargin' in xwdriv.c).

   px_xoff = nint(0.25d0*pgxpin(pgid))
   px_yoff = nint(0.25d0*pgypin(pgid))
   call grexec( grgtyp, GET_WIN_SIZE_PIX, rbuf, ibuf, chr, lchr )
   px_win_height = ibuf(4)
   do k = 1, nbspot
      ispot(k) = nint(pgxorg(pgid) + xspot(k)*pgxscl(pgid)) + px_xoff
      jspot(k) = px_win_height                                          &
             - ( nint(pgyorg(pgid) + yspot(k)*pgyscl(pgid)) + px_yoff )
   end do
   pgband_dyn_multispot = grcurs_dyn_multispot( cursor,                 &
                                ispot, jspot, nbspot, sel_spot, i, j, ch )
   x = (i - pgxorg(pgid))/pgxscl(pgid)
   y = (j - pgyorg(pgid))/pgyscl(pgid)

end function pgband_dyn_multispot
