! PG_STORE_CONTOUR -- Auxiliary routine used by PGCONT (quadrangular cells)
! PG_STORE_TRICONTOUR -- Auxiliary routine used by PGTRICONTOUR (triangular cells)

subroutine PG_STORE_CONTOUR( k, i, j, x, y, z, s )

   integer,          intent(in)           :: k, i, j
   double precision, intent(in)           :: x, y
   double precision, intent(in), optional :: z
   integer,          intent(in), optional :: s

   ! MFPLOT (internal routine): Store one contour segment (for use by
   ! PGCNSC_EC, via PGCN01_EC).
   !
   ! Arguments:
   !
   ! K    : K=0, start of a new continuous contour line
   !        K=1, continuation of the current contour line
   ! I, J : Cell concerned is [i,i+1], [j,j+1]
   ! X    : X-intermediate coordinate of end point.
   ! Y    : Y-intermediate coordinate of end point.
   ! Z    : contour level (referenced only if K=0)
   ! S    : side numbering (1 to 4)
   !
   ! (7-Oct-2021)
   !-----------------------------------------------------------------------

   ! Les coordonnées des points (ainsi que des indices associés) des contours
   ! sont stockés dans les tableaux suivants
   !      XY_cont(2,:)    de type real
   !      IJ_cont(2,:)    de   "  integer
   !      IS_cont(:)      de   "  integer (mais uniquement si 's' est présent)
   ! tous trois de taille variable (extensible au fur et à mesure qu'on
   ! ajoute des points, cf. extend_XY_cont).
   ! --------------
   ! Si order=1 (segment = straight line), alors un seul point (x,y) est
   !    ajouté pour chaque cellule;
   ! si order=2 (segment = Quadratic Bézier), alors deux points (x,y) sont
   !    ajoutés pour chaque cellule.
   integer :: n_points
   logical :: side_present

   !------ end of declarations -- execution starts hereafter  ------

   if( present(s) ) then
      side_present = .true.
   else
      side_present = .false.
   end if

!!print "(A)", "(PG_STORE_CONTOUR:)   ---- new contour ----"
   if( k == 0 ) then
      if( .not. allocated(XY_cont) ) then
         XY_cont_size = 100
         allocate( XY_cont(2,XY_cont_size) )
         allocate( IJ_cont(2,XY_cont_size) )
         if( present(s) ) then
            allocate( IS_cont(XY_cont_size) )
         end if
#ifndef _OPTIM
         XY_cont = 0
         IJ_cont = 0
         if( present(s) ) then
            IS_cont = 0
         end if
#endif
         XY_cont_current_pos = 0
      else
         ! on finalise le contour courant
         n_points = XY_cont_current_pos - XY_cont_current_beg
         XY_cont(2,XY_cont_current_beg) = n_points
         XY_contour_finalized = .true.
      end if
      ! new contour
      XY_cont_nb_cont = XY_cont_nb_cont + 1
      ! adding the current level in the first line (the number of points
      ! will be added later on, in the second line)
      XY_cont_current_beg = XY_cont_current_pos + 1
      XY_cont_current_pos = XY_cont_current_beg
      if( XY_cont_current_beg > XY_cont_size ) then
         call extend_XY_cont()
      end if
      XY_cont(1,XY_cont_current_beg) = z
      XY_contour_finalized = .false. ! nb of points is not yet known...
   end if

!!print "(A,G0.4,1X,G0.4)",    "(PG_STORE_CONTOUR:)    (x,y) = ", xx, yy

!### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ###
! check duplicated points
if( XY_cont(1,XY_cont_current_pos) == x .and.                           &
    XY_cont(2,XY_cont_current_pos) == y ) then
!!   print *, "pg_store_contour: duplicated point detected (coords)."
   if( IJ_cont(1,XY_cont_current_pos) == i .and.                        &
       IJ_cont(2,XY_cont_current_pos) == j ) then
!!      print *, "                    (and same IJ_cont)"
   end if
!!pause "for dubugging purpose..."
   return ! test... pas sûr que ça marche !
end if
!### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ###
   XY_cont_current_pos = XY_cont_current_pos + 1
   if( XY_cont_current_pos > XY_cont_size ) then
      call extend_XY_cont()
   end if
   XY_cont(1,XY_cont_current_pos) = x
   XY_cont(2,XY_cont_current_pos) = y
   IJ_cont(1,XY_cont_current_pos) = i
   IJ_cont(2,XY_cont_current_pos) = j
   if( present(s) ) then
      IS_cont(  XY_cont_current_pos) = s
   end if

contains !--------------------------------------------------------------

   subroutine extend_XY_cont()

      integer :: new_size
      double precision, allocatable :: tmp(:,:)
      integer, allocatable :: itmp(:,:), itmp2(:)

      new_size = 2*XY_cont_size

      allocate( tmp(2,new_size) )
      allocate( itmp(2,new_size) )
      if( side_present ) then
         allocate( itmp2(new_size) )
      end if
#ifndef _OPTIM
      tmp = 0
      itmp = 0
      if( side_present ) then
         itmp2 = 0
      end if
#endif

      tmp(1:2,1:XY_cont_size)  = XY_cont(1:2,1:XY_cont_size)
      itmp(1:2,1:XY_cont_size) = IJ_cont(1:2,1:XY_cont_size)
      if( side_present ) then
         itmp2(1:XY_cont_size) = IS_cont(1:XY_cont_size)
      end if

      deallocate( XY_cont )
      allocate( XY_cont(2,new_size) )

      deallocate( IJ_cont )
      allocate( IJ_cont(2,new_size) )

      if( side_present ) then
         deallocate( IS_cont)
         allocate( IS_cont(new_size) )
      end if
#ifndef _OPTIM
      XY_cont = 0
      IJ_cont = 0
      if( side_present ) then
         IS_cont = 0
      end if
#endif

      XY_cont(1:2,1:XY_cont_size) = tmp(1:2,1:XY_cont_size)
      IJ_cont(1:2,1:XY_cont_size) = itmp(1:2,1:XY_cont_size)
      if( side_present ) then
         IS_cont(1:XY_cont_size) = itmp2(1:XY_cont_size)
      end if
      XY_cont_size = new_size

   end subroutine extend_XY_cont

end subroutine PG_STORE_CONTOUR
!_______________________________________________________________________
!
subroutine PG_STORE_TRICONTOUR( k, x, y, z, face )

   integer,          intent(in)           :: k
   double precision, intent(in)           :: x, y
   double precision, intent(in), optional :: z
   integer,          intent(in), optional :: face

   ! MFPLOT (internal routine): Store one contour segment (for use by
   ! PGTRICNSC_EC).
   !
   ! Arguments:
   !
   ! K (integer): K=0, start of a new continuous contour line
   !              K=1, continuation of the current contour line
   !
   ! X (real): X-intermediate coordinate of end point.
   ! Y (real): Y-intermediate coordinate of end point.
   ! Z (real): contour level (referenced only if K=0)
   !
   ! FACE (integer) : face numbering (global) in the mesh. This face
   !                  contains the point (x,y).
   !
   ! (18-Mar-2024)
   !-----------------------------------------------------------------------

   ! Les coordonnées des points des contours sont stockés dans le tableau
   !      XY_cont(2,:)    de type real
   ! de taille variable (extensible au fur et à mesure qu'on ajoute des
   ! points, cf. extend_XY_cont).
   ! --------------
   ! Si order=1 (segment = straight line), alors un seul point (x,y) est
   !    ajouté pour chaque cellule;
   ! si order=2 (segment = Quadratic Bézier), alors deux points (x,y) sont
   !    ajoutés pour chaque cellule.
   integer :: n_points
   integer, save :: first_face
   integer       :: last_face

   !------ end of declarations -- execution starts hereafter  ------

   if( k == 0 ) then
      if( .not. allocated(XY_cont) ) then
         XY_cont_size = 100
         allocate( XY_cont(2,XY_cont_size) )
#ifndef _OPTIM
         XY_cont = 0
#endif
         XY_cont_current_pos = 0
      else
         ! si besoin, on finalise le contour courant
         if( .not. XY_contour_finalized ) then
            n_points = XY_cont_current_pos - XY_cont_current_beg
!!print "(A,I0,A,I0)", "  finaliz. of the prev. cont.: npt = ", n_points, &
!!                     " added in XY_cont at pos. ", XY_cont_current_beg
            XY_cont(2,XY_cont_current_beg) = n_points
            XY_contour_finalized = .true.
         end if
      end if
      ! new contour
      XY_cont_nb_cont = XY_cont_nb_cont + 1
      ! adding the current level in the first line (the number of points
      ! will be added later on, in the second line)
      XY_cont_current_beg = XY_cont_current_pos + 1
      XY_cont_current_pos = XY_cont_current_beg
      if( XY_cont_current_beg > XY_cont_size ) then
         call extend_XY_cont()
      end if
!!print "(A,ES10.3,A,I0)", "  pg_store_tricontour: level = ", z, &
!!                         " added in XY_cont at pos. ", XY_cont_current_beg
      XY_cont(1,XY_cont_current_beg) = z
      XY_contour_finalized = .false. ! nb of points is not yet known...
   end if

!!print "(A,G0.4,1X,G0.4)",    "(PG_STORE_TRICONTOUR:)    (x,y) = ", xx, yy

   XY_cont_current_pos = XY_cont_current_pos + 1
   if( XY_cont_current_pos > XY_cont_size ) then
      call extend_XY_cont()
   end if
   XY_cont(1,XY_cont_current_pos) = x
   XY_cont(2,XY_cont_current_pos) = y

contains
!_______________________________________________________________________
!
   subroutine extend_XY_cont()

      integer :: new_size
      double precision, allocatable :: tmp(:,:)

      new_size = 2*XY_cont_size

      allocate( tmp(2,new_size) )
#ifndef _OPTIM
      tmp = 0
#endif

      tmp(1:2,1:XY_cont_size) = XY_cont(1:2,1:XY_cont_size)

      deallocate( XY_cont )
      allocate( XY_cont(2,new_size) )

#ifndef _OPTIM
      XY_cont = 0
#endif

      XY_cont(1:2,1:XY_cont_size) = tmp(1:2,1:XY_cont_size)
      XY_cont_size = new_size

   end subroutine extend_XY_cont
!_______________________________________________________________________
!
end subroutine PG_STORE_TRICONTOUR
