! Keep this file encoded in ISO-8859-15 (Latin-1). We have to send special
! UTF-8 chars using two chars (e.g. \oe, \OE, ...)

subroutine grtext_xft( orient, x0, y0, string, clipping )

   double precision, intent(in)            :: orient, x0, y0
   character(len=*), intent(in)            :: string
   logical,          intent(in), optional  :: clipping
   !------ API end ------

   ! GRPCKG: Write a text string using antialiased feature of Xft.
   ! It have exactly the same arguments (and same behaviour) as GRTEXT.
   ! The text may be windowed in the current viewport, or may extend over
   ! the whole view surface, according to the value of clipping.
   ! Among line attributes, only color applies to the text. The current
   ! pen position after a call to this routine is undefined.
   !
   ! Implements the same features as 'grparse_ps_font'.
   !
   ! Arguments:
   !
   !     ORIENT (in) : angle, in degrees, that the baseline is to make
   !                   with the horizontal, increasing counter-clockwise
   !                   (0.0 is horizontal).
   !     X0, Y0 (in) : device coordinates. The string is drawn from the
   !                   X0 abscissa, in such a way that the baseline of all
   !                   characters passes through point (X0,Y0).
   !     STRING (in) : the character string to be plotted. This may
   !                   include standard escape-sequences to represent
   !                   non-ASCII characters and special commands. The
   !                   number of characters in STRING (i.e., LEN(STRING))
   !                   should not exceed 256.
   !  CLIPPING (out) : boolean value specifying if clipping must be done.
   !--
   ! 14-Aug-2018 - Creation. Add GREXEC support only for X11 (OPCODE=31).
   !               Only the normal font is currently supported (Helvetica),
   !               and color is always black. Escaped sequences in the
   !               string (for sub- or superscript, change of font, etc.)
   !               are not yet implemented [EC].
   ! 16-Aug-2018 - support of clipping. Add GREXEC support only for X11,
   !               OPCODE=32, to send the viewport device coordinates to
   !               xwdriv [EC].
   ! 20-Aug-2018 - support of vertical position modification, via upward
   !               and downward escaped sequences [EC].
   ! 14-Apr-2021 - Take care of integer overflow in the X11 driver
   !               (In Xlib, XPoint coordinates are 'short int').
   !-----------------------------------------------------------------------

   character(len=48), parameter :: greek = "ABGDEZHQIKLMNXOPRSTUFCYW"   &
                                        // "abgdezhqiklmnxoprstufcyw"

   character(len=8), parameter :: greek_names(52) =                     &
                [ "Alpha   ", "Beta    ", "Gamma   ", "Delta   ",       &
                  "Epsilon ", "Zeta    ", "Eta     ", "Theta   ",       &
                  "Iota    ", "Kappa   ", "Lambda  ", "Mu      ",       &
                  "Nu      ", "Xi      ", "Omicron ", "Pi      ",       &
                  "Rho     ", "Sigma   ", "Tau     ", "Upsilon ",       &
                  "Phi     ", "Chi     ", "Psi     ", "Omega   ",       &
                  "alpha   ", "beta    ", "gamma   ", "delta   ",       &
                  "epsilon ", "zeta    ", "eta     ", "theta   ",       &
                  "iota    ", "kappa   ", "lambda  ", "mu      ",       &
                  "nu      ", "xi      ", "omicron ", "pi      ",       &
                  "rho     ", "sigma   ", "tau     ", "upsilon ",       &
                  "varphi  ", "chi     ", "psi     ", "omega   ",       &
                  "vartheta", "varpi   ", "varsigma", "phi     " ]

   integer, parameter :: greek_ind(24) =                                &
                [ 65, 66, 71, 68, 69, 90, 72, 81, 73, 75, 76, 77,       &
                  78, 88, 79, 80, 82, 83, 84, 85, 70, 67, 89, 87 ]

   ! current position for writing a subpart of the string
   double precision :: x, y, dx, dy, width
   double precision :: FNTFAC, RFNTBAS
   double precision :: angle, factor, font_size
   logical :: clipped, found
   integer :: dash_length

   character :: c
   integer :: IFNTLV, i, j, k, n, buf_start, buf_len
   character :: fontcode, old_fontcode

   ! memory of char widths (for cumulative backspace)
   ! width_mem:  successive widths for each character drawn
   double precision :: width_mem(256)
   integer :: i_mem, i_back

   character(len=11) :: guess

   double precision :: rbuf(1)
   integer :: ibuf(5), lchr
   character(len=256) :: chr

   !------ end of declarations -- execution starts hereafter  ------

   ! Check that there is something to be plotted.
   if( len(string) == 0 ) return

   ! Check that a device is selected.
   if( grcide < 1 ) then
      call grwarn('GRTEXT - no graphics device is active.')
      return
   end if

   if( present(clipping) ) then
      clipped = clipping
   else
      clipped = .false.
   end if

   ! Before parsing the string, and sending it by parts to xwdriv, constant
   ! parameters of the write (angle and clipping) have to be sent.
   angle = -orient*deg_to_rad ! angle in radian
   rbuf(1) = angle
   if( clipped ) then
      ibuf(1) = 1
      ! Prepare the clipping by sending the (TL,BR) viewport's corners
      ! (yes, it is different from (BL,TR) corners already registered,
      !  because X11 direction of the Y-axis is downward!).
      !
      ! Overflow in integers may lead to a change of sign; in such a case
      ! some drawn lines may appear as wrong. Device coordinates must be
      ! restricted to short integers in Xlib (see the 'restrict_to_short'
      ! function in the MOD_GRPLOT module).
      ibuf(2) = restrict_to_short( grxmin(grcide) )
      ibuf(3) = restrict_to_short( grymax(grcide) )
      ibuf(4) = restrict_to_short( grxmax(grcide) )
      ibuf(5) = restrict_to_short( grymin(grcide) )
   else
      ibuf(1) = 0
   end if
   call grexec( grgtyp, SET_ANG_CLIP_XFT, rbuf, ibuf, chr, lchr )

   ! Compute base scaling and orientation.
   ! (assuming ratio=1, i.e. the X11 device has the same resolution
   ! in both direction -- pixels are squared)
   factor = grcfac(grcide)/2.5d0 ! old value of GRTEXT for Hershey font
   ! the multiplicative coefficient below has been determined to obtain
   ! the same character dimension (width and height) as in EPS and PDF.
   font_size = factor * 22.9d0

   !###############################################################
   ! Parse and draw the string
   !###############################################################

   ! Trailing blanks are allowed, therefore don't use len_trim()
   n = len(string)
   buf_len = 0

   ! Current position
   x = x0
   y = y0

   ! Misc. initialization
   IFNTLV = 0
   FNTFAC = 1.0d0
   i_mem = 0
   i_back = 0

   ! Default font
   fontcode = "n"

   i = 1
   do

      if( i >= n+1 ) then
         call flush_buffer()
         exit
      end if

      if( string(i:i) == "-" ) then
         ! Hyphen
         dash_length = 1
         i = i + 1
         if( gr_minus_sign_math_mode ) then
            ! For writing numerical labels of axes, a minus in math mode
            ! is systematically used...
            dash_length = 2
         end if
         call flush_buffer()
         if( dash_length == 1 ) then
            ! Le tiret "hyphen" doit tre crit en StandardEncoding, sinon
            ! il est trs long (comme "minus" en mode math). Pour avoir un
            ! "minus" en mode math, utiliser "\-".
            call ps_font_char_width( 45, fontcode, width )
            width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
            i_mem = i_mem + 1
            width_mem(i_mem) = width
            i_back = i_mem
            call send_few_chars( "-", 1, UTF8=.false. ) ! ( (C 45)
         else ! dash_length == 2
            ! "minus" en mode math, ou "endash"
            call ps_font_char_width( 29, fontcode, width )
            width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
            i_mem = i_mem + 1
            width_mem(i_mem) = width
            i_back = i_mem
            call send_few_chars( "–", 3, UTF8=.true. ) ! endash (C 29)
         end if
         cycle
      else if( string(i:i) == "\" ) then
         call flush_buffer()
         if( i+2 <= n ) then
            select case( string(i+1:i+2) )
               case( "fn" )
                  fontcode = string(i+2:i+2)
                  i = i + 3
                  cycle
               case( "fN" )
                  fontcode = string(i+2:i+2)
                  i = i + 3
                  cycle
               case( "fi" )
                  fontcode = string(i+2:i+2)
                  i = i + 3
                  cycle
               case( "fI" )
                  fontcode = string(i+2:i+2)
                  i = i + 3
                  cycle
               case( "fr" )
                  fontcode = string(i+2:i+2)
                  i = i + 3
                  cycle
               case( "fR" )
                  fontcode = string(i+2:i+2)
                  i = i + 3
                  cycle
               case( "fs" )
                  fontcode = string(i+2:i+2)
                  i = i + 3
                  cycle
               case( "OE" )
                  call ps_font_char_width( 128, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "Œ", 2, UTF8=.true. )
                  i = i + 3
                  cycle
               case( "oe" )
                  call ps_font_char_width( 136, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "œ", 2, UTF8=.true. )
                  i = i + 3
                  cycle
               case( "s1" )
                  ! Standard space
                  ! Get width of standard space
                  call ps_font_char_width( 32, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  dx = width
                  call rmoveto( dx, 0.0d0 )
                  i_mem = i_mem + 1
                  width_mem(i_mem) = dx
                  i_back = i_mem
                  i = i + 3
                  cycle
               case( "s2" )
                  ! Thin space (half of standard one)
                  ! Get width of standard space
                  call ps_font_char_width( 32, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  dx = width/2.0d0
                  call rmoveto( dx, 0.0d0 )
                  i_mem = i_mem + 1
                  width_mem(i_mem) = dx
                  i_back = i_mem
                  i = i + 3
                  cycle
               case( "s3" )
                  ! Very thin space (one fourth of standard one)
                  ! Get width of standard space
                  call ps_font_char_width( 32, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  dx = width/4.0d0
                  call rmoveto( dx, 0.0d0 )
                  i_mem = i_mem + 1
                  width_mem(i_mem) = dx
                  i_back = i_mem
                  i = i + 3
                  cycle
               case default
                  ! Do nothing
            end select
         end if
         if( i+1 <= n ) then
            select case( string(i+1:i+1) )
               case( "u" )
                  IFNTLV = IFNTLV + 1
                  RFNTBAS = + 0.65d0*FNTFAC
                  dy = RFNTBAS*ps_char_height
                  call rmoveto( 0.0d0, dy )
                  FNTFAC = GR_SUBSCRIPT_SCALING**ABS(IFNTLV)
                  i = i + 2
                  cycle
               case( "d" )
                  IFNTLV = IFNTLV - 1
                  FNTFAC = GR_SUBSCRIPT_SCALING**ABS(IFNTLV)
                  RFNTBAS = - 0.65d0*FNTFAC
                  dy = RFNTBAS*ps_char_height
                  call rmoveto( 0.0d0, dy )
                  i = i + 2
                  cycle
               case( "b" )
                  ! Go backspace -- width of the previous character(s)
                  dx = width_mem(i_back)
                  call rmoveto( -dx, 0.0d0 )
                  i_back = i_back - 1
                  i = i + 2
                  cycle
               case( "(" )
                  ! Explicit name for a greek letter
                  found = .false.
                  ! Searching end of letter
                  do k = i+2, n
                     if( string(k:k) == ")" ) then
                        found = .true.
                        exit
                     end if
                  end do
                  if( .not. found ) then
                     print *, 'error: ")" not found in: "', string(i+2:n), '"'
                     return
                  end if
                  ! Decode greek letter from string(i+2:k-1)
                  found = .false.
                  do j = 1, size(greek_names)
                     if( string(i+2:k-1) == greek_names(j) ) then
                        found = .true.
                        exit
                     end if
                  end do
                  if( .not. found ) then
                     print *, 'error: greek name: "', string(i+2:k-1),  &
                              '" not found in table.'
                     return
                  end if
                  if( j <= 24 ) then
                     ! First part:  upper case
                     c = char( greek_ind(j) )
                  else if( j >= 49 ) then
                     if( j == 49 ) then
                        c = "J"
                     else if( j == 50 ) then
                        c = "v"
                     else if( j == 51 ) then
                        c = "V"
                     else if( j == 52 ) then
                        c = "f"
                     end if
                  else
                     ! Second part: lower case
                     c = char( greek_ind(j-24)+32 )
                  end if
                  old_fontcode = fontcode
                  fontcode = "g"
                  call ps_font_char_width( ichar(c), fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( c, 1, UTF8=.false. )
                  fontcode = old_fontcode
                  i = k + 1
                  cycle
               case( "g" )
                  !  Short escaped sequence for a greek letter
                  if( i+2 > n ) then
                     call PrintMessage( "grtext", "W",                           &
                                        "interpreting your string: '" // string // "'", &
                                        "got an error at: '" // string(i:) // "'", &
                                        "found '\g' so a greek equivalent letter is expected." )
                     return
                  end if
                  c = string(i+2:i+2)
                  old_fontcode = fontcode
                  fontcode = "g"
                  call ps_font_char_width( ichar(c), fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  buf_start = i+2
                  buf_len = 1
                  call flush_buffer()
                  fontcode = old_fontcode
                  i = i + 3
                  cycle
               case( "A" )
                  call ps_font_char_width( 197, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "Å", 2, UTF8=.true. ) ! Aring (C 197)
                  i = i + 2
                  cycle
               case( "x" )
                  call ps_font_char_width( 215, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "×", 2, UTF8=.true. ) ! multiply (C 215)
                  i = i + 2
                  cycle
               case( "." )
                  call ps_font_char_width( 183, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "·", 2, UTF8=.true. ) ! periodcentered (C 183)
                  i = i + 2
                  cycle
               case( "1" )
                  call ps_font_char_width( 185, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "¹", 2, UTF8=.true. ) ! onesuperior (C 185)
                  i = i + 2
                  cycle
               case( "2" )
                  call ps_font_char_width( 178, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "²", 2, UTF8=.true. ) ! twosuperior (C 178)
                  i = i + 2
                  cycle
               case( "3" )
                  call ps_font_char_width( 179, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "³", 2, UTF8=.true. ) ! threesuperior (C 179)
                  i = i + 2
                  cycle
               case( "\" )
                  call ps_font_char_width( 92, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "\", 1, UTF8=.false. ) ! backslash (C 92)
                  i = i + 2
                  cycle
               case( "-" )
                  ! "minus" en mode math, ou "endash"
                  call ps_font_char_width( 29, fontcode, width )
                  width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
                  i_mem = i_mem + 1
                  width_mem(i_mem) = width
                  i_back = i_mem
                  call send_few_chars( "–", 3, UTF8=.true. ) ! endash (C 29)
                  i = i + 2
                  cycle
               case default
                  call guess_greek_letter( string(i:), guess )
                  if( len_trim(guess) > 0 ) then
                     call PrintMessage( "grtext", "W",                  &
                                        "interpreting your string: '" // string // "'", &
                                        "got an error at: '" // string(i:) // "'", &
                                        "found one backslash but don't understand the following characters: '"//string(i+1:i+2)//"'", &
                                        "Did you thought about '" // trim(guess) // "'?" )
                  else
                     call PrintMessage( "grtext", "W",                  &
                                        "interpreting your string: '" // string // "'", &
                                        "got an error at: '" // string(i:) // "'", &
                                        "found one backslash but don't understand the following characters: '"//string(i+1:i+2)//"'" )
                  end if
                  i = i + 3
                  cycle
            end select
         else
            call PrintMessage( "grtext", "W",                           &
                               "interpreting your string: '" // string // "'", &
                               "got an error at: '" // string(i:) // "'", &
                               "found one backslash but nothing after...", &
                               "Did you thought about '\\'?" )
            return
         end if
      else

         call ps_font_char_width( ichar(string(i:i)), fontcode, width )
!### TODO: Warning: le facteur 1.03 (empirique) ci-dessous, sert  avoir
!          le bon espacement entre les groupes de lettres, par comparaison
!          avec la chane crite d'un seul coup !
!### TODO: mais cela dpend de la taille de la fentre X11 (j'ai test)
!          =========>  corriger !
         width = width * ps_font_size*fontmul(fontcode)*FNTFAC * 1.03d0
         i_mem = i_mem + 1
         width_mem(i_mem) = width
         i_back = i_mem

         ! Fill buffer
         if( buf_len == 0 ) then
            buf_start = i
         end if
         buf_len = buf_len + 1

         i = i + 1
      end if

   end do

contains
!_______________________________________________________________________
!
   subroutine flush_buffer( )

      character(len=80) :: fontname
      integer :: string_utf8, i

      if( buf_len > 0 ) then

         write(fontname,"(F7.2)") font_size*fontmul(fontcode)*FNTFAC
         i = index( fontname, "*" )
         if(i /= 0 ) then
            call grwarn('(Muesli FGL:) grtext_xft: marker scale too large!')
            print *, "  -> unexpected results may occur..."
         end if
         fontname = adjustl(fontname)
         select case( fontcode )
            case( "n" )
               fontname = "Helvetica-" // fontname
            case( "N" )
               fontname = "Helvetica-" // trim(fontname) // ":Bold"
            case( "i" )
               fontname = "Times-" // trim(fontname) // ":Italic"
            case( "I" )
               fontname = "Times-" // trim(fontname) // ":Bold:Italic"
            case( "r" )
               fontname = "Times-" // fontname
            case( "R" )
               fontname = "Times-" // trim(fontname) // ":Bold"
            case( "g" )
               fontname = "Symbol-" // fontname
            case( "s" )
               fontname = "English 157-" // fontname
            case default
               print *, "(Muesli FGL:) Internal error: unknown fontcode"
               return
         end select

         ! Send the font (name and attributes)
         chr = trim(fontname) // char(0)
         lchr = len_trim(fontname) + 1
         ibuf(1) = 0 ! ISO-8859-1, not UTF-8
         call grexec( grgtyp, SET_FONT_XFT, rbuf, ibuf, chr, lchr )

         ! Send a part of the string and its position
         chr = string(buf_start:buf_start+buf_len-1) // char(0)
         lchr = buf_len + 1
         ! Overflow in integers may lead to a change of sign; in such a case
         ! some drawn lines may appear as wrong. Device coordinates must be
         ! restricted to [?32,767, +32,767], which is the range for the XPoint
         ! coordinates in Xlib. See also the 'restrict_to_short' function in
         ! the MOD_GRPLOT module.
         ibuf(1) = restrict_to_short(x)
         ibuf(2) = restrict_to_short(y)
         call grexec( grgtyp, DRAW_CHAR_STRING, rbuf, ibuf, chr, lchr )

         ! Next position: cumulate last widths
         dx = sum( width_mem(i_mem-(buf_len-1):i_mem) )
         call rmoveto( dx, 0.0d0 )

         buf_len = 0

      end if

   end subroutine flush_buffer
!_______________________________________________________________________
!
   subroutine send_few_chars( s, s_len, utf8 )

      character(len=*), intent(in)           :: s
      integer,          intent(in)           :: s_len
      logical,          intent(in), optional :: utf8
      !------ API end ------

      ! default: utf8=0 (=> ISO-8859-1)

      character(len=80) :: fontname
      integer :: string_utf8, i

      if( present(utf8) ) then
         if( utf8 ) then
            string_utf8 = 1
         else
            string_utf8 = 0
         end if
      else
         string_utf8 = 0
      end if

      write(fontname,"(F7.2)") font_size*fontmul(fontcode)*FNTFAC
      i = index( fontname, "*" )
      if(i /= 0 ) then
         call grwarn('(Muesli FGL:) grtext_xft: marker scale too large!')
         print *, "  -> unexpected results may occur..."
      end if
      fontname = adjustl(fontname)
      select case( fontcode )
         case( "n" )
            fontname = "Helvetica-" // fontname
         case( "N" )
            fontname = "Helvetica-" // trim(fontname) // ":Bold"
         case( "i" )
            fontname = "Times-" // trim(fontname) // ":Italic"
         case( "I" )
            fontname = "Times-" // trim(fontname) // ":Bold:Italic"
         case( "r" )
            fontname = "Times-" // fontname
         case( "R" )
            fontname = "Times-" // trim(fontname) // ":Bold"
         case( "g" )
            fontname = "Symbol-" // fontname
         case( "s" )
            fontname = "English 157-" // fontname
      end select

      ! Send the font (name and attributes)
      chr = trim(fontname) // char(0)
      lchr = len_trim(fontname) + 1
      ibuf(1) = string_utf8
      call grexec( grgtyp, SET_FONT_XFT, rbuf, ibuf, chr, lchr )

      ! Send the character and its position
      chr = trim(s) // char(0)
      lchr = s_len + 1
      ! Overflow in integers may lead to a change of sign; in such a case
      ! some drawn lines may appear as wrong. Device coordinates must be
      ! restricted to [?32,767, +32,767], which is the range for the XPoint
      ! coordinates in Xlib. See also the 'restrict_to_short' function in
      ! the MOD_GRPLOT module.
      ibuf(1) = restrict_to_short(x)
      ibuf(2) = restrict_to_short(y)
      call grexec( grgtyp, DRAW_CHAR_STRING, rbuf, ibuf, chr, lchr )

      ! Next position: cumulate last widths
      dx = sum( width_mem(i_mem:i_mem) )
      call rmoveto( dx, 0.0d0 )

   end subroutine send_few_chars
!_______________________________________________________________________
!
   subroutine rmoveto( dx, dy )

      double precision, intent(in) :: dx, dy
      !------ API end ------

      x = x + dx*cos(-angle) - dy*sin(-angle)
      y = y + dx*sin(-angle) + dy*cos(-angle)

   end subroutine rmoveto
!_______________________________________________________________________
!
   double precision function fontmul( fontcode )

      character, intent(in) :: fontcode
      !------ API end ------

      ! Additional multiplicative factor the the font size, in order
      ! to obtain approximately the same heights for letters (both
      ! in lower- or uppercase).
      !
      ! See: 'Relative_sizes.eps'

      if( fontcode == "n" .or. fontcode == "N" ) then
         fontmul = 0.85d0
      else if( fontcode == "i" .or. fontcode == "I" ) then
         fontmul = 1.00d0
      else if( fontcode == "r" .or. fontcode == "R" ) then
         fontmul = 1.00d0
      else if( fontcode == "g" ) then
         fontmul = 0.95d0
      else if( fontcode == "s" ) then
         fontmul = 1.33d0
      end if

   end function fontmul
!_______________________________________________________________________
!
   subroutine guess_greek_letter( string, guess )
      character(len=*), intent(in)  :: string
      character(len=11)             :: guess

      ! We try to find the origin of an error. Perhaps the user type
      ! '\alpha' instead of '\(alpha)' ? Same for other greek letters...

      integer :: i, n

      guess = ""

      if( string(1:1) /= "\" ) return

      n = len_trim(string)

      do i = 1, 52
         if( string(2:n) == greek_names(i)(1:n-1) ) then
            guess = "\(" // trim(greek_names(i)) // ")"
            return
         end if
      end do

   end subroutine guess_greek_letter
!_______________________________________________________________________
!
end subroutine
