subroutine grtext_ps_font( orient, x0, y0, string )

   double precision, intent(in)           :: orient, x0, y0
   character(len=*), intent(in)           :: string

   ! *** called only during the EPS printing ***

   ! GRPCKG: Write a text string using a PostScript font.
   ! It have exactly the same arguments (and same behaviour) as GRTEXT.
   ! The text may be windowed in the current viewport, or may extend over
   ! the whole view surface, according to the value of clipping (defined
   ! at a higher level). Line attributes (color, intensity thickness)
   ! apply to text, but line-style is ignored. The current pen position
   ! after a call to this routine is undefined.
   !
   ! Arguments:
   !
   ! ORIENT (input): angle, in degrees, that the baseline is to make
   !                 with the horizontal, increasing counter-
   !                 clockwise (0.0 is horizontal).
   ! X, Y   (input): device coordinates. The string is drawn from the
   !                 X abscissa, in such a way that the baseline of
   !                 all characters passing through point (X,Y).
   ! STRING (input): the character string to be plotted. This may
   !                 include standard escape-sequences to represent
   !                 non-ASCII characters and special commands. The max
   !                 number of characters in STRING should not exceed
   !                 256.
   !--
   ! 13-Jul-2010 - Creation from GRTEXT.
   ! 25-Dec-2013 - Fix format to write 'orient' in string (F6.2 instead of
   !               F4.2). String is not trimmed when displayed (useful in
   !               PGCONL to add both a leading and a trailing blank to the
   !               label).
   ! 16-Apr-2020 - Modified to take into account suppression of panels.
   ! 17-Feb-2021 - Fix the bbox computation when clipping in "on".
   ! 18-Mar-2021 - Remove clipping processing (it is now done at a higher
   !               level).
   !--------------------------------------------------------------------

   character(len=132) :: inline

   double precision :: width
   double precision :: h_min, h_max
   double precision :: xbox(4), ybox(4), bbox(4), cosa, sina

   character(len=10) :: str1

   !------ end of declarations -- execution starts hereafter  ------

   ! Check that there is something to be plotted.
   if( len_trim(string) == 0 ) return

   ! Check that a device is selected.
   if( grcide < 1 ) then
      call grwarn( "GRTEXT - no graphics device is active." )
      return
   end if

   ! move to the required location
   write( inline, "(I0,1X,I0,1X,A)" ) nint(x0), nint(y0), "m"
   call gresc( trim(inline) )

   ! fix: 6 characters are required to write negative angles!
   !      (now, use F7.2 instead of F5.2)
   if( orient /= 0.0d0 ) then
      call rm_trail_zeros_in_flt(  orient, "(F7.2)", str1 )
      inline = "q " // trim(str1) // " rotate"
      call gresc( trim(inline) )
   end if

   ! draw the string and compute the BBox
   call grparse_ps_font( string, width, h_min, h_max, draw=.true. )

   if( orient /= 0.0d0 ) then
      call gresc( "Q" )
      LAST_COLOR_IS_VALID = .false.
      LAST_LINEWIDTH_IS_VALID = .false.
      LAST_LINESTYLE_IS_VALID = .false.
      LAST_FONT_ATTRIB_IS_VALID = .false.
   end if

   bbox(1) = 0.0d0
   bbox(2) = h_min
   bbox(3) = width
   bbox(4) = h_max
   if( orient /= 0.0d0 ) then
      ! Rotate bounding box.
      cosa = cos(orient*deg_to_rad)
      sina = sin(orient*deg_to_rad)
      xbox(1) = x0 + (cosa*bbox(1) - sina*bbox(2))
      ybox(1) = y0 + (sina*bbox(1) + cosa*bbox(2))
      xbox(2) = x0 + (cosa*bbox(1) - sina*bbox(4))
      ybox(2) = y0 + (sina*bbox(1) + cosa*bbox(4))
      xbox(3) = x0 + (cosa*bbox(3) - sina*bbox(4))
      ybox(3) = y0 + (sina*bbox(3) + cosa*bbox(4))
      xbox(4) = x0 + (cosa*bbox(3) - sina*bbox(2))
      ybox(4) = y0 + (sina*bbox(3) + cosa*bbox(2))
   else ! orient == 0
      xbox(1) = x0 + bbox(1)
      ybox(1) = y0 + bbox(2)
      xbox(2) = x0 + bbox(1)
      ybox(2) = y0 + bbox(4)
      xbox(3) = x0 + bbox(3)
      ybox(3) = y0 + bbox(4)
      xbox(4) = x0 + bbox(3)
      ybox(4) = y0 + bbox(2)
   end if

end subroutine
