subroutine grtext_pdf_font( orient, x0, y0, string )

   double precision :: orient, x0, y0
   character(len=*) :: string

   ! *** called only during the PDF printing ***

   ! GRPCKG: Write a text string using a PostScript font, in a PDF file.
   ! It have exactly the same arguments (and same behaviour) as GRTEXT.
   ! The text may be windowed in the current viewport, or may extend over
   ! the whole view surface, according to the value of clipping. Line
   ! attributes (color, intensity thickness) apply to text, but line-style
   ! is ignored. The current pen position after a call to this routine is
   ! undefined.
   !
   ! Arguments:
   !
   ! ORIENT (input): angle, in degrees, that the baseline is to make
   !                 with the horizontal, increasing counter-
   !                 clockwise (0.0 is horizontal).
   ! X, Y   (input): device coordinates. The string is drawn from the
   !                 X abscissa, in such a way that the baseline of
   !                 all characters passing through point (X,Y).
   ! STRING (input): the character string to be plotted. This may
   !                 include standard escape-sequences to represent
   !                 non-ASCII characters and special commands. The max
   !                 number of characters in STRING should not exceed
   !                 256.
   !--
   ! 02-Jul-2018 - Creation from grtext_ps_font [EC].
   ! 16-Apr-2020 - Modified to take into account suppression of panels [EC].
   ! 17-Feb-2021 - Fix the bbox computation when clipping in "on" [EC].
   ! 21-Mar-2021 - Remove clipping processing (it is now done at a higher
   !               level).
   !--------------------------------------------------------------------

   character(len=132) :: inline

   double precision :: width
   double precision :: h_min, h_max
   double precision :: xbox(4), ybox(4), bbox(4), cosa, sina

   character(len=10) :: str1, str2, str3, str4

   !------ end of declarations -- execution starts hereafter  ------

   ! Check that there is something to be plotted.
   if( len_trim(string) == 0 ) return

   ! Check that a device is selected.
   if( grcide < 1 ) then
      call grwarn( "GRTEXT - no graphics device is active." )
      return
   end if

   write( inline, "(A)" ) "BT" ! Begin Text marker
   call gresc( trim(inline) )

   ! move to the required location, and rotate if needed
   if( orient /= 0.0d0 ) then
      cosa = cos(orient*deg_to_rad)
      sina = sin(orient*deg_to_rad)
   else ! orient == 0
      cosa = 1.0d0
      sina = 0.0d0
   end if
   ! compact write of the following PDF command:
   ! (PDF standard says than integer or real number may be used for coordinates,
   !  but for angle we must take care an adequate resolution)
   ! cosa sina -sina cosa nint(x0) nint(y0) Tm
   write( inline, "(I0,1X,I0,A)" ) nint(x0), nint(y0), " Tm"
   call rm_trail_zeros_in_flt(  cosa, "(F6.3)", str1 )
   call rm_trail_zeros_in_flt(  sina, "(F6.3)", str2 )
   call rm_trail_zeros_in_flt( -sina, "(F6.3)", str3 )
   call rm_trail_zeros_in_flt(  cosa, "(F6.3)", str4 )
   inline = trim(adjustl(str1)) // " " // trim(adjustl(str2)) // " " // &
            trim(adjustl(str3)) // " " // trim(adjustl(str4)) // " " // inline
   call gresc( trim(inline) )

   ! draw the string and compute the BBox
   call grparse_pdf_font( string, width, h_min, h_max, .true. )

   write( inline, "(A)" ) "ET" ! End Text marker
   call gresc( trim(inline) )

   bbox(1) = 0.0d0
   bbox(2) = h_min
   bbox(3) = width
   bbox(4) = h_max
   if( orient /= 0.0d0 ) then
      ! Rotate bounding box.
      xbox(1) = x0 + (cosa*bbox(1) - sina*bbox(2))
      ybox(1) = y0 + (sina*bbox(1) + cosa*bbox(2))
      xbox(2) = x0 + (cosa*bbox(1) - sina*bbox(4))
      ybox(2) = y0 + (sina*bbox(1) + cosa*bbox(4))
      xbox(3) = x0 + (cosa*bbox(3) - sina*bbox(4))
      ybox(3) = y0 + (sina*bbox(3) + cosa*bbox(4))
      xbox(4) = x0 + (cosa*bbox(3) - sina*bbox(2))
      ybox(4) = y0 + (sina*bbox(3) + cosa*bbox(2))
   else ! orient == 0
      xbox(1) = x0 + bbox(1)
      ybox(1) = y0 + bbox(2)
      xbox(2) = x0 + bbox(1)
      ybox(2) = y0 + bbox(4)
      xbox(3) = x0 + bbox(3)
      ybox(3) = y0 + bbox(4)
      xbox(4) = x0 + bbox(3)
      ybox(4) = y0 + bbox(2)
   end if

end subroutine
