subroutine GRPARSE_PS_FONT( string, w, h_min, h_max, draw )

   character(len=*), intent(in)  :: string
   logical,          intent(in)  :: draw
   double precision, intent(out) :: w, h_min, h_max
   !------ API end ------

   ! Parses a string, computes its BBox and, optionally (if draw = TRUE),
   ! writes the appropriate commands in an EPS file.

   ! On output, gives some information about the string drawn in the EPS:
   !   w: width; h_min: min. height; h_max: max. height (heights are always
   !   defined with respect to the baseline of the characters). These values
   !   are expressed as device coordinates; h_min may be negative if the
   !   drawing is below the baseline of the string.
   !   Be aware that h_min and h_max are not necessary the actual heights
   !   of the character string, but they correspond to the BBOX of the font
   !   (as if string always contains 'p' and 'd' characters, which have
   !   greater heights than 'x').

   ! Are processed:
   !   - font changes (total nb of fonts is 8);
   !   - height-level changes (up, down);
   !   - cumulative backspaces, possibly with different fonts;
   !   - greek letters (two different ways);
   !   - oe, OE (ligatures) and other special characters defined by
   !     escaped sequences.

   ! Must be distinguished:
   !   - the simple hyphen: "-"
   !   - and the long dash, like the minus sign in math mode: "\-"

   !********************************************************************
   !*** WARNING: this routine may be called also during a PDF print! ***
   !***          (but only with draw=.false.)                        ***
   !********************************************************************

   integer :: IFNTLV, i, j, k, n, buf_start, buf_len
   logical :: found
   double precision :: FNTFAC, FNTBAS, RFNTBAS
   character(len=256) :: inline
   character(len=32) :: fontname
   integer :: fontsize

   ! these variables must be saved
   character(len=32), save :: old_fontname
   integer,           save :: old_fontsize
   logical,           save :: old_encoding_ISO, old_encoding_STD

   character(len=48), parameter :: greek = "ABGDEZHQIKLMNXOPRSTUFCYW"   &
                                        // "abgdezhqiklmnxoprstufcyw"

   character(len=8), parameter :: greek_names(52) =                     &
                [ "Alpha   ", "Beta    ", "Gamma   ", "Delta   ",       &
                  "Epsilon ", "Zeta    ", "Eta     ", "Theta   ",       &
                  "Iota    ", "Kappa   ", "Lambda  ", "Mu      ",       &
                  "Nu      ", "Xi      ", "Omicron ", "Pi      ",       &
                  "Rho     ", "Sigma   ", "Tau     ", "Upsilon ",       &
                  "Phi     ", "Chi     ", "Psi     ", "Omega   ",       &
                  "alpha   ", "beta    ", "gamma   ", "delta   ",       &
                  "epsilon ", "zeta    ", "eta     ", "theta   ",       &
                  "iota    ", "kappa   ", "lambda  ", "mu      ",       &
                  "nu      ", "xi      ", "omicron ", "pi      ",       &
                  "rho     ", "sigma   ", "tau     ", "upsilon ",       &
                  "varphi  ", "chi     ", "psi     ", "omega   ",       &
                  "vartheta", "varpi   ", "varsigma", "phi     " ]

   integer, parameter :: greek_ind(24) =                                &
                [ 65, 66, 71, 68, 69, 90, 72, 81, 73, 75, 76, 77,       &
                  78, 88, 79, 80, 82, 83, 84, 85, 70, 67, 89, 87 ]

   character :: c
   double precision :: width, w_pos, h1, h2
   integer :: dash_length

   ! PS_Std_Fonts_presence(1:8) is initialized in print_eps()

   ! memory of chars (for cumulative backspace)
   ! back_mem:  successive character codes which have been drawn
   ! back_font: successive fontname which have been used
   !            n, N, r, R, i, I, s, g (N, R, and I are bold fonts)
   integer :: back_mem(256)
   character :: back_font(256)
   integer :: i_back

   !------ end of declarations -- execution starts hereafter  ------

   ! trailing blanks are allowed, therefore don't use len_trim()
   n = len(string)

   ! misc. initialization
   IFNTLV = 0
   FNTFAC = 1.0
   FNTBAS = 0.0
   buf_len = 0
   i = 1
   fontname = "/Helvetica"
   PS_Std_Fonts_presence(1) = .true.
   back_mem = 0
   back_font = ""
   i_back = 0

   ! initialize old values only once
   ! (at EPS driver initialization, EPS_driver_font_begin is set to TRUE)
   if( EPS_driver_font_begin ) then
      old_fontname = ""
      old_fontsize = -1
      old_encoding_ISO = .false.
      old_encoding_STD = .false.
      EPS_driver_font_begin = .false.
   end if

   ! because of write forward and move backward, two variables
   ! are necessary:
   !    - w_pos is the current right position during writing
   !    - w is the maximum value
   w_pos = 0.0
   w = 0.0
   h_min =  1.0d+30
   h_max = -1.0d+30

   ! before processing the string, we have to check if the "script" font
   ! (English157BT) has to be embed in the EPS file... but not in PDF
   if( PRINTING_EPS ) then
      if( index( string, "\fs" ) /= 0 ) then
         if( .not. EnglishBT_embedded ) then
            call embed_EnglishBT_PFA()
            EnglishBT_embedded = .true.
         end if
      end if
   end if

   ! required anyway if string begins by an escaped sequence
   if( draw ) call insert_ISO_FMSR()

   do

      if( i >= n+1 ) then
         if( draw ) call flush_buffer()
         exit
      end if

      if( string(i:i) == "-" ) then
         ! hyphen
         dash_length = 1
         i = i + 1
         if( gr_minus_sign_math_mode ) then
            ! for writing numerical labels of axes, a minus in math mode
            ! is systematically used...
            dash_length = 2
         end if
         if( draw ) call flush_buffer()
         if( dash_length == 1 ) then
            ! A "hyphen" must use StandardEncoding, else it will be very
            ! long (as "minus" in math mode). To get a "minus" in math mode,
            ! use "\-" instead.
            call ps_font_char_width( 45, encode_font(), width, h1, h2 )
            width = width * ps_font_size*fontmul()*FNTFAC
            w_pos = w_pos + width
            w = max( w, w_pos )
            h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
            h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
            if( draw ) then
               call insert_STD_FMSR()
               call gresc( "(-) show" ) ! ( (C 45)
            end if
            i_back = i_back + 1
            back_mem(i_back) = 45
            back_font(i_back) = encode_font()
         else ! dash_length == 2
            ! "minus" in math mode, or "endash"
            call ps_font_char_width( 29, encode_font(), width, h1, h2 )
            width = width * ps_font_size*fontmul()*FNTFAC
            w_pos = w_pos + width
            w = max( w, w_pos )
            h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
            h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
            if( draw ) then
               call insert_ISO_FMSR()
               call gresc( "(-) show" ) ! endash (C 29)
            end if
            i_back = i_back + 1
            back_mem(i_back) = 29
            back_font(i_back) = encode_font()
         end if
         cycle
      else if( string(i:i) == "(" ) then
         ! special character in PostScript language
         if( draw ) call flush_buffer()
         call ps_font_char_width( 40, encode_font(), width, h1, h2 )
         width = width * ps_font_size*fontmul()*FNTFAC
         w_pos = w_pos + width
         w = max( w, w_pos )
         h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         if( draw ) then
            call insert_ISO_FMSR()
            ! insert octal value
            call gresc( "(\050) show" ) ! ( (C 40)
         end if
         i_back = i_back + 1
         back_mem(i_back) = 40
         back_font(i_back) = encode_font()
         i = i + 1
         cycle
      else if( string(i:i) == ")" ) then
         ! special character in PostScript language
         if( draw ) call flush_buffer()
         call ps_font_char_width( 41, encode_font(), width, h1, h2 )
         width = width * ps_font_size*fontmul()*FNTFAC
         w_pos = w_pos + width
         w = max( w, w_pos )
         h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         if( draw ) then
            call insert_ISO_FMSR()
            ! insert octal value
            call gresc( "(\051) show" ) ! ) (C 41)
         end if
         i_back = i_back + 1
         back_mem(i_back) = 41
         back_font(i_back) = encode_font()
         i = i + 1
         cycle
      else if( string(i:i) == "\" ) then
         if( draw ) call flush_buffer()
         if( i+2 <= n ) then
            select case( string(i+1:i+2) )
               case( "fn" )
                  fontname = "/Helvetica"
                  PS_Std_Fonts_presence(1) = .true.
                  i = i + 3
                  cycle
               case( "fN" )
                  fontname = "/Helvetica-Bold"
                  PS_Std_Fonts_presence(2) = .true.
                  i = i + 3
                  cycle
               case( "fi" )
                  fontname = "/Times-Italic"
                  PS_Std_Fonts_presence(3) = .true.
                  i = i + 3
                  cycle
               case( "fI" )
                  fontname = "/Times-BoldItalic"
                  PS_Std_Fonts_presence(4) = .true.
                  i = i + 3
                  cycle
               case( "fr" )
                  fontname = "/Times-Roman"
                  PS_Std_Fonts_presence(5) = .true.
                  i = i + 3
                  cycle
               case( "fR" )
                  fontname = "/Times-Bold"
                  PS_Std_Fonts_presence(6) = .true.
                  i = i + 3
                  cycle
               case( "fs" )
                  fontname = "/English157BT-Regular"
                  PS_Std_Fonts_presence(8) = .true.
                  i = i + 3
                  cycle
               case( "OE" )
                  call ps_font_char_width( 128, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_STD_FMSR
                     ! insert octal value
                     call gresc( "(\352) show" ) ! OE (C 128)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 128
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case( "oe" )
                  call ps_font_char_width( 136, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_STD_FMSR
                     ! insert octal value
                     call gresc( "(\372) show" ) ! oe (C 136)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 136
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case( "s1" )
                  ! standard space
                  ! get width of standard space
                  call ps_font_char_width( 32, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_ISO_FMSR()
                     call gresc( "( ) show" ) ! space (C 32)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 32
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case( "s2" )
                  ! thin space (half of standard one)
                  ! get width of standard space
                  call ps_font_char_width( 32, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width/2.0
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_ISO_FMSR()
                     write(inline,"(I0)") nint(width/2.0)
                     call gresc( trim(inline) // " 0 rmoveto" )
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = -2
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case( "s3" )
                  ! very thin space (one fourth of standard one)
                  ! get width of standard space
                  call ps_font_char_width( 32, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width/4.0
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_ISO_FMSR()
                     write(inline,"(I0)") nint(width/4.0)
                     call gresc( trim(inline) // " 0 rmoveto" )
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = -3
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case default
                  ! do nothing
            end select
         end if
         if( i+1 <= n ) then
            select case( string(i+1:i+1) )
               case( "u" )
                  IFNTLV = IFNTLV + 1
                  FNTBAS = FNTBAS + 0.65*ps_char_height*FNTFAC
                  RFNTBAS = + 0.65*FNTFAC
                  if( draw ) then
                     write(inline,"(I0)") nint(RFNTBAS*ps_char_height)
                     call gresc( "0 " // trim(inline) // " rmoveto" )
                  end if
                  FNTFAC = GR_SUBSCRIPT_SCALING**ABS(IFNTLV)
                  i = i + 2
                  cycle
               case( "d" )
                  IFNTLV = IFNTLV - 1
                  FNTFAC = GR_SUBSCRIPT_SCALING**ABS(IFNTLV)
                  FNTBAS = FNTBAS - 0.65*ps_char_height*FNTFAC
                  RFNTBAS = - 0.65*FNTFAC
                  if( draw ) then
                     write(inline,"(I0)") nint(RFNTBAS*ps_char_height)
                     call gresc( "0 " // trim(inline) // " rmoveto" )
                  end if
                  i = i + 2
                  cycle
               case( "b" )
                  ! go backspace -- width of the previous character(s)
                  call ps_font_char_width( back_mem(i_back), back_font(i_back), &
                                           width, h1, h2 )
                  width = width * ps_font_size*fontmul(back_font(i_back))*FNTFAC
                  i_back = i_back - 1
                  w_pos = w_pos - width
                  if( draw ) then
                     write(inline,"(I0)") nint(width)
                     call gresc( "-" // trim(inline) // " 0 rmoveto" )
                  end if
                  i = i + 2
                  cycle
               case( "(" )
                  PS_Std_Fonts_presence(7) = .true.
                  if( draw ) then
                     ! selecting /Symbol font for greek letters
                     fontsize = nint(ps_font_size*fontmul()*FNTFAC)
                     write(inline,"(A,I0,A)") "/Symbol findfont ",      &
                                              fontsize, " scalefont setfont"
                     call gresc( trim(inline) )
                     old_fontname = "/Symbol"
                     old_fontsize = fontsize
                  end if
                  ! searching end of letter
                  found = .false.
                  do k = i+2, n
                     if( string(k:k) == ")" ) then
                        found = .true.
                        exit
                     end if
                  end do
                  if( .not. found ) then
                     print *, 'error: ")" not found in: "', string(i+2:n), '"'
                     return
                  end if
                  ! decode greek letter from string(i+2:k-1)
                  found = .false.
                  do j = 1, size(greek_names)
                     if( string(i+2:k-1) == greek_names(j) ) then
                        found = .true.
                        exit
                     end if
                  end do
                  if( .not. found ) then
                     print *, 'error: greek name: "', string(i+2:k-1),  &
                              '" not found in table.'
                     return
                  end if
                  if( j <= 24 ) then
                     ! first part:  upper case
                     c = char( greek_ind(j) )
                  else if( j >= 49 ) then
                     if( j == 49 ) then
                        c = "J"
                     else if( j == 50 ) then
                        c = "v"
                     else if( j == 51 ) then
                        c = "V"
                     else if( j == 52 ) then
                        c = "f"
                     end if
                  else
                     ! second part: lower case
                     c = char( greek_ind(j-24)+32 )
                  end if
                  call ps_font_char_width( ichar(c), encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     inline = "(" // c // ") show"
                     call gresc( trim(inline) )
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = ichar(c)
                  back_font(i_back) = "g"
                  i = k + 1
               case( "g" )
                  !  short escaped sequence for greek letters
                  PS_Std_Fonts_presence(7) = .true.
                  c = string(i+2:i+2)
                  call ps_font_char_width( ichar(c), encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     ! selecting /Symbol font for greek letters
                     fontsize = nint(ps_font_size*fontmul()*FNTFAC)
                     write(inline,"(A,I0,A)") "/Symbol findfont ",         &
                                              fontsize, " scalefont setfont"
                     call gresc( trim(inline) )
                     old_fontname = "/Symbol"
                     old_fontsize = fontsize
                  end if
                  if( draw ) then
                     inline = "(" // c // ") show"
                     call gresc( trim(inline) )
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = ichar(c)
                  back_font(i_back) = "g"
                  i = i + 3
                  cycle
               case( "A" )
                  call ps_font_char_width( 197, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\305) show" ) ! Aring (C 197)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 197
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "x" )
                  call ps_font_char_width( 215, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\327) show" ) ! multiply (C 215)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 215
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "." )
                  call ps_font_char_width( 183, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\267) show" ) ! periodcentered (C 183)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 183
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "1" )
                  call ps_font_char_width( 185, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\271) show" ) ! onesuperior (C 185)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 185
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "2" )
                  call ps_font_char_width( 178, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\262) show" ) ! twosuperior (C 178)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 178
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "3" )
                  call ps_font_char_width( 179, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\263) show" ) ! threesuperior (C 179)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 179
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "\" )
                  call ps_font_char_width( 92, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\134) show" ) ! backslash (C 92)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 92
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "-" )
                  ! "minus" en mode math, ou "endash"
                  call ps_font_char_width( 29, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(-) show" ) ! endash (C 29)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 29
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case default
                  call PrintMessage( "grtext", "W",                     &
                                     "parsing your string: '" // string // "'", &
                                     "got an error at: '" // string(i:) // "'" )
                  ! don't use return: if the error occurs at the beginning of
                  ! the string, then {w,h_min,h_max} contains infinite values!
                  i = i + 2
                  cycle
            end select
         else
            call PrintMessage( "grtext", "W",                           &
                               "parsing your string: '" // string // "'", &
                               "got an error at: '" // string(i:) // "'", &
                               "found one backslash but nothing after...", &
                               "did you thought about '\\'?" )
            return
         end if
      else
         call ps_font_char_width( ichar( string(i:i) ), encode_font(),  &
                                  width, h1, h2 )
         width = width * ps_font_size*fontmul()*FNTFAC
         w_pos = w_pos + width
         w = max( w, w_pos )
         h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         if( draw ) then
            ! fill buffer
            if( buf_len == 0 ) then
               buf_start = i
            end if
            buf_len = buf_len + 1
         end if
         i_back = i_back + 1
         back_mem(i_back) = ichar( string(i:i) )
         back_font(i_back) = encode_font()
         i = i + 1
      end if
   end do

   ! fix a badly scaling (on X11)
   w = w * 1.04d0

contains
!_______________________________________________________________________
!
   subroutine flush_buffer()

      if( buf_len > 0 ) then

         call insert_ISO_FMSR()

         inline = "(" // string(buf_start:buf_start+buf_len-1) // ") show"
         call gresc( trim(inline) )
         buf_len = 0

      end if

   end subroutine flush_buffer
!_______________________________________________________________________
!
   subroutine insert_STD_FMSR()

      fontsize = nint(ps_font_size*fontmul()*FNTFAC)
      if( fontname /= old_fontname .or. fontsize /= old_fontsize .or.   &
          .not. old_encoding_STD .or. .not. LAST_FONT_ATTRIB_IS_VALID ) then
         write(inline,"(A,I0,A)") trim(fontname) // " /StandardEncoding ", &
                                  fontsize, " FMSR"
         call gresc( trim(inline) )
         old_fontname = fontname
         old_fontsize = fontsize
         old_encoding_STD = .true.
         old_encoding_ISO = .false.
         LAST_FONT_ATTRIB_IS_VALID = .true.
      end if

   end subroutine insert_STD_FMSR
!_______________________________________________________________________
!
   subroutine insert_ISO_FMSR()

      fontsize = nint(ps_font_size*fontmul()*FNTFAC)
      if( fontname /= old_fontname .or. fontsize /= old_fontsize .or.   &
          .not. old_encoding_ISO .or. .not. LAST_FONT_ATTRIB_IS_VALID ) then
         write(inline,"(A,I0,A)") trim(fontname) // " /ISOLatin1Encoding ", &
                                  fontsize, " FMSR"
         call gresc( trim(inline) )
         old_fontname = fontname
         old_fontsize = fontsize
         old_encoding_STD = .false.
         old_encoding_ISO = .true.
         LAST_FONT_ATTRIB_IS_VALID = .true.
      end if

   end subroutine insert_ISO_FMSR
!_______________________________________________________________________
!
   character function encode_font()

      if( fontname == "/Helvetica" ) then
         encode_font = "n"
      else if( fontname == "/Times-Italic" ) then
         encode_font = "i"
      else if( fontname == "/Times-Roman" ) then
         encode_font = "r"
      else if( fontname == "/Symbol" ) then
         encode_font = "g"
      else if( fontname == "/Helvetica-Bold" ) then
         encode_font = "N"
      else if( fontname == "/Times-BoldItalic" ) then
         encode_font = "I"
      else if( fontname == "/Times-Bold" ) then
         encode_font = "R"
      else if( fontname == "/English157BT-Regular" ) then
         encode_font = "s"
      end if

   end function encode_font
!_______________________________________________________________________
!
   double precision function fontmul( fontcode )

      character, optional :: fontcode

      ! Additional multiplicative factor for the font size, in order to
      ! obtain approximately the same heights for letters (both in in lower-
      ! or uppercase) when using different fonts.
      !
      ! see: 'Relative_sizes.eps'

      if( present(fontcode) ) then

         if( fontcode == "n" .or. fontcode == "N" ) then
            fontmul = 0.85d0
         else if( fontcode == "i" .or. fontcode == "I" ) then
            fontmul = 1.00d0
         else if( fontcode == "r" .or. fontcode == "R" ) then
            fontmul = 1.00d0
         else if( fontcode == "g" ) then
            fontmul = 0.95d0
         else if( fontcode == "s" ) then
            fontmul = 1.33d0
         end if

      else

         if( fontname == "/Helvetica" .or. fontname == "/Helvetica-Bold" ) then
            fontmul = 0.85d0
         else if( fontname == "/Times-Italic" .or. fontname == "/Times-BoldItalic" ) then
            fontmul = 1.00d0
         else if( fontname == "/Times-Roman" .or. fontname == "/Times-Bold" ) then
            fontmul = 1.00d0
         else if( fontname == "/Symbol" ) then
            fontmul = 0.95d0
         else if( fontname == "/English157BT-Regular" ) then
            fontmul = 1.33d0
         end if

      end if

   end function fontmul
!_______________________________________________________________________
!
end subroutine
