subroutine GRPARSE_PDF_FONT( string, w, h_min, h_max, draw )

   character(len=*), intent(in)  :: string
   logical,          intent(in)  :: draw
   double precision, intent(out) :: w, h_min, h_max
   !------ API end ------

   ! Parses a string, computes its BBox and, optionally (if draw = TRUE),
   ! writes the appropriate commands in a PDF file.

   ! On output, gives some information about the string drawn in the PDF:
   !   w: width; h_min: min. height; h_max: max. height (heights are always
   !   defined with respect to the baseline of the characters).

   ! Are processed:
   !   - font changes (total nb of fonts is 8);
   !   - height-level changes (up, down);
   !   - cumulative backspaces, possibly with different fonts;
   !   - greek letters (two different ways);
   !   - oe, OE (ligatures) and other special characters defined by
   !     escaped sequences.

   ! Must be distinguished:
   !   - the simple hyphen: "-"
   !   - and the long dash, like the minus sign in math mode: "\-"

   integer :: IFNTLV, i, j, k, n, buf_start, buf_len
   logical :: found
   double precision :: FNTFAC, FNTBAS, RFNTBAS
   character(len=256) :: inline
   character(len=32) :: fontname
   integer :: fontsize

   ! these variables must be saved
   integer,           save :: old_fontsize
   character(len=32), save :: old_fontname

   character(len=48), parameter :: greek = "ABGDEZHQIKLMNXOPRSTUFCYW"   &
                                        // "abgdezhqiklmnxoprstufcyw"

   character(len=8), parameter :: greek_names(52) =                     &
                [ "Alpha   ", "Beta    ", "Gamma   ", "Delta   ",       &
                  "Epsilon ", "Zeta    ", "Eta     ", "Theta   ",       &
                  "Iota    ", "Kappa   ", "Lambda  ", "Mu      ",       &
                  "Nu      ", "Xi      ", "Omicron ", "Pi      ",       &
                  "Rho     ", "Sigma   ", "Tau     ", "Upsilon ",       &
                  "Phi     ", "Chi     ", "Psi     ", "Omega   ",       &
                  "alpha   ", "beta    ", "gamma   ", "delta   ",       &
                  "epsilon ", "zeta    ", "eta     ", "theta   ",       &
                  "iota    ", "kappa   ", "lambda  ", "mu      ",       &
                  "nu      ", "xi      ", "omicron ", "pi      ",       &
                  "rho     ", "sigma   ", "tau     ", "upsilon ",       &
                  "varphi  ", "chi     ", "psi     ", "omega   ",       &
                  "vartheta", "varpi   ", "varsigma", "phi     " ]

   integer, parameter :: greek_ind(24) =                                &
                [ 65, 66, 71, 68, 69, 90, 72, 81, 73, 75, 76, 77,       &
                  78, 88, 79, 80, 82, 83, 84, 85, 70, 67, 89, 87 ]

   character :: c
   double precision :: width, w_pos, h1, h2
   integer :: dash_length

   ! PS_Std_Fonts_presence(1:8) is initialized in print_pdf()

   ! memory of chars (for cumulative backspace)
   ! back_mem:  successive character codes which have been drawn
   ! back_font: successive fontname which have been used
   !            n, N, r, R, i, I, s, g (N, R, and I are bold fonts)
   integer :: back_mem(256)
   character :: back_font(256)
   integer :: i_back

   ! PDF standard allows balanced parenthesis in a string (only in this case,
   ! they don't need to be escaped by a backslash)
   logical :: balanced_parenthesis

   !------ end of declarations -- execution starts hereafter  ------

   ! trailing blanks are allowed, therefore don't use len_trim()
   n = len(string)

   ! check that parenthesis are balanced and that string will not be splitted
   ! by some font change
   balanced_parenthesis = check_parenthesis( )

   ! misc. initialization
   IFNTLV = 0
   FNTFAC = 1.0
   FNTBAS = 0.0
   buf_len = 0
   i = 1
   fontname = "/Helvetica"
   back_mem = 0
   back_font = ""
   i_back = 0

   ! initialize old values only once
   ! (at PDF driver initialization, PDF_driver_font_begin is set to TRUE)
   if( PDF_driver_font_begin ) then
      old_fontname = ""
      old_fontsize = -1
      PDF_driver_font_begin = .false.
   end if

   ! because of write forward and move backward, two variables
   ! are necessary:
   !    - w_pos is the current right position during writing
   !    - w is the maximum value
   w_pos = 0.0
   w = 0.0
   h_min =  1.0d+30
   h_max = -1.0d+30

   ! because perhaps a previous text already defined sub- or superscript...
   call gresc( "0 Ts" )

   ! required anyway if string begins by an escaped sequence
   if( draw ) call insert_font()

   do
      if( i >= n+1 ) then
         if( draw ) call flush_buffer()
         exit
      end if

      if( string(i:i) == "-" ) then
         ! hyphen
         dash_length = 1
         if( gr_minus_sign_math_mode ) then
            ! for writing numerical labels of axes, a minus in math mode
            ! is systematically used...
            dash_length = 2
         end if
         if( dash_length == 1 ) then
            call ps_font_char_width( ichar( string(i:i) ), encode_font(), &
                                     width, h1, h2 )
            width = width * ps_font_size*fontmul()*FNTFAC
            w_pos = w_pos + width
            w = max( w, w_pos )
            h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
            h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
            if( draw ) then
               ! fill buffer
               if( buf_len == 0 ) then
                  buf_start = i
               end if
               buf_len = buf_len + 1
            end if
            i_back = i_back + 1
            back_mem(i_back) = ichar( string(i:i) )
            back_font(i_back) = encode_font()
         else ! dash_length == 2
            ! "minus" en mode math, ou "endash"
            call ps_font_char_width( 29, encode_font(), width, h1, h2 )
            width = width * ps_font_size*fontmul()*FNTFAC
            w_pos = w_pos + width
            w = max( w, w_pos )
            h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
            h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
            if( draw ) then
               call flush_buffer()
               call insert_font()
               ! insert octal value
               call gresc( "(\226) Tj" ) ! endash (C 29)
            end if
            i_back = i_back + 1
            back_mem(i_back) = 29
            back_font(i_back) = encode_font()
         end if
         i = i + 1
         cycle
      else if( string(i:i) == "(" ) then
         ! in a PDF string, must be escaped only if unbalanced
         call ps_font_char_width( 40, encode_font(), width, h1, h2 )
         width = width * ps_font_size*fontmul()*FNTFAC
         w_pos = w_pos + width
         w = max( w, w_pos )
         h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         if( draw ) then
            if( balanced_parenthesis ) then
               ! fill buffer
               if( buf_len == 0 ) then
                  buf_start = i
               end if
               buf_len = buf_len + 1
            else
               call flush_buffer()
               call insert_font()
               ! insert escaped character
               call gresc( "(\() Tj" ) ! ( (C 40)
            end if
         end if
         i_back = i_back + 1
         back_mem(i_back) = 40
         back_font(i_back) = encode_font()
         i = i + 1
         cycle
      else if( string(i:i) == ")" ) then
         ! in a PDF string, must be escaped only if unbalanced
         call ps_font_char_width( 41, encode_font(), width, h1, h2 )
         width = width * ps_font_size*fontmul()*FNTFAC
         w_pos = w_pos + width
         w = max( w, w_pos )
         h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         if( draw ) then
            if( balanced_parenthesis ) then
               ! fill buffer
               if( buf_len == 0 ) then
                  buf_start = i
               end if
               buf_len = buf_len + 1
            else
               call flush_buffer()
               call insert_font()
               ! insert escaped character
               call gresc( "(\)) Tj" ) ! ) (C 41)
            end if
         end if
         i_back = i_back + 1
         back_mem(i_back) = 41
         back_font(i_back) = encode_font()
         i = i + 1
         cycle
      else if( string(i:i) == "\" ) then
         if( draw ) call flush_buffer()
         if( i+2 <= n ) then
            select case( string(i+1:i+2) )
               case( "fn" )
                  fontname = "/Helvetica"
                  PS_Std_Fonts_presence(1) = .true.
                  i = i + 3
                  cycle
               case( "fN" )
                  fontname = "/Helvetica-Bold"
                  PS_Std_Fonts_presence(2) = .true.
                  i = i + 3
                  cycle
               case( "fr" )
                  fontname = "/Times-Roman"
                  PS_Std_Fonts_presence(3) = .true.
                  i = i + 3
                  cycle
               case( "fR" )
                  fontname = "/Times-Bold"
                  PS_Std_Fonts_presence(4) = .true.
                  i = i + 3
                  cycle
               case( "fi" )
                  fontname = "/Times-Italic"
                  PS_Std_Fonts_presence(5) = .true.
                  i = i + 3
                  cycle
               case( "fI" )
                  fontname = "/Times-BoldItalic"
                  PS_Std_Fonts_presence(6) = .true.
                  i = i + 3
                  cycle
               case( "fs" )
                  fontname = "/English157BT-Regular"
                  PS_Std_Fonts_presence(8) = .true.
                  i = i + 3
                  cycle
               case( "OE" )
                  call ps_font_char_width( 128, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_font()
                     ! insert octal value
                     call gresc( "(\214) Tj" ) ! OE (C 128)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 128
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case( "oe" )
                  call ps_font_char_width( 136, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_font()
                     ! insert octal value
                     call gresc( "(\234) Tj" ) ! oe (C 136)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 136
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case( "s1" )
                  ! standard space
                  ! get width of standard space
                  call ps_font_char_width( 32, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_font()
                     call gresc( "( ) Tj" ) ! space (C 32)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 32
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case( "s2" )
                  ! thin space (half of standard one)
                  ! get width of standard space
                  call ps_font_char_width( 32, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width/2.0
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_font()
                     call gresc( "50 Tz ( ) Tj 100 Tz" )
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = -2
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case( "s3" )
                  ! very thin space (one fourth of standard one)
                  ! get width of standard space
                  call ps_font_char_width( 32, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width/4.0
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call insert_font()
                     call gresc( "25 Tz ( ) Tj 100 Tz" )
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = -3
                  back_font(i_back) = encode_font()
                  i = i + 3
                  cycle
               case default
                  ! do nothing
            end select
         end if
         if( i+1 <= n ) then
            select case( string(i+1:i+1) )
               case( "u" )
                  IFNTLV = IFNTLV + 1
                  FNTBAS = FNTBAS + 0.65*ps_char_height*FNTFAC
                  if( draw ) then
                     write(inline,"(I0)") nint(FNTBAS)
                     call gresc( trim(inline) // " Ts" )
                  end if
                  FNTFAC = GR_SUBSCRIPT_SCALING**ABS(IFNTLV)
                  i = i + 2
                  cycle
               case( "d" )
                  IFNTLV = IFNTLV - 1
                  FNTFAC = GR_SUBSCRIPT_SCALING**ABS(IFNTLV)
                  FNTBAS = FNTBAS - 0.65*ps_char_height*FNTFAC
                  if( draw ) then
                     write(inline,"(I0)") nint(FNTBAS)
                     call gresc( trim(inline) // " Ts" )
                  end if
                  i = i + 2
                  cycle
               case( "b" )
                  if( draw ) call flush_buffer()
                  ! go backspace -- width of the previous character(s)
                  call ps_font_char_width( back_mem(i_back), back_font(i_back), &
                                           width, h1, h2 )
                  width = width * ps_font_size*fontmul(back_font(i_back))*FNTFAC
                  w_pos = w_pos - width
                  if( draw ) then
                     call gresc( "3 Tr -100 Tz" ) ! invisible, left direction
                     if( back_mem(i_back) == -2 ) then
                        ! code for "thin space"
                        call gresc( "50 Tz ( ) Tj 100 Tz" )
                     else if( back_mem(i_back) == -3 ) then
                        ! code for "very thin space"
                        call gresc( "25 Tz ( ) Tj 100 Tz" )
                     else if( back_mem(i_back) == 40 ) then
                        ! code for "("
                        call gresc( "(\050) Tj" )
                     else if( back_mem(i_back) == 41 ) then
                        ! code for ")"
                        call gresc( "(\051) Tj" )
                     else if( back_mem(i_back) == 128 ) then
                        ! code for "OE"
                        call gresc( "(\214) Tj" )
                     else if( back_mem(i_back) == 136 ) then
                        ! code for "oe"
                        call gresc( "(\234) Tj" )
!### TODO: il manque d'autre cas particulier, mais je suppose qu'on les mets
!          trop rarement comme indices ou exposants...
                     else ! normal ichar
                        call gresc( "(" // char(back_mem(i_back)) // ") Tj" )
                     end if
                     call gresc( "100 Tz 0 Tr" ) ! visible and right direction
                  end if
                  i_back = i_back - 1
                  i = i + 2
                  cycle
               case( "(" )
                  old_fontname = fontname
                  fontname = "/Symbol"
                  PS_Std_Fonts_presence(7) = .true.
                  if( draw ) then
                     ! selecting /Symbol font for greek letters
                     write(inline,"(A,I0,A)") "/Fg ",                   &
                                  nint(ps_font_size*fontmul()*FNTFAC), " Tf"
                     call gresc( trim(inline) )
                  end if
                  ! searching end of letter
                  found = .false.
                  do k = i+2, n
                     if( string(k:k) == ")" ) then
                        found = .true.
                        exit
                     end if
                  end do
                  if( .not. found ) then
                     print *, 'error: ")" not found in: "', string(i+2:n), '"'
                     go to 99
                  end if
                  ! decode greek letter from string(i+2:k-1)
                  found = .false.
                  do j = 1, size(greek_names)
                     if( string(i+2:k-1) == greek_names(j) ) then
                        found = .true.
                        exit
                     end if
                  end do
                  if( .not. found ) then
                     print *, 'error: greek name: "', string(i+2:k-1),  &
                              '" not found in table.'
                     go to 99
                  end if
                  if( j <= 24 ) then
                     ! first part:  upper case
                     c = char( greek_ind(j) )
                  else if( j >= 49 ) then
                     if( j == 49 ) then
                        c = "J"
                     else if( j == 50 ) then
                        c = "v"
                     else if( j == 51 ) then
                        c = "V"
                     else if( j == 52 ) then
                        c = "f"
                     end if
                  else
                     ! second part: lower case
                     c = char( greek_ind(j-24)+32 )
                  end if
                  call ps_font_char_width( ichar(c), encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     inline = "(" // c // ") Tj"
                     call gresc( trim(inline) )
                  end if
                  fontname = old_fontname
                  old_fontname = ""
                  i_back = i_back + 1
                  back_mem(i_back) = ichar(c)
                  back_font(i_back) = "g"
                  i = k + 1
                  cycle
               case( "g" )
                  !  short escaped sequence for greek letters
                  old_fontname = fontname
                  fontname = "/Symbol"
                  PS_Std_Fonts_presence(7) = .true.
                  c = string(i+2:i+2)
                  call ps_font_char_width( ichar(c), encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     ! selecting /Symbol font for greek letters
                     write(inline,"(A,I0,A)") "/Fg ",                   &
                                  nint(ps_font_size*fontmul()*FNTFAC), " Tf"
                     call gresc( trim(inline) )
                  end if
                  if( draw ) then
                     inline = "(" // c // ") Tj"
                     call gresc( trim(inline) )
                  end if
                  fontname = old_fontname
                  old_fontname = ""
                  i_back = i_back + 1
                  back_mem(i_back) = ichar(c)
                  back_font(i_back) = "g"
                  i = i + 3
                  cycle
               case( "A" )
                  call ps_font_char_width( 197, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\305) Tj" ) ! Aring (C 197)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 197
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "x" )
                  call ps_font_char_width( 215, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\327) Tj" ) ! multiply (C 215)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 215
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "." )
                  call ps_font_char_width( 183, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\267) Tj" ) ! periodcentered (C 183)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 183
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "1" )
                  call ps_font_char_width( 185, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\271) Tj" ) ! onesuperior (C 185)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 185
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "2" )
                  call ps_font_char_width( 178, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\262) Tj" ) ! twosuperior (C 178)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 178
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "3" )
                  call ps_font_char_width( 179, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\263) Tj" ) ! threesuperior (C 179)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 179
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "\" )
                  call ps_font_char_width( 92, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\134) Tj" ) ! backslash (C 92)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 92
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case( "-" )
                  ! "minus" en mode math, ou "endash"
                  call ps_font_char_width( 29, encode_font(), width, h1, h2 )
                  width = width * ps_font_size*fontmul()*FNTFAC
                  w_pos = w_pos + width
                  w = max( w, w_pos )
                  h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
                  if( draw ) then
                     call gresc( "(\226) Tj" ) ! endash (C 29)
                  end if
                  i_back = i_back + 1
                  back_mem(i_back) = 29
                  back_font(i_back) = encode_font()
                  i = i + 2
                  cycle
               case default
                  call PrintMessage( "grtext", "W",                     &
                                     "parsing your string: '" // string // "'", &
                                     "got an error at: '" // string(i:) // "'" )
                  ! don't use return: if the error occurs at the beginning of
                  ! the string, then {w,h_min,h_max} contains infinite values!
                  i = i + 2
                  cycle
            end select
         else
            call PrintMessage( "grtext", "W",                           &
                               "parsing your string: '" // string // "'", &
                               "got an error at: '" // string(i:) // "'", &
                               "found one backslash but nothing after...", &
                               "did you thought about '\\'?" )
            go to 99
         end if
      else
         call ps_font_char_width( ichar( string(i:i) ), encode_font(),  &
                                  width, h1, h2 )
         width = width * ps_font_size*fontmul()*FNTFAC
         w_pos = w_pos + width
         w = max( w, w_pos )
         h_min = min( h_min, h1 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         h_max = max( h_max, h2 * ps_font_size*fontmul()*FNTFAC + FNTBAS )
         if( draw ) then
            ! fill buffer
            if( buf_len == 0 ) then
               buf_start = i
            end if
            buf_len = buf_len + 1
         end if
         i_back = i_back + 1
         back_mem(i_back) = ichar( string(i:i) )
         back_font(i_back) = encode_font()
         i = i + 1
      end if
   end do

99 continue

   ! fix the presence of the default font, not easily detected during the
   ! processing of the string
   if( count(PS_Std_Fonts_presence) == 0 ) then
      PS_Std_Fonts_presence(1) = .true.
   end if

contains
!_______________________________________________________________________
!
   subroutine flush_buffer()

      if( buf_len == 0 ) return

      call insert_font()

      inline = "(" // string(buf_start:buf_start+buf_len-1) // ") Tj"
      call gresc( trim(inline) )
      buf_len = 0

   end subroutine flush_buffer
!_______________________________________________________________________
!
   subroutine insert_font()

      fontsize = nint(ps_font_size*fontmul()*FNTFAC)
      if( fontname /= old_fontname .or. fontsize /= old_fontsize .or.   &
          .not. LAST_FONT_ATTRIB_IS_VALID ) then
         write(inline,"(A,I0,A)") "/F"//encode_font()//" ", fontsize, " Tf"
         call gresc( trim(inline) )
         old_fontname = fontname
         old_fontsize = fontsize
         LAST_FONT_ATTRIB_IS_VALID = .true.
      end if

   end subroutine insert_font
!_______________________________________________________________________
!
   character function encode_font()

      if( fontname == "/Helvetica" ) then
         encode_font = "n"
         PS_Std_Fonts_presence(1) = .true.
      else if( fontname == "/Helvetica-Bold" ) then
         encode_font = "N"
         PS_Std_Fonts_presence(2) = .true.
      else if( fontname == "/Times-Roman" ) then
         encode_font = "r"
         PS_Std_Fonts_presence(3) = .true.
      else if( fontname == "/Times-Bold" ) then
         encode_font = "R"
         PS_Std_Fonts_presence(4) = .true.
      else if( fontname == "/Times-Italic" ) then
         encode_font = "i"
         PS_Std_Fonts_presence(5) = .true.
      else if( fontname == "/Times-BoldItalic" ) then
         encode_font = "I"
         PS_Std_Fonts_presence(6) = .true.
      else if( fontname == "/Symbol" ) then
         encode_font = "g"
         PS_Std_Fonts_presence(7) = .true.
      else if( fontname == "/English157BT-Regular" ) then
         encode_font = "s"
         PS_Std_Fonts_presence(8) = .true.
      end if

   end function encode_font
!_______________________________________________________________________
!
   double precision function fontmul( fontcode )

      character, optional :: fontcode

      ! additional multiplicative factor for the font size, in order
      ! to obtain approximately the same heights for letters (both
      ! in lower- or uppercase) when using different fonts.
      !
      ! see: 'Relative_sizes.eps'

      if( present(fontcode) ) then

         if( fontcode == "n" .or. fontcode == "N" ) then
            fontmul = 0.85d0
         else if( fontcode == "i" .or. fontcode == "I" ) then
            fontmul = 1.00d0
         else if( fontcode == "r" .or. fontcode == "R" ) then
            fontmul = 1.00d0
         else if( fontcode == "g" ) then
            fontmul = 0.95d0
         else if( fontcode == "s" ) then
            fontmul = 1.33d0
         end if

      else

         if( fontname == "/Helvetica" .or. fontname == "/Helvetica-Bold" ) then
            fontmul = 0.85d0
         else if( fontname == "/Times-Italic" .or. fontname == "/Times-BoldItalic" ) then
            fontmul = 1.00d0
         else if( fontname == "/Times-Roman" .or. fontname == "/Times-Bold" ) then
            fontmul = 1.00d0
         else if( fontname == "/Symbol" ) then
            fontmul = 0.95d0
         else if( fontname == "/English157BT-Regular" ) then
            fontmul = 1.33d0
         end if

      end if

   end function fontmul
!_______________________________________________________________________
!
   logical function check_parenthesis( )

      ! In a string, a backslash may lead to a font (or size) change,
      ! therefore the PDF Tj operator will be splitted over several lines
      ! and parenthesis will be no longer balanced.

      integer :: i, k

      check_parenthesis = .false.

      k = 0
      do i = 1, n

         if( string(i:i) == "\" ) then
            return
         else if( string(i:i) == "(" ) then
            k = k + 1
         else if( string(i:i) == ")" ) then
            k = k - 1
         end if

         ! no need to wait the end of the string: quick return
         if( k < 0 ) return

      end do

      if( k == 0 ) check_parenthesis = .true.

   end function check_parenthesis
!_______________________________________________________________________
!
end subroutine
