! PGWEDG -- Annotate an image plot with a wedge (lin or log version)

subroutine pgwedg( side, disp, width, fg, bg, label, axis )

   character(len=*), intent(in) :: side, label
   double precision, intent(in) :: disp, width, fg, bg
   integer,          intent(in) :: axis
   !------ API end ------

   ! Plot an annotated grey-scale or color wedge parallel to a given axis
   ! of the current viewport. This routine is designed to provide a
   ! brightness/color scale for an image drawn with PGIMAG.
   ! The wedge will be drawn with the transfer function set by PGSITF and
   ! using the color index range set by PGSCIR.
   !
   ! Arguments:
   !  SIDE   (input)  : The first character must be one of the characters
   !                    'B' or 'R' signifying the Bottom or Right edge of
   !                    the viewport.
   !                    The second character must be always 'I', to use
   !                    PGIMAG to draw the wedge.
   !  DISP   (input)  : the displacement of the wedge from the specified
   !                    edge of the viewport, measured outwards from the
   !                    viewport in units of the character height. Use a
   !                    negative value to write inside the viewport, a
   !                    positive value to write outside.
   !  WIDTH  (input)  : The total width of the wedge including annotation,
   !                    in units of the character height.
   !  FG     (input)  : The value which is to appear with shade
   !                    1 ("foreground"). Use the values of FG and BG
   !                    that were supplied to PGIMAG.
   !  BG     (input)  : the value which is to appear with shade
   !                    0 ("background").
   !  LABEL  (input)  : Optional units label. If no label is required
   !                    use ' '.
   !
   !  AXIS   (input)  : 1 or 2, for linear or logarithmic graduation.
   !--
   !  15-Oct-1992: New routine (MCS)
   !   2-Aug-1995: no longer needs common (TJP).
   !--------------------------------------------------------------------
   ! modif. by É. Canot
   ! 10-Apr-2004 - Increase size of WDGARR (from 100 to 500).
   ! 11-Jan-2006 - Delete the use of unity character height.
   ! 01-Feb-2006 - Increase size of tempo WDGPIX from 500 to 1500.
   ! 17-Oct-2011 - Increase size of tempo WDGPIX from 1500 to 3000
   !               (today, large screens are not unusual).
   !               To avoid numerical absorption when FG and BG are very
   !               close together, we construct the dummy WDGARR array
   !               between 0 and 1.
   ! 04-Jan-2012 - Impose that all args are intent(in), because FG and BG
   !               was modified, due to the previous modification.
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   ! 10-Apr-2020 - Remove the calls to PGGRAY, no longer used.
   ! 08-May-2021 - Merge with the log version (pgwedg_log), because they
   !               were very similar.
   !--------------------------------------------------------------------

   ! Temporary window coord storage.
   double precision :: wxa, wxb, wya, wyb, xa, xb, ya, yb
   ! Viewport coords of wedge.
   double precision :: vxa, vxb, vya, vyb
   ! Size of unit character height (NDC units).
   double precision :: ndcsiz
   ! True if wedge plotted horizontally.
   logical :: horiz
   ! Symbolic version of SIDE.
   integer :: nside, i
   integer, parameter :: bot = 1, rgt = 4
   double precision :: wedwid, wdginc, vwidth, vdisp, xch, ych, labwid
   double precision :: fg0, bg0, fg1, bg1
   ! Set the fraction of WIDTH used for annotation.
   double precision, parameter :: txtfrc = 0.6d0
   ! Char separation between numbers and LABEL.
   double precision, parameter :: txtsep = 2.2d0
   ! Array to draw wedge in.
   integer, parameter :: wdgpix = 3000
   double precision :: wdgarr(wdgpix)
   ! Define the coordinate-mapping function.
   double precision, save :: tr(6) = [ 0.0d0, 1.0d0, 0.0d0, 0.0d0, 0.0d0, 1.0d0 ]
   character(len=9) :: xopt, yopt
   character :: c

   !------ end of declarations -- execution starts hereafter  ------

   ! Get a numeric version of SIDE.
   c = to_lower(side(1:1))
   if( c == 'b' ) then
      nside = bot
      horiz = .true.
   else if( c == 'r' ) then
      nside = rgt
      horiz = .false.
   else
      call grwarn( 'PGWEDG: Invalid "SIDE" argument (internal error)' )
      pause "for debugging purpose only"
      return
   end if

   ! Check that (from Muesli 2020-04-10) PGIMAG is always used.
   c = to_lower(side(2:2))
   if( c /= 'i' ) then
      call grwarn( 'PGWEDG: Invalid "SIDE" argument (internal error)' )
      pause "for debugging purpose only"
      return
   end if

   call pgbbuf()

   ! Store current world and viewport coords.
   call pgqwin( wxa, wxb, wya, wyb )
   call pgqvp( 0, xa, xb, ya, yb )

   ! Determine the unit character height in normalized device coords.
   call pgqcs( 0, xch, ych )
   if( horiz ) then
      ndcsiz = ych
   else
      ndcsiz = xch
   end if

   ! Convert 'WIDTH' and 'DISP' into viewport units.
   vwidth = width * ndcsiz
   vdisp  = disp  * ndcsiz

   ! Determine the number of character heights required under the wedge.
   labwid = txtsep
   if( label /= " " ) labwid = labwid + 1.0d0

   ! Determine the width of the wedge part of the plot minus the annotation.
   ! (NDC units).
   wedwid = vwidth*(1.0d0-txtfrc)

   ! Use these to determine viewport coordinates for the wedge + annotation.
   vxa = xa
   vxb = xb
   vya = ya
   vyb = yb
   if( nside == bot ) then
      vyb = ya - vdisp
      vya = vyb - wedwid
   else ! nside = rgt
      vxa = xb + vdisp
      vxb = vxa + wedwid
   end if

   ! Set the viewport for the wedge.
   call pgsvp( vxa, vxb, vya, vyb )

   ! Swap FG/BG if necessary to get axis direction right.

   ! Working with FG0, BG0 because we must not modify the intent(in) args!
   fg0 = fg
   bg0 = bg

   fg1 = max(fg0,bg0)
   bg1 = min(fg0,bg0)

   ! Create a dummy wedge array to be plotted. It will not depend of actual
   ! values of FG and BG; in order to avoid floating-point absorption, we
   ! reset the range to [0,1].
   if( fg0 > bg0 ) then
      bg0 = 0.0d0
      fg0 = 1.0d0
   else
      fg0 = 0.0d0
      bg0 = 1.0d0
   end if

   wdginc = 1.0d0/(wdgpix-1)
   do i = 1, wdgpix
      wdgarr(i) = (i-1)*wdginc
   end do

   ! Set a global variable (pgplot module) to tell the PDF driver that
   ! the image comes from a colorbar, not a bitmap image...
   MF_PDF_IN_COLORBAR = .true.

   ! Draw the wedge then change the world coordinates for labelling.
   if( horiz ) then
      call pgswin( 1.0d0, dble(wdgpix), 0.9d0, 1.1d0 )
      call pgimag( wdgarr, wdgpix, 1, 1, wdgpix, 1, 1, fg0, bg0, tr )
      call pgswin( bg1, fg1, 0.0d0, 1.0d0 )
   else
      call pgswin( 0.9d0, 1.1d0, 1.0d0, dble(wdgpix) )
      call pgimag( wdgarr, 1, wdgpix, 1, 1, 1, wdgpix, fg0, bg0, tr )
      call pgswin( 0.0d0, 1.0d0, bg1, fg1 )
   endif

   MF_PDF_IN_COLORBAR = .false.

   ! Draw a labelled frame around the wedge.
   if( nside == bot ) then
      xopt = "BCNST"
      if( axis == 2 ) then
         xopt = trim(xopt) // "L2"
      end if
      yopt = "BC"
   else ! nside = rgt
      xopt = "BC"
      yopt = "BCMST"
      if( axis == 2 ) then
         yopt = trim(yopt) // "L2"
      end if
   end if
   call pgbox( xopt, 0.0d0, 0, yopt, 0.0d0, 0 )

   ! Write the units label.
   if( label /= " " ) then
      call pgmtxt( side, txtsep, 1.0d0, 1.0d0, label )
   end if

   ! Reset the original viewport and world coordinates.
   call pgsvp( xa, xb, ya, yb )
   call pgswin( wxa, wxb, wya, wyb )

   call pgebuf()

end subroutine
