! PGVECT -- Vector map of a 2D data array, with blanking

subroutine PGVECT( a, b, idim, jdim, i1, i2, j1, j2, c, nc, tr, blank )

   integer :: idim, jdim, i1, i2, j1, j2, nc
   double precision :: a(idim,jdim), b(idim, jdim), tr(6), blank, c

   ! Draw a vector map of two arrays.  This routine is similar to
   ! PGCONB in that array elements that have the "magic value" defined by
   ! the argument BLANK are ignored, making gaps in the vector map.  The
   ! routine may be useful for data measured on most but not all of the
   ! points of a grid. Vectors are displayed as arrows; the style of the
   ! arrowhead can be set with routine PGSAH, and the the size of the
   ! arrowhead is determined by the current character size, set by PGSCH.
   !
   ! Arguments:
   !  A      (input)  : horizontal component data array.
   !  B      (input)  : vertical component data array.
   !  IDIM   (input)  : first dimension of A and B.
   !  JDIM   (input)  : second dimension of A and B.
   !  I1,I2  (input)  : range of first index to be mapped (inclusive).
   !  J1,J2  (input)  : range of second index to be mapped (inclusive).
   !  C      (input)  : scale factor for vector lengths, if 0.0, C will be
   !                    set so that the longest vector is equal to the
   !                    smaller of TR(2)+TR(3) and TR(5)+TR(6).
   !  NC     (input)  : vector positioning code.
   !                    <0 vector head positioned on coordinates
   !                    >0 vector base positioned on coordinates
   !                    =0 vector centered on the coordinates
   !  TR     (input)  : array defining a transformation between the I,J
   !                    grid of the array and the world coordinates. The
   !                    world coordinates of the array point A(I,J) are
   !                    given by:
   !                      X = TR(1) + TR(2)*I + TR(3)*J
   !                      Y = TR(4) + TR(5)*I + TR(6)*J
   !                    Usually TR(3) and TR(5) are zero - unless the
   !                    coordinate transformation involves a rotation
   !                    or shear.
   !  BLANK   (input) : elements of arrays A or B that are exactly equal to
   !                    this value are ignored (blanked).
   !--
   !  4-Sep-1992 - Derived from PGCONB [J. Crane].
   ! 26-Nov-1992 - Revised to use PGARRO [TJP].
   ! 25-Mar-1994 - Correct error for NC not =0 [G. Gonczi].
   !  5-Oct-1996 - Correct error in computing max vector length [TJP;
   !               thanks to David Singleton].
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !-----------------------------------------------------------------------

   integer :: i, j
   double precision ::x, y, x1, y1, x2, y2, cc

   !------ end of declarations -- execution starts hereafter  ------

   ! Define grid to world transformation
   x(i,j) = tr(1) + tr(2)*i + tr(3)*j
   y(i,j) = tr(4) + tr(5)*i + tr(6)*j

   ! Check arguments.
   if( i1 < 1 .or. i2 > idim .or. i1 >= i2 .or.                    &
       j1 < 1 .or. j2 > jdim .or. j1 >= j2 ) then
!!      CALL GRWARN('PGVECT: invalid range I1:I2, J1:J2')
      return
   end if

   ! Check for scale factor C.
   cc = c
   if( cc == 0.0 ) then
      do j = j1, j2
         do i = i1, i2
            if( a(i,j) /= blank .and. b(i,j) /= blank )               &
                 cc = max(cc,sqrt(a(i,j)**2+b(i,j)**2))
         end do
      end do
      if( cc == 0.0 ) return
      cc = sqrt(min(tr(2)**2+tr(3)**2,tr(5)**2+tr(6)**2))/cc
   end if

   call pgbbuf()

   do j = j1, j2
      do i = i1, i2

         ! Ignore vector if element of A and B are both equal to BLANK
         if( .not.(a(i,j) == blank .and. b(i,j) == blank) ) then

            ! Define the vector starting and end points according to NC.
            if( nc < 0 ) then
               x2 = x(i,j)
               y2 = y(i,j)
               x1 = x2 - a(i,j)*cc
               y1 = y2 - b(i,j)*cc
            else if( nc == 0 ) then
               x2 = x(i,j) + 0.5*a(i,j)*cc
               y2 = y(i,j) + 0.5*b(i,j)*cc
               x1 = x2 - a(i,j)*cc
               y1 = y2 - b(i,j)*cc
            else
               x1 = x(i,j)
               y1 = y(i,j)
               x2 = x1 + a(i,j)*cc
               y2 = y1 + b(i,j)*cc
            end if

            ! Draw vector.
            call pgarro( x1, y1, x2, y2, clipping=.true. )
         end if
      end do
   end do

   call pgebuf()

end subroutine
