! PGTBX4 -- Support routine for PGTBOX

subroutine PGTBX4( doday, suptyp, axis, convtl, first, tmin, tmax,      &
                   tscale, tick, do2, dopara, mod24 )

   double precision :: tmin, tmax, tick
   integer :: tscale
   character :: axis*(*), suptyp*(*)
   logical :: first, doday, convtl, do2, dopara, mod24

   ! Label an axis in (DD) HH MM SS.S style.
   ! This is the main workhorse of the PGTBOX routines.
   !
   ! This is a support subroutine for PGTBOX and should not be
   ! called by the user.
   !
   ! Inputs:
   !  DODAY  :  Write labels as DD HH MM SS.S else HH MM SS.S with
   !            hours ranging above 24.  Useful for declination labels
   !  SUPTYP :  If 'DHMS' then superscript the fields with d, h, m, & s
   !            If ' DMS' then superscript the fields with    o, '  & ''
   !              Good for declination plots.  You should obviously not
   !              ask for the day field for this to do anything sensible.
   !            If '    ' then no superscripting is done.
   !  AXIS   :  'X' for x-axis, 'Y' for y-axis
   !  CONVTL :  If TRUE, write the labels in the conventional axis
   !            locations (bottom and left for 'X' and 'Y').  Otherwise
   !            write them on the top and right axes ('X' and 'Y')
   !  FIRST  :  If FALSE then omit the first label.
   !  TMIN   :  Start time (seconds)
   !  TMAX   :  End time (seconds)
   !  TSCALE :  Determines finest units of axis
   !              1 => ss, 60 => mm, 3600 => hh, 3600*24 => dd
   !  TICK   :  Major tick interval in seconds
   !  DO2    :  If TRUE, write labels less than 10 with a leading zero.
   !  DOPARA :  Y axis label parallel to axis, else perpendicular
   !  MOD24  :  HH field labelled as modulo 24
   !
   ! 05-Sep-1988 - New routine (Neil Killeen)
   ! 20-Apr-1991 - Add support for new DD (day) field [nebk]
   ! 10-Jun-1993 - Complete rewrite & rename from PGTLAB. Fixes user given
   !               ticks bug too [nebk]
   ! 15-Jan-1995 - Add argument MOD24.
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !  8-Apr-2020 - No longer use UNITS=2 in call to PGLEN. Used 4 instead.
   ! 27-Mar-2021 - Fix initialization not done for ivalz(:) when izero=1.
   !  3-Apr-2021 - Fix location of PGBBUF, and replace 'return' by a move
   !               to the end of the routine, where a call of PGEBUF is done.
   ! 25-Apr-2021 - Fix first tick computation, which was wrong in half of
   !               cases.
   !-----------------------------------------------------------------------

   integer, parameter :: maxtik = 1000

   double precision :: ss(maxtik), tfrac(maxtik)
   integer :: dd(maxtik), hh(maxtik), mm(maxtik)
   character*1 :: asign(maxtik), asignl

   double precision :: time, lenx, leny, coord, fjust, rval, ssl, disp, &
                       lenx2, leny2
   integer :: is, sd, nt, izero, ipos, ineg, it, i, j, k, sprec,        &
              jst(2), jend(2), tlen, last, ival(3), ivalo(3), ivalz(3), &
              ivalf(3), ivall(3), npass, inc, ddl, hhl, mml
   character :: signf*1, text*80, axloc*2
   logical :: writ(4)

   !------ end of declarations -- execution starts hereafter  ------

   ! Direction signs
   sd = 1
   if( tmax < tmin ) sd = -1
   is = 1
   if( tmin < 0.0d0 ) is = -1

   ! Find first tick.  Return if none (this should mean that time interval
   ! is too small).
   nt = nint( tmin / tick )

   if( sd*is*(abs(nt)*tick - abs(tmin)) < 0 ) nt = nt + sd

   time = nt*tick
   if( sd ==  1 ) then
      if( time < tmin .or. tmax < time ) return
   end if
   if( sd == -1 ) then
      if( time < tmax .or. tmin < time ) return
   end if

   call pgbbuf()

   ! Now step through time range in TICK increments and convert
   ! times in seconds at each tick to  +/- (DD) HH MM SS.S
   izero = 0
   it = 1
10 if( (sd == 1  .and. time > (tmax+1.0d-5)) .or.                       &
       (sd == -1 .and. time < (tmax-1.0d-5)) ) goto 20
      if( it > maxtik ) then
         call grwarn( 'PGTBX4: storage exhausted -- you have ' //       &
                      'asked for far too many ticks' )
         goto 20
      end if

      ! Convert to (DD) HH MM SS.S and find fraction of window that this
      ! tick falls at
      call pgtbx5( doday, time, asign(it), dd(it), hh(it),              &
                   mm(it), ss(it) )
      tfrac(it) = (time - tmin) / (tmax - tmin)

      ! note zero tick
      if( nt == 0 ) izero = it

      ! Increment time
      nt = nt + sd
      time = nt*tick
      it = it + 1

      goto 10
20 continue
   it = it - 1

   ! Work out the precision with which to write fractional seconds
   ! labels into the SS.S field.   All other fields have integer labels.
   sprec = 0
   if( tscale == 1 ) then
      if( tick < 0.01d0 ) then
         sprec = 3
      else if( tick < 0.1d0 ) then
         sprec = 2
      else if( tick < 1.0d0 ) then
         sprec = 1
      end if
   end if

   ! Label special case of first tick.  Prepare fields and label
   call pgtbx6( doday, mod24, tscale, dd(1), hh(1), mm(1),              &
                ss(1), ivalf, rval, writ )
   signf = 'H'
   if( doday ) signf = 'D'
   call pgtbx7( suptyp, signf, asign(1), ivalf, rval, writ,             &
                sprec, do2, text, tlen, last )

   ! Set label displacements from axes.  This is messy for labels oriented
   ! perpendicularly on the right hand axis as we need to know how long
   ! the longest string we are going to write is before we write any
   ! labels as they are right justified.
   if( axis == 'X' ) then
      if( convtl ) then
         axloc = 'B'
         if( suptyp /= 'NONE' ) then
            disp = 1.4d0
         else
            disp = 1.2d0
         end if
      else
         axloc = 'T'
         disp = 0.7d0
      end if
   else if( axis == 'Y' ) then
      if( convtl) then
         axloc = 'LV'
         if( dopara ) axloc = 'L'
         disp = 0.7d0
      else
         if( dopara ) then
            axloc = 'R'
            if( suptyp /= 'NONE' ) then
            disp = 1.7d0
            else
            disp = 1.9d0
            end if
         else

            ! Work out number of characters in first label
            axloc = 'RV'
            if( asign(1) /= '-' .and. tmin*tmax < 0.0d0 ) then
            call pglen( 4, ' -'//text(1:tlen), lenx, leny )
            else
            call pglen( 4, ' '//text(1:tlen), lenx, leny )
            end if
            call pgqcs( 4, lenx2, leny2 )
            disp = lenx/lenx2
         end if
      end if
   end if

   ! Now write the label to the plot.  The X-axis label for the first tick is
   ! centered such that the last field of the label is centered on the tick
   if( first ) then
      call pglen( 5, text(last:tlen), lenx, leny )

      if( axis == 'X' ) then
         coord = tfrac(1) + lenx / 2.0d0
         fjust = 1.0d0
      else if( axis == 'Y' ) then
         if( dopara ) then
            coord = tfrac(1) + leny / 2.0d0
            fjust = 1.0d0
         else
            fjust = 1.0d0
            coord = tfrac(1)
         end if
      end if
      call pgmtxt( axloc, disp, coord, fjust, text(1:tlen) )
   end if
   if( it == 1 ) go to 99

   ! Designate which field out of DD or HH will carry the sign, depending
   ! on whether you want the day field or not for the rest of the ticks
   signf = 'H'
   if( doday ) signf = 'D'

   ! Set up labelling justifications for the rest of the labels
   if( axis == 'X' ) then
      fjust = 0.5d0
   else if( axis == 'Y' ) then
      if( dopara) then
         fjust = 0.5d0
      else
         fjust = 1.0d0
      end if
   end if

   ! Note zero crossings; IPOS is the first positive tick and
   ! INEG is the first negative tick on either side of 0
   ipos = 0
   ineg = 0

   if( izero /= 0 ) then
      j = izero - 1
      if( j >= 1 ) then
         if( asign(j) == '-' ) then
            ineg = j
         else if( asign(j) == ' ' ) then
            ipos = j
         end if
      end if
      j = izero + 1
      if( j <= it ) then
         if( asign(j) == '-' ) then
            ineg = j
         else if( asign(j) == ' ' ) then
            ipos = j
         end if
      end if
   end if

   ! Now label special case of zero tick. It carries the sign change
   ! when going from positive to negative time, left to right.
   if( izero /= 0 .and. izero /= 1 ) then
      call pgtbx6( doday, mod24, tscale, dd(izero), hh(izero),          &
                   mm(izero), ss(izero), ivalz, rval, writ )

      if( asign(izero-1) == ' ' ) asign(izero) = '-'
      call pgtbx7( suptyp, signf, asign(izero), ivalz, rval, writ,      &
                   sprec, do2, text, tlen, last )

      coord = tfrac(izero)
      call pgmtxt( axloc, disp, coord, fjust, text(1:tlen) )
   end if

   ! We may need an extra "virtual" tick if there is no zero crossing
   ! and SD=-1 & IS=1 or SD=1 & IS=-1.  It is used to work out which
   ! fields to label on the right most tick which is labelled first.
   if( izero == 0 ) then
      if( sd*is == -1 ) then
         if( (sd == -1 .and. time <= 0.0d0) .or.                        &
             (sd ==  1 .and. time >= 0.0d0) ) time = 0.0d0
         call pgtbx5( doday, time, asignl, ddl, hhl, mml, ssl )
         call pgtbx6( doday, mod24, tscale, ddl, hhl, mml, ssl,         &
                      ivall, rval, writ )
      end if
   end if

   ! We want to label in the direction(s) away from zero, so we may  need
   ! two passes. Determine the start and end ticks for each required pass.
   jst(2) = 0
   jend(2) = 0
   npass = 1
   if( izero == 0 ) then
      if( is*sd == 1 ) then
         jst(1) = 1
         jend(1) = it
      else
         jst(1) = it
         jend(1) = 1
      end if
   else
      if( ineg == 0 .or. ipos == 0 ) then
         jst(1) = izero
         jend(1) = it
         if( izero == it ) jend(1) = 1
      else
         npass = 2
         jst(1) = izero
         jend(1) = 1
         jst(2) = izero
         jend(2) = it
      end if
   end if

   ! Now label the rest of the ticks.  Always label away from 0
   do i = 1, npass

      ! Initialize previous tick values.  Use virtual tick if labelling
      ! left to right without a zero (one pass)
!### TODO: not sure, but it is to avoid valgrind detecting ivalz(k)
!          as non initialized
ivalz(1:3) = 0
      do k = 1, 3
         if( izero == 0 ) then
            if( jst(i) == 1 ) then
               ivalo(k) = ivalf(k)
            else
               ivalo(k) = ivall(k)
            end if
         else
            ivalo(k) = ivalz(k)
!!print *, "PGTBX4:       k, ivalz(k) = ", k, ivalz(k)
         end if
      end do

      inc = 1
      if( jend(i) < jst(i) ) inc = -1
      do j = jst(i), jend(i), inc

         ! First and zero tick already labelled
         if( j /= 1 .and. j /= izero ) then

            ! Prepare fields
            call pgtbx6( doday, mod24, tscale, dd(j), hh(j), mm(j),     &
                         ss(j), ival, rval, writ )

            ! Don't write unchanging fields
            do k = 1, 3
               if( ival(k) == ivalo(k) ) writ(k) = .false.
            end do

            ! Prepare label
            call pgtbx7( suptyp, signf, asign(j), ival, rval, writ,     &
                         sprec, do2, text, tlen, last )

            ! Write label
            coord = tfrac(j)
            call pgmtxt( axloc, disp, coord, fjust, text(1:tlen) )

            ! Update old values
            do k = 1, 3
               ivalo(k) = ival(k)
            end do
         end if
      end do
   end do

99 continue

   call pgebuf()

end subroutine
