! PGTBX1 -- Support routine for PGTBOX

subroutine PGTBX1( axis, doday, dopara, tmin, tmax, tick, nsub, tscale )

   double precision :: tmin, tmax, tick
   integer :: nsub, tscale
   logical :: doday, dopara
   character(len=1) :: axis

   ! Work out what the finest units the time labels will be in and
   ! return the tick increments if the user does not set them.
   !
   ! This is a support routine for PGTBOX and should not
   ! be called by the user.
   !
   ! Input:
   !  AXIS   :  'X' or 'Y' for use in determining if labels overwrite
   !  TMIN   :  Start time in seconds
   !  TMAX   :  End   time in seconds
   !  DOPARA :  True if label to be parallel to axis, else perpendicular
   ! Input/output:
   !  DODAY  :  Write labels as DD HH MM SS.S else HH MM SS.S with
   !            hours ranging above 24.  Useful for declination labels
   !  TICK   :  Major tick interval in seconds.  If 0.0 on input, will
   !            be set here.
   !  NSUB   :  Number of minor ticks between major ticks. If 0 on input
   !            will be set here.
   ! Outputs:
   !  TSCALE :  Determines finest unit of labelling
   !            (1 => ss, 60 => mm, 3600 => hh, 3600*24 => dd)
   !
   ! 05-Sep-1988 - New routine (Neil Killeen)
   ! 08-Apr-1991 - Correctly work out HH MM SS when the time > 60 h [nebk]
   ! 20-Apr-1991 - Revise to add support for new DD (day) field and
   !               do lots of work on tick algorithm [nebk]
   ! 10-Jun-1993 - Deal with user given ticks & rename from PGTIME [nebk/jm
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------

   INTEGER, parameter :: NLIST1 = 19, NLIST2 = 10, NLIST3 = 6,          &
                         NLIST4 = 8, NTICMX = 8

   double precision TICKS1(NLIST1), TICKS2(NLIST2), TICKS3(NLIST3),              &
   TICKS4(NLIST4), TOCK, TOCK2, TINT, TINTS, TMINS, TMAXS
   INTEGER NSUBS1(NLIST1), NSUBS2(NLIST2), NSUBS3(NLIST3),              &
   NSUBS4(NLIST4), NPL, NTICK, ITICK, STRLEN
   CHARACTER STR*15

   SAVE TICKS1, TICKS2, TICKS3, TICKS4
   SAVE NSUBS1, NSUBS2, NSUBS3, NSUBS4

   DATA TICKS1 /0.001d0,  0.002d0,                 0.005d0,             &
                0.01d0,   0.02d0,                  0.05d0,              &
                0.1d0,    0.2d0,                   0.5d0,               &
                1.0d0,    2.0d0,   3.0d0,  4.0d0,  5.0d0,               &
                6.0d0,   10.0d0,  15.0d0, 20.0d0, 30.0d0 /
   DATA NSUBS1 / 4,        4,                       2,                  &
                 4,        4,                       2,                  &
                 4,        4,                       2,                  &
                 4,        4,       3,      4,      5,                  &
                 3,        2,       3,      2,      3 /

   DATA TICKS2 /1.0d0,  2.0d0,  3.0d0,  4.0d0,  5.0d0,                  &
                6.0d0, 10.0d0, 15.0d0, 20.0d0, 30.0d0 /
   DATA NSUBS2 / 4,      4,      3,      4,      5,                     &
                 3,      2,      3,      2,      3 /

   DATA TICKS3 /1.0d0, 2.0d0, 3.0d0, 4.0d0, 6.0d0, 12.0d0 /
   DATA NSUBS3 / 4,     4,     3,     4,     3,      2 /

   DATA TICKS4 /1.0d0, 2.0d0, 3.0d0, 4.0d0, 5.0d0, 6.0d0, 8.0d0, 9.0d0 /
   DATA NSUBS4 / 4,     4,     3,     4,     5,     3,     4,     3 /

   !------ end of declarations -- execution starts hereafter  ------

   ! Turn off DD (day) field if it has been unnecessarily asked for
   if( abs(tmin)<24.0*3600.0 .and. abs(tmax)>24.0*3600.0 ) doday = .false.

   ! If a tick size is provided, use it to determine TSCALE
   tint = abs(tmax - tmin)
   tick = abs(tick)
   if( tick /= 0.0 ) then
      if( tick >= tint ) then
         call grwarn( 'PGTBX1: user given tick bigger than time '       &
                      //'interval; will auto-tick')
         tick = 0.0
      else if( tick < 0.001 ) then
         call grwarn( 'PGTBX1: user given tick too small (< 1 ms); '    &
                      //'will auto-tick')
         tick = 0.0
      else
         if( mod(tick, 60.0)  /=  0.0 ) then
            tscale = 1
         else if( mod(tick, 3600.0) /= 0.0 ) then
            tscale = 60
         else if( .not.doday ) then
            tscale = 3600
         else if( mod(tick,(24.0*3600.0)) /= 0.0 ) then
            tscale = 3600
         else
            tscale = 24 * 3600
         endif

         ! Make a simple default for the number of minor ticks and bug out
         if( nsub == 0 ) nsub = 2
         return
      end if
   end if

   ! Work out label units depending on time interval if user
   ! wants auto-ticking
   if( tint <= 5*60 ) then
      tscale = 1
   else if( tint <= 5*3600 ) then
      tscale = 60
   else
      if( .not.doday ) then
         tscale = 3600
      else
         if( tint <= 5*24*3600 ) then
            tscale = 3600
         else
            tscale = 3600*24
         end if
      end if
   end if

   !!!!!
   ! Divide interval into NTICK major ticks and NSUB minor intervals
   ! The tick choosing algorithm is not very robust, so watch out
   ! if you fiddle anything.
   !!!!!
   tints = tint / tscale
   if( tscale == 1 ) then

      ! Time in seconds.  If the time interval is very small, may need to
      ! label with up to 3 decimal places.  Have less ticks to help prevent
      ! label overwrite. STR is a dummy tick label to assess label
      ! overwrite potential
      if( dopara ) then
         if( tints <= 0.01d0 ) then
            ntick = 4
            str = '60.423'
            strlen = 6
         else if( tints <= 0.1d0 ) then
            ntick = 5
            str = '60.42'
            strlen = 5
         else if( tints <= 1.0 ) then
            ntick = 6
            str = '60.4'
            strlen = 4
         else
            ntick = 6
            str = '60s'
            strlen = 3
         end if
      else
         ntick = 6
         str = ' '
         strlen = 1
      end if
      tock = tints / ntick

      ! Select nearest tick to TOCK from list.
      call pgtbx2( tock, nlist1, ticks1, nsubs1, tick, nsub, itick )

      ! Check label overwrite and/or too many ticks.
      call pgtbx3( doday, 0, tscale, tints, nticmx, nlist1, ticks1,     &
                   nsubs1, itick, axis, dopara, str(1:strlen),          &
                   tick, nsub )
   else if( tscale == 60 ) then

      ! Time in minutes
      ntick = 6
      tock = tints / ntick

      ! Select nearest tick from list
      call pgtbx2( tock, nlist2, ticks2, nsubs2, tick, nsub, itick )

      ! Check label overwrite and/or too many ticks.
      if( dopara ) then
         str = '42m'
         strlen = 3
      else
         str = ' '
         strlen = 1
      end if
      call pgtbx3( doday, 0, tscale, tints, nticmx, nlist2, ticks2,     &
                   nsubs2, itick, axis, dopara, str(1:strlen),          &
                   tick, nsub )
   else
      if( tscale == 3600 .and. doday ) then

         ! Time in hours with the day field
         ntick = 6
         tock = tints / ntick

         ! Select nearest tick from list
         call pgtbx2( tock, nlist3, ticks3, nsubs3, tick, nsub, itick )

         ! Check label overwrite and/or too many ticks.
         if( dopara ) then
            str = '42h'
            strlen = 3
         else
            str = ' '
            strlen = 1
         end if
         call pgtbx3( doday, 0, tscale, tints, nticmx, nlist3, ticks3,  &
                      nsubs3, itick, axis, dopara, str(1:strlen),       &
                      tick, nsub )
      else

         ! Time in hours with no day field or time in days. Have less
         ! ticks for big numbers or the parallel labels will overwrite.
         if( dopara ) then
            tmins = abs(tmin) / tscale
            tmaxs = abs(tmax) / tscale
            call pgnpl( -1, nint(max(tints,tmins,tmaxs)), npl )
            if( npl <= 3 ) then
               ntick = 6
            else if( npl == 4 ) then
               ntick = 5
            else
               ntick = 4
            end if
            str = '345678912'
            str(npl+1:) = 'd'
            strlen = npl + 1
         else
            str = ' '
            strlen = 1
            ntick = 6
         end if
         tock = tints / ntick

         ! Select nearest tick from list; 1 choose nearest nice integer
         ! scaled by the appropriate power of 10
         call pgnpl( -1, nint(tock), npl )
         tock2 = tock / 10**(npl-1)

         call pgtbx2( tock2, nlist4, ticks4, nsubs4, tick, nsub, itick )
         tick = tick * 10**(npl-1)

         ! Check label overwrite and/or too many ticks.
         call pgtbx3( doday, npl, tscale, tints, nticmx, nlist4,        &
                      ticks4, nsubs4, itick, axis, dopara,              &
                      str(1:strlen), tick, nsub )
      end if
   end if

   !  Convert tick to seconds
   tick = tick * tscale

end subroutine
