! PGTBOX -- Draw frame and write (DD) HH MM SS.S labelling

subroutine PGTBOX( xopt, xtick, nxsub, yopt, ytick, nysub )

   character(len=*), intent(in) :: xopt, yopt
   double precision, intent(in) :: xtick, ytick
   integer,          intent(in) :: nxsub, nysub

   ! Draw a box and optionally label one or both axes with (DD) HH MM SS
   ! style numeric labels (useful for time). If this style of labelling is
   ! desired, then PGSWIN should have been called previously with the
   ! extrema in SECONDS of time.
   !
   ! In the seconds field, you can have at most 3 places after the decimal
   ! point, so that 1 ms is the smallest time interval you can time label.
   !
   ! Large numbers are coped with by fields of 6 characters long.  Thus
   ! you could have times with days or hours as big as 999999.  However,
   ! in practice, you might have trouble with labels overwriting  themselves
   ! with such large numbers unless you
   !    a) use a small time INTERVAL,
   !    b) use a small character size or
   !    c) choose your own sparse ticks in the call to PGTBOX.
   !
   ! PGTBOX will attempt, when choosing its own ticks, not to overwrite
   ! the labels, but this algorithm is not very bright and may fail.
   !
   ! PGTBOX's labelling philosophy is that the left-most or bottom tick of
   ! the axis contains a full label.  Thereafter, only changing fields are
   ! labelled.  Negative fields are given a '-' label, positive fields
   ! have none.   Axes that have the DD (or HH if the day field is not
   ! used) field on each major tick carry the sign on each field.  If the
   ! axis crosses zero, the zero tick will carry a full label and sign.
   !
   ! This labelling style can cause a little confusion with some special
   ! cases, but as long as you know its philosophy, the truth can be divined.
   ! Consider an axis with TMIN=20s, TMAX=-20s.   The labels will look like
   !
   !        +----------+----------+----------+----------+
   !     0h0m20s      10s      -0h0m0s      10s        20s
   !
   ! Knowing that the left field always has a full label and that
   ! positive fields are unsigned, informs that time is decreasing
   ! from left to right, not vice versa.   This can become very
   ! unclear if you have used the 'F' option, but that is your problem !
   !
   ! Exceptions to this labelling philosophy are when the finest time
   ! increment being displayed is hours (with option 'Y') or days.
   ! Then all fields carry a label.  For example,
   !
   !        +----------+----------+----------+----------+
   !      -10h        -8h        -6h        -4h        -2h
   !
   !
   ! PGTBOX can be used in place of PGBOX; it calls PGBOX and only invokes
   ! time labelling if requested. Other options are passed intact to PGBOX.
   !
   ! Inputs:
   !  XOPT   :  X-options for PGTBOX.  Same as for PGBOX (see below) plus
   !
   !             'Z' for (DD) HH MM SS.S time labelling
   !             'Y' means don't include the day field so that labels
   !                 are HH MM SS.S rather than DD HH MM SS.S   The hours
   !                 will accumulate beyond 24 if necessary in this case.
   !             'X' label the HH field as modulo 24.  Thus, a label
   !                 such as 25h 10m would come out as 1h 10m
   !             'H' means superscript numbers with d, h, m, & s  symbols
   !             'D' means superscript numbers with o, ' and '' degree symbols
   !             'F' causes the first label (left- or bottom-most) to
   !                 be omitted. Useful for sub-panels that abut each other.
   !                 Care is needed because first label carries sign as well.
   !             'O' means omit leading zeros in numbers < 10
   !                 E.g.  3h 3m 1.2s rather than 03h 03m 01.2s  Useful
   !                 to help save space on X-axes. The day field does not
   !                 use this facility.
   !
   !  YOPT   :  Y-options for PGTBOX.  See above.
   !  XTICK  :  X-axis major tick increment.  0.0 for default.
   !  YTICK  :  Y-axis major tick increment.  0.0 for default.
   !            If the 'Z' option is used then XTICK and/or YTICK must
   !            be in seconds.
   !  NXSUB  :  Number of intervals for minor ticks on X-axis. 0 for default
   !  NYSUB  :  Number of intervals for minor ticks on Y-axis. 0 for default
   !
   !  The regular XOPT and YOPT axis options for PGBOX are
   !
   !  A : draw Axis (X axis is horizontal line Y=0, Y axis is vertical
   !      line X=0).
   !  B : draw bottom (X) or left (Y) edge of frame.
   !  C : draw top (X) or right (Y) edge of frame.
   !  G : draw Grid of vertical (X) or horizontal (Y) lines.
   !  I : Invert the tick marks; ie draw them outside the viewport
   !      instead of inside.
   !  L : label axis Logarithmically (see below).
   !  N : write Numeric labels in the conventional location below the
   !      viewport (X) or to the left of the viewport (Y).
   !  P : extend ("Project") major tick marks outside the box (ignored if
   !      option I is specified).
   !  M : write numeric labels in the unconventional location above the
   !      viewport (X) or to the right of the viewport (Y).
   !  T : draw major Tick marks at the major coordinate interval.
   !  S : draw minor tick marks (Subticks).
   !  V : orient numeric labels Vertically. This is only applicable to Y.
   !      The default is to write Y-labels parallel to the axis.
   !  1 : force decimal labelling, instead of automatic choice (see PGNUMB).
   !  2 : force exponential labelling, instead of automatic.
   !
   !     ******************        EXCEPTIONS       *******************
   !
   !     Note that
   !       1) PGBOX option 'L' (log labels) is ignored with option 'Z'
   !       2) The 'O' option will be ignored for the 'V' option as it
   !          makes it impossible to align the labels nicely
   !       3) Option 'Y' is forced with option 'D'
   !
   !     ***************************************************************
   !
   !--
   ! 05-Sep-1988 - New routine
   ! 20-Apr-1991 - Add support for new DD (day) field and implement
   !               labelling on any axis (bottom,top,left,right)
   ! 10-Jun-1993 - Add option 'O' for leading zeros, correctly deal with
   !               user ticks, fully support 'V' and 'NM' options, modify
   !               slightly meaning of 'F' option
   ! 16-Jan-1995 - Add option 'X'
   ! 16-Aug-1996 - Bring axis labelling displacements more in line with
   !               those of pgbox.f
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !-----------------------------------------------------------------------

   double precision :: xtickd, ytickd, xmin, xmax, ymin, ymax
   integer :: ipt, tscalx, tscaly, nxsubd, nysubd
   character :: xxopt*15, yyopt*15, suptyp*4
   logical :: xtime, ytime, first, dodayx, dodayy, do2, dopara, mod24

   !------ end of declarations -- execution starts hereafter  ------

   ! keep the following lines for external call from
   ! 'call_mf_pgtbox_from_x11_driver'
   call grtoup( XOPT_SAVE, xopt )
   call grtoup( YOPT_SAVE, yopt )

   call grsci(1)
   call grsls(1)

   gr_minus_sign_math_mode = .true.

   ! Copy inputs
   xtickd = xtick
   ytickd = ytick
   nxsubd = nxsub
   nysubd = nysub

   ! Get window in world coordinates
   call pgqwin( xmin, xmax, ymin, ymax )

   ! X-axis first
   call grtoup( xxopt, xopt )
   xtime = .false.
   if( index(xxopt,'Z') /= 0 ) then

      ! Work out units for labelling and find the tick increments.
      if( abs(xmax-xmin) < 0.001 ) then
         call grwarn( 'PGTBOX: X-axis time interval too small ' //      &
                      '(< 1 ms) for time labels' )
      else
         xtime = .true.
         dodayx = .true.
         if( index(xxopt,'Y') /= 0 .or. index(xxopt,'D') /= 0 ) dodayx = .false.

         dopara = .true.
         call pgtbx1( 'X', dodayx, dopara, xmin, xmax, xtickd,          &
                      nxsubd, tscalx )
      end if
   end if

   ! Same again for Y-axis
   call grtoup( yyopt, yopt )
   ytime = .false.
   if( index(yyopt,'Z') /= 0 ) then
      if( abs(ymax-ymin) < 0.001) then
         call grwarn( 'PGTBOX: Y-axis time interval too small ' //      &
                      '(< 1ms) for time labels' )
      else
         ytime = .true.
         dodayy = .true.
         if( index(yyopt,'Y') /= 0 .or. index(yyopt,'D') /= 0 ) dodayy = .false.

         dopara = .true.
         if( index(yyopt,'V') /= 0 ) dopara = .false.

         call pgtbx1( 'Y', dodayy, dopara, ymin, ymax, ytickd,          &
                      nysubd, tscaly )
      end if
   end if

   ! Parse options list.  For call to PGBOX when doing time labelling, we
   ! don't want L (log), N or M (write numeric labels).
   if( xtime ) then
      ipt = index(xxopt,'L')
      if( ipt /= 0 ) xxopt(ipt:ipt) = ' '
      ipt = index(xxopt,'N')
      if( ipt /= 0 ) xxopt(ipt:ipt) = ' '
      ipt = index(xxopt,'M')
      if( ipt /= 0 ) xxopt(ipt:ipt) = ' '
   end if

   if( ytime ) then
      ipt = index(yyopt,'L')
      if( ipt /= 0 ) yyopt(ipt:ipt) = ' '
      ipt = index(yyopt,'N')
      if( ipt /= 0 ) yyopt(ipt:ipt) = ' '
      ipt = index(yyopt,'M')
      if( ipt /= 0 ) yyopt(ipt:ipt) = ' '
   end if

   ! Draw box and ticks
   call pgbox( xxopt, xtickd, nxsubd, yyopt, ytickd, nysubd )

   ! Add (DD) HH MM SS labels if desired.  Go back to the original user
   ! specified options list.
   xxopt = ' '
   call grtoup( xxopt, xopt )
   if( xtime .and. (index(xxopt,'N') /= 0 .or.                          &
                    index(xxopt,'M') /= 0) ) then
      first = .true.
      if( index(xxopt,'F') /= 0 ) first = .false.

      suptyp = 'NONE'
      if( index(xxopt,'D') /= 0 ) suptyp = ' DMS'
      if( index(xxopt,'H') /= 0 ) suptyp = 'DHMS'

      do2 = .true.
      if( index(xxopt,'O') /= 0 ) do2 = .false.

      dopara = .true.

      mod24 = .false.
      if( index(xxopt,'X') /= 0 ) mod24 = .true.

      if( index(xxopt,'N') /= 0 )                                       &
         call pgtbx4( dodayx, suptyp, 'X', .true., first,               &
                      xmin, xmax, tscalx, xtickd, do2, dopara, mod24 )

      if( index(xxopt,'M') /= 0 )                                       &
         call pgtbx4( dodayx, suptyp, 'X', .false., first,              &
                      xmin, xmax, tscalx, xtickd, do2, dopara, mod24 )
   end if

   yyopt = ' '
   call grtoup( yyopt, yopt )
   if( ytime .and. (index(yyopt,'N') /= 0 .or.                          &
                    index(yyopt,'M') /= 0) ) then
      first = .true.
      if( index(yyopt,'F') /= 0 ) first = .false.

      suptyp = 'NONE'
      if( index(yyopt,'D') /= 0 ) suptyp = ' DMS'
      if( index(yyopt,'H') /= 0 ) suptyp = 'DHMS'

      DOPARA = .TRUE.
      if( index(yyopt,'V') /= 0 ) dopara = .false.

      do2 = .true.
      if( dopara .and. index(yyopt,'O') /= 0 ) do2 = .false.

      mod24 = .false.
      if( index(yyopt,'X') /= 0 ) mod24 = .true.

      if( index(yyopt,'N') /= 0 )                                       &
         call pgtbx4( dodayy, suptyp, 'Y', .true., first,               &
                      ymin, ymax, tscaly, ytickd, do2, dopara, mod24 )

      if( index(yyopt,'M') /= 0 )                                       &
         call pgtbx4( dodayy, suptyp, 'Y', .false., first,              &
                      ymin, ymax, tscaly, ytickd, do2, dopara, mod24 )

   end if

   gr_minus_sign_math_mode = .false.

end subroutine
