! PGSCRN -- Set color representation by name

subroutine PGSCRN( ci, name, ier )

   integer,          intent(in)  :: ci
   character(len=*), intent(in)  :: name
   integer,          intent(out) :: ier

   ! Set color representation: i.e., define the color to be associated with a
   ! color index.  Ignored for devices which do not support variable color or
   ! intensity.  This is an alternative to routine PGSCR. The color
   ! representation is defined by name instead of (R,G,B) components.
   !
   ! Color names are defined in an external file which is read the first time
   ! that PGSCRN is called. The name of the external file is found as follows:
   ! 1. if environment variable (logical name) MFPLOT_RGB is defined, its
   !    value is used as the file name;
   ! 2. otherwise, the file "rgb.txt", provided with the library MFPLOT
   !    is used.
   ! If all of these fail to find a file, an error is reported and the routine
   ! does nothing.
   !
   ! Each line of the file "rgb.txt" defines one color, with four blank-
   ! separated fields per line. The first three fields are the R, G, B
   ! components, are integers in the range 0 (zero intensity) to 255
   ! (maximum intensity). The fourth field is the color name. The color name
   ! may include embedded blanks. Example:
   !
   ! 255   0   0 red
   ! 255 105 180 hot pink
   ! 255 255 255 white
   !   0   0   0 black
   !
   ! Be aware that the writing of each 'line' is constrained by the following
   ! format:           line(1:12) -> I3, 1X, I3, 1X, I3, 1X
   !                   line(13:)  -> any string
   ! In all cases, the rgb.txt file cannot embed more than maxcol colors
   ! (below, maxcol is defined to 1000).
   !
   ! Arguments:
   !  CI   (input) : the color index to be defined, in the range 0-max???.
   !                 If a color index greater than the device maximum is
   !                 specified, the call is ignored. Color index 0 applies
   !                 to the background color.
   !  NAME (input) : the name of the color to be associated with this color
   !                 index. This name must be in the external file. The names
   !                 are not case-sensitive. If the color is not listed in the
   !                 file, the color representation is not changed.
   !  IER  (output): returns 0 if the routine was successful, 1 if an error
   !                 occurred (either the external file could not be read, or
   !                 the requested color was not defined in the file).
   !--
   ! 12-Oct-1992 [TJP]
   ! 31-May-1993 [TJP] use GROPTX to open file.
   !  7-Nov-1994 [TJP] better error messages.
   !
   ! 17-May-2011 - Don't use the name MFPLOT_RGB: this may confuse the
   !               MUESLI user.
   ! 10-Nov-2018 - Don't use the name MFPLOT_DIR (anyway, it was used in
   !               GRGFIL, not in the current routine).
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   ! 16-Nov-2021 - Simplified the code, by using standard Fortran 90 functions.
   !               Restore the use of the environment variable MFPLOT_RGB
   !               in the case where the user want to use a great number of
   !                user-defined colors.
   ! 29-Jan-2024 - MFPLOT_DIR renamed MFPLOT_DIR
   !-----------------------------------------------------------------------

   integer, parameter :: maxcol=1000

   integer :: i, ir, ig, ib, unit, ios, L
   character(len=20) :: creq
   character(len=255) :: text

   integer, save :: ncol = 0
   character(len=20), save :: cname(maxcol)
   double precision, save :: rr(maxcol), rg(maxcol), rb(maxcol)

   !------ end of declarations -- execution starts hereafter  ------

   ! On first call, find the RGB database and read it.
   if( ncol == 0 ) then
      ! Checking for an environment variable called MFPLOT_RGB...
      call grgenv( "RGB", text, L )
      if( L == 0 ) then
         if( MFPLOT_DIR == "" ) then
            call set_muesli_install_dir_fgl()
         end if
         text = trim(MFPLOT_DIR) // "/rgb.txt"
      end if
      call grglun(unit)
      open( unit=unit, file=trim(text), status="old", iostat=ios )
      if( ios /= 0 ) goto 99
      do i = 1, maxcol
         read( unit, "(A)", err=10, end=10 ) text
         read( text(1:12), * ) ir, ig, ib
         ncol = ncol+1
         call grtoup( cname(ncol), text(13:) )
         rr(ncol) = ir/255.0
         rg(ncol) = ig/255.0
         rb(ncol) = ib/255.0
      end do
10    close (unit)
      call grflun(unit)
   end if

   ! Look up requested color and set color representation if found.
   call grtoup( creq, name )
   do i = 1, ncol
      if( creq == cname(i) ) then
         call grscr( ci, rr(i), rg(i), rb(i) )
         ier = 0
         return
      end if
   end do

   ! Color not found.
   ier = 1
   call grwarn( "Color not found: " // trim(name) )
   return

99 continue

   ! Database not found.
   ier = 1
   ncol = -1
   call grflun(unit)
   call grwarn( "Unable to read color file: " // trim(text) )

end subroutine
