! PGQTXT -- Find bounding box of text string

subroutine PGQTXT( x, y, angle, hjust, text, xbox, ybox, device )

   double precision, intent(in)           :: x, y, angle, hjust
   character(len=*), intent(in)           :: text
   double precision, intent(out)          :: xbox(4), ybox(4)
   logical,          intent(in), optional :: device

   ! This routine returns a bounding rectangle (may be inclined) for a text
   ! string. Instead of drawing the string as routine PGPTXT does, it returns
   ! in XBOX and YBOX the coordinates of the corners of a rectangle parallel
   ! to the string baseline that just encloses the string. The four corners
   ! are in the order: lower left, upper left, upper right, lower right
   ! (where left and right refer to the first and last characters in the
   ! string).
   !
   !                   2 +-------------------+ 3
   !                     |                   |
   !                   1 +-------------------+ 4
   !
   ! If the string is blank or contains no drawable characters, all four
   ! elements of XBOX and YBOX are assigned the starting point of the string,
   ! (X,Y).
   !
   ! Arguments:
   !  X, Y, ANGLE, HJUST, TEXT (input) : these arguments are the same as
   !                    the corresponding arguments in PGPTXT.
   !  XBOX, YBOX (output) : arrays of dimension 4; on output, they
   !                    contain the world coordinates of the bounding
   !                    box in (XBOX(1), YBOX(1)), ..., (XBOX(4), YBOX(4)).
   !
   !  device (input) : if not present (or FALSE), world coordinates are
   !                   returned in XBOX, YBOX (this is the default).
   !                   if present and TRUE, device coordinates are returned.
   !--
   ! 12-Sep-1993 - New routine [TJP].
   !  8-Nov-1994 - Return something for blank string [TJP].
   ! 14-Jan-1997 - Additional explanation [TJP].
   !-- E.C. modif.
   ! 27-Jul-2010 - Modification according the interface change of GRQTXT.
   !               Code is now close to those of PGPTXT.
   ! 18-Apr-2014 - Rotation of the box is done only if angle is not zero.
   !               Also, added the optional argument 'device' (default is
   !               false) to return the coords in device coordinates
   !               instead of world-coordinates.
   ! 17-Aug-2018 - Renamed 'fjust' in 'hjust' (horizontal justification).
   !               grparse_ps_font and grqtxt are called only one time.
   !               Fixed computation of BBOX from PS fonts (grqtxt cannot
   !               be called for EPS and PDF).
   ! 18-Aug-2018 - Calls to  grlen and grqtxt are removed, and replaced by
   !               a call to grparse_ps_font. 'ratio' is also removed,
   !               assuming it is used only for pixel-based device.
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------

   character(len=12) :: type
   integer :: length

   integer :: i, L
   double precision :: d, xp, yp, bbox(4), factor, cosa, sina, angle_rad
   double precision :: h_min, h_max
   logical :: grparse_ps_font_ok
   logical :: device_e

   !------ end of declarations -- execution starts hereafter  ------

   L = len_trim(text)
   if( L == 0 ) then
      do i = 1, 4
         xbox(i) = x
         ybox(i) = y
      end do
      return
   end if

   ! Convert to device coordinates
   xp = pgxorg(pgid) + x*pgxscl(pgid)
   yp = pgyorg(pgid) + y*pgyscl(pgid)

   call pgqinf( "TYPE", type, length )

   angle_rad = angle*deg_to_rad

   grparse_ps_font_ok = .false.

   if( hjust /= 0.0d0 ) then
      call grparse_ps_font( text(1:L), d, h_min, h_max, draw=.false. )
      grparse_ps_font_ok = .true.
      if( angle == 0.0d0 ) then
         xp = xp - d*hjust
      else
         xp = xp - d*hjust*cos(angle_rad)
         yp = yp - d*hjust*sin(angle_rad)
      end if
   end if

   if( .not. grparse_ps_font_ok ) then
      call grparse_ps_font( text(1:L), d, h_min, h_max, draw=.false. )
   end if
   bbox(1) = 0.0d0
   bbox(2) = h_min
   bbox(3) = d
   bbox(4) = h_max

   if( angle /= 0.0d0 ) then
      ! Rotate bounding box.
      cosa = cos(angle_rad)
      sina = sin(angle_rad)
      xbox(1) = xp + (cosa*bbox(1) - sina*bbox(2))
      ybox(1) = yp + (sina*bbox(1) + cosa*bbox(2))
      xbox(2) = xp + (cosa*bbox(1) - sina*bbox(4))
      ybox(2) = yp + (sina*bbox(1) + cosa*bbox(4))
      xbox(3) = xp + (cosa*bbox(3) - sina*bbox(4))
      ybox(3) = yp + (sina*bbox(3) + cosa*bbox(4))
      xbox(4) = xp + (cosa*bbox(3) - sina*bbox(2))
      ybox(4) = yp + (sina*bbox(3) + cosa*bbox(2))
   else ! angle == 0
      xbox(1) = xp + bbox(1)
      ybox(1) = yp + bbox(2)
      xbox(2) = xp + bbox(1)
      ybox(2) = yp + bbox(4)
      xbox(3) = xp + bbox(3)
      ybox(3) = yp + bbox(4)
      xbox(4) = xp + bbox(3)
      ybox(4) = yp + bbox(2)
   end if

   if( present(device) ) then
      device_e = device
   else
      device_e = .false.
   end if

   if( .not. device_e ) then
      ! Convert bounding box to world coordinates.
      do i = 1, 4
         xbox(i) = (xbox(i) - pgxorg(pgid))/pgxscl(pgid)
         ybox(i) = (ybox(i) - pgyorg(pgid))/pgyscl(pgid)
      end do
   end if

end subroutine
