! PGQCS  -- Inquire character height in a variety of units

subroutine PGQCS( units, xch, ych )

   integer, intent(in) :: units
   double precision :: xch, ych

   ! Return the current MFPLOT character height in a variety of units.
   ! This routine provides facilities that are not available via PGQCH.
   ! Use PGQCS if the character height is required in units other than
   ! those used in PGSCH.
   !
   ! The MFPLOT "character height" is a dimension that scales with the
   ! size of the view surface and with the scale-factor specified with
   ! routine PGSCH. The default value is 1/40th of the height or width
   ! of the view surface (whichever is less); this value is then
   ! multiplied by the scale-factor supplied with PGSCH.
   !
   ! Arguments:
   !  UNITS  (input)  : Used to specify the units of the output value:
   !                    UNITS = 0 : normalized device coordinates
   !                    UNITS = 1 : inches
   !                    UNITS = 2 : millimeters
   !                    UNITS = 3 : pixels
   !                    UNITS = 4 : world coordinates
   !                    Other values give an error message, and are
   !                    treated as 0.
   !  XCH    (output) : The character height for text written with a
   !                    vertical baseline.
   !  YCH    (output) : The character height for text written with
   !                    a horizontal baseline (the usual case).
   !
   ! The character height is returned in both XCH and YCH.
   !
   ! If UNITS=1 or UNITS=2, XCH and YCH both receive the same value.
   !
   ! If UNITS=3, XCH receives the height in horizontal pixel units, and YCH
   ! receives the height in vertical pixel units; on devices for which the
   ! pixels are not square, XCH and YCH will be different.
   !
   ! If UNITS=4, XCH receives the height in horizontal world coordinates
   ! (as used for the x-axis), and YCH receives the height in vertical
   ! world coordinates (as used for the y-axis). Unless special care has
   ! been taken to achieve equal world-coordinate scales on both axes, the
   ! values of XCH and YCH will be different.
   !
   ! If UNITS=0, XCH receives the character height as a fraction of the
   ! horizontal dimension of the view surface, and YCH receives the
   ! character height as a fraction of the vertical dimension of the view
   ! surface.
   !--
   ! 15-Oct-1992 - New routine [MCS].
   !  4-Dec-1992 - Added more explanation [TJP].
   !  5-Sep-1995 - Add UNITS=4; correct error for non-square pixels [TJP].
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   ! 16-Apr-2020 - Modified to take into account suppression of panels [EC].
   !-----------------------------------------------------------------------

   double precision :: ratio
   ! Conversion factor inches -> mm
   double precision, parameter :: intomm = 25.4

   !------ end of declarations -- execution starts hereafter  ------

   ratio = pgypin(pgid)/pgxpin(pgid)

   ! Return the character height in the required units.

   if( units == 1 ) then ! Inches.
      xch = pgysp(pgid)/pgxpin(pgid)
      ych = xch
   else if( units == 2 ) then ! Millimeters.
      xch = pgysp(pgid)/pgxpin(pgid) * intomm
      ych = xch
   else if( units == 3 ) then ! Pixels.
      xch = pgysp(pgid)
      ych = pgysp(pgid)*ratio
   else if( units == 4 ) then ! World coordinates.
      xch = pgysp(pgid)/pgxscl(pgid)
      ych = pgysp(pgid)*ratio/pgyscl(pgid)
   else ! Normalized device coords, or unknown.
      xch = pgysp(pgid)/grxmxa(grcide)
      ych = pgysp(pgid)*ratio/grymxa(grcide)
      if( units /= 0 ) call grwarn('Invalid "UNITS" argument in PGQCS.')
   end if

end subroutine
