! PGPTXT -- Write text at arbitrary position and angle

subroutine PGPTXT( x, y, angle, hjust, vjust, text,                     &
                   to_be_trimmed, clipping, rxbox, rybox, pix_voffset )

   double precision,  intent(in)            :: x, y, angle, hjust, vjust
   character(len=*),  intent(in)            :: text
   logical,           intent(in),  optional :: to_be_trimmed
   logical,           intent(in),  optional :: clipping
   double precision,  intent(out), optional :: rxbox(4), rybox(4)
   logical,           intent(in),  optional :: pix_voffset

   ! Primitive routine for drawing text. The text may be drawn at any
   ! angle with the horizontal, and may be centered or left- or right-
   ! justified at a specified position. Text is drawn using the current
   ! values of attributes color-index, line-width, character-height, and
   ! character-font.  Text is subject to clipping or not at the edge of
   ! the window, according to the value of the optional argument.
   !
   ! Arguments:
   !  X, Y   (input)  : world coordinates. The string is drawn with the
   !                    baseline of all the characters passing through
   !                    point (X,Y); the positioning of the string along
   !                    this line is controlled by argument HJUST.
   !  ANGLE  (input)  : angle, in degrees, that the baseline is to make
   !                    with the horizontal, increasing counter-clockwise
   !                    (0.0 is horizontal).
   !  HJUST  (input)  : controls horizontal justification of the string.
   !                    If HJUST = 0.0, the string will be left-justified
   !                    at the point (X,Y); if HJUST = 0.5, it will be
   !                    centered, and if HJUST = 1.0, it will be right
   !                    justified. [Other values of HJUST give other
   !                    justifications.]
   !  VJUST  (input)  : controls vertical justification of the string.
   !                    If VJUST = 0.0, the string will be bottom-justified
   !                    at the point (X,Y); if VJUST = 0.5, it will be
   !                    centered, and if VJUST = 1.0, it will be top
   !                    justified. [Other values of HJUST give other
   !                    justifications.]
   !  TEXT   (input)  : the character string to be plotted.
   !--
   ! (2-May-1983)
   ! 31-Jan-1985 - Convert to Fortran-77 standard...
   ! 13-Feb-1988 - Correct a PGBBUF/PGEBUF mismatch if string is blank.
   ! 16-Oct-1993 - Erase background of opaque text.
   !-- E.C. modif.
   ! 08-Jul-2010 - Optimize some work (avoids computing some expressions
   !               if HJUST or ANGLE = 0.0).
   !             - Quick return if string length is zero.
   ! 13-Jul-2010 - Added support of PS Font (calling another version
   !               of GRLEN and GRTEXT).
   ! 25-Dec-2013 - Added an optional arg. to prevent the trimming of the
   !               string.
   ! 09-Mar-2014 - Added the optional arg. 'clipping' to force the clipping
   !               of some text.
   ! 14-Mar-2014 - Added the optional args. 'rxbox', 'rybox' to get the
   !               rectangular (and perhaps inclined) bounding box.
   ! 25-Mar-2014 - Renamed 'fjust' in 'hjust' (horizontal justification)
   !               added 'vjust' for vertical justification.
   !               An empirical correction has been added for computing
   !               the character height from 'grqtxt()'.
   !               Keep this correction to 20 instead of 6!
   ! 28-Mar-2014 - Added the optional arg 'pix_voffset' for a better reading
   !               when the bottom (vjust=0) or the top (vjust=1) character
   !               bbox is in contact with a drawn line.
   !               Works only with angle = 0.
   !  2-Jul-2018 - Added support for PDF. Actually PDF uses also PS fonts,
   !               so just few tests added.
   ! 14-Aug-2018 - Calls now GRTEXT_XFT to obtain antialiased character
   !               display in the X11 window (libXft).
   ! 17-Aug-2018 - grparse_ps_font and grqtxt are called only one time.
   ! 18-Aug-2018 - Calls to  grlen and grqtxt are removed, and replaced by
   !               a call to grparse_ps_font. 'ratio' is also removed,
   !               assuming it is used only for pixel-based device.
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !  8-Apr-2021 - Use GRXORG/GRYORG instead of PGXORG/PGYORG because the
   !               last couple is not update during the Pan mode.
   !  9-Nov-2021 - Idem above, for BBOX computation as well.
   ! 24-Nov-2021 - Take care about of 'text backgound color' during clipping.
   !-----------------------------------------------------------------------

   character(len=12) :: type
   integer :: length

   integer :: ci, i, L
   double precision :: d, xp, yp, factor, cosa, sina, angle_rad
   double precision :: xbox(4), ybox(4), bbox(4)
   double precision :: h_min, h_max, h
   logical :: grparse_ps_font_ok

   logical :: trimmed, clipped, clipping_disabled, pixvoffset

   double precision :: rbuf(1)
   integer :: ibuf(1), lchr
   character(len=1) :: chr

   !------ end of declarations -- execution starts hereafter  ------

   if( present(to_be_trimmed) ) then
      trimmed = to_be_trimmed
   else
      trimmed = .true.
   end if

   ! grtrim count the characters number, not including trailing blanks
   if( trimmed ) then
      L = len_trim(text)
   else
      L = len(text)
   end if
   if( L == 0 ) return

   if( present(clipping) ) then
      clipped = clipping
   else
      clipped = .false.
   end if

   if( present(pix_voffset) ) then
      pixvoffset = pix_voffset
   else
      pixvoffset = .false.
   end if

   ! Convert to device coordinates.
   xp = GRXORG(grcide) + x*GRXSCL(grcide)
   yp = GRYORG(grcide) + y*GRYSCL(grcide)

   call pgqinf( "TYPE", type, length )

!### TODO: pas un très bon emplacement, mais bon... il faudrait le mettre
!          plus haut, puisqu'on DOIT absolument lire ces métriques (X11,
!          EPS, PDF, tous les devices possibles quoi !)
   ! on lit les fichiers AFM (Adobe Font Metrics) des huit polices
   ! utilisées pour le texte. (tables limitées à 22:255)
   if( .not. MF_AFM_READ ) then
      ! on remplit des tableaux définis dans "grpckg1.inc"
      call gr_read_AFM_files()
      MF_AFM_READ = .true.
   end if
!### TODO 1: pourquoi ne pas refermer aussitôt ? cela évite d'employer de
!            nombreux FILE_UNIT
   ! la fermeture des fichiers AFM s'effectuera dans 'msExitFgl()'

   grparse_ps_font_ok = .false.

   angle_rad = angle*deg_to_rad

   if( hjust /= 0.0d0 ) then
      call grparse_ps_font( text(1:L), d, h_min, h_max, draw=.false. )
      grparse_ps_font_ok = .true.
      if( angle == 0.0d0 ) then
         xp = xp - d*hjust
      else
         xp = xp - d*hjust*cos(angle_rad)
         yp = yp - d*hjust*sin(angle_rad)
      end if
   end if

   if( vjust /= 0.0d0 ) then
      if( .not. grparse_ps_font_ok ) then
         call grparse_ps_font( text(1:L), d, h_min, h_max, draw=.false. )
         grparse_ps_font_ok = .true.
      end if
      h = h_max - h_min
      if( h < 0.0d0 ) then
         print *, "(pgptxt:) Internal ERROR: when computing 'h' from PS_FONT,"
         print *, "          it should be a positive real number!"
         pause "only for debugging purpose"
         stop
      end if

      ! adding a shift for better reading when character string is along
      ! a drawn line
      if( angle == 0.0d0 ) then
         if( pixvoffset ) then
            h = h + 4.0d0
         end if
         yp = yp - h*vjust
         if( pixvoffset ) then
            yp = yp + 2.0d0
         end if
      else
         xp = xp + h*vjust*sin(angle_rad)
         yp = yp - h*vjust*cos(angle_rad)
      end if
   else ! vjust = 0
      if( pixvoffset ) then
         if( angle == 0.0d0 ) then
            yp = yp + 2.0d0
         end if
      end if
   end if

   ! from Muesli 2020-02-11, bbox of the string is always computed, in order
   ! to decide of an eventual quick return
   if( type == 'EPS' .or. type == 'PDF' ) then
      if( .not. grparse_ps_font_ok ) then
         call grparse_ps_font( text(1:L), d, h_min, h_max, draw=.false. )
      end if
   else
      if( .not. grparse_ps_font_ok ) then
         call grparse_ps_font( text(1:L), d, h_min, h_max, draw=.false. )
      end if
   end if
   bbox(1) = 0.0
   bbox(2) = h_min
   bbox(3) = d
   bbox(4) = h_max
   if( angle /= 0.0 ) then
      ! Rotate bounding box.
      cosa = cos(angle_rad)
      sina = sin(angle_rad)
      xbox(1) = xp + (cosa*bbox(1) - sina*bbox(2))
      ybox(1) = yp + (sina*bbox(1) + cosa*bbox(2))
      xbox(2) = xp + (cosa*bbox(1) - sina*bbox(4))
      ybox(2) = yp + (sina*bbox(1) + cosa*bbox(4))
      xbox(3) = xp + (cosa*bbox(3) - sina*bbox(4))
      ybox(3) = yp + (sina*bbox(3) + cosa*bbox(4))
      xbox(4) = xp + (cosa*bbox(3) - sina*bbox(2))
      ybox(4) = yp + (sina*bbox(3) + cosa*bbox(2))
   else ! angle == 0
      xbox(1) = xp + bbox(1)
      ybox(1) = yp + bbox(2)
      xbox(2) = xp + bbox(1)
      ybox(2) = yp + bbox(4)
      xbox(3) = xp + bbox(3)
      ybox(3) = yp + bbox(4)
      xbox(4) = xp + bbox(3)
      ybox(4) = yp + bbox(2)
   end if

   if( clipped ) then
      ! Quick return if possible (using device coordinates); very efficient
      ! for interactive zoom, when a redraw of all grobjs is made during
      ! pan or zoom...
      if( maxval(xbox) < grxmin(grcide) ) return
      if( minval(xbox) > grxmax(grcide) ) return
      if( maxval(ybox) < grymin(grcide) ) return
      if( minval(ybox) > grymax(grcide) ) return
   end if

   ! Convert bounding box back to world coordinates.
      do i = 1, 4
         xbox(i) = (xbox(i)-GRXORG(pgid))/GRXSCL(pgid)
         ybox(i) = (ybox(i)-GRYORG(pgid))/GRYSCL(pgid)
      end do

   !============ display of the string and its background ===========
   call pgbbuf()

   if( type == 'EPS' ) then

      if( .not. clipped .and. .not. gr_text_in_legend ) then
         call EPS_no_clip_beg()
      end if

      ! pgtbci: text background color index
      if( pgtbci(pgid) >= 0 ) then
         call grqci( ci ) ! save color index
         call grsci( pgtbci(pgid) )
         call grfa( 4, xbox, ybox )
         call grsci( ci ) ! restore color index
      end if

      call grtext_ps_font( angle, xp, yp, text(1:L) )

      if( .not. clipped .and. .not. gr_text_in_legend ) then
         call EPS_no_clip_end()
      end if

   else if( type == 'PDF' ) then

      if( .not. clipped .and. .not. gr_text_in_legend ) then
         call PDF_no_clip_beg()
      end if

      ! pgtbci: text background color index
      if( pgtbci(pgid) >= 0 ) then
         call grqci( ci ) ! save color index
         call grsci( pgtbci(pgid) )
         call grfa( 4, xbox, ybox )
         call grsci( ci ) ! restore color index
      end if

      if( .not. clipped ) then
         call grqci( ci ) ! request last color index
         call grsci( ci ) ! force color selection after 'PDF_no_clip_beg'
                          ! which do a 'restore', which in turn select an
                          ! out-of-date color.
      end if

      call grtext_pdf_font( angle, xp, yp, text(1:L) )

      if( .not. clipped .and. .not. gr_text_in_legend ) then
         call PDF_no_clip_end()
      end if

   else! X11 or NULL device

      clipping_disabled = .false.

      ! pgtbci: text background color index
      if( pgtbci(pgid) >= 0 ) then
         call grqci( ci ) ! save color index
         call grsci( pgtbci(pgid) )

         if( .not. clipped ) then
            if( CLIPPING_IN_AXES ) then
               ibuf(1) = 0 ! no clipping
               call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr ) ! set clipping
               clipping_disabled = .true.
            end if
         end if

         call grfa( 4, xbox, ybox )

         call grsci( ci ) ! restore color index
      end if

      call grtext_xft( angle, xp, yp, text(1:L), clipping=clipped )

      if( clipping_disabled ) then
         ! restore the clipping, which should be the default
         ibuf(1) = 1 ! clipping at viewport
         call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr ) ! set clipping
      end if

   end if

   call pgebuf()
   !========================= end of display ========================

   if( present(rxbox) .and. present(rybox) ) then
      rxbox(1:4) = xbox(1:4)
      rybox(1:4) = ybox(1:4)
   end if

end subroutine
