! PGPOLY -- Draw a polygon, using fill-area attributes

subroutine PGPOLY( n, xpts, ypts, clipping )

   integer,             intent(in)            :: n
   double precision,    intent(in)            :: xpts(*), ypts(*)
   logical,             intent(in),  optional :: clipping

   ! Fill-area primitive routine: shade the interior of a closed polygon
   ! in the current window.  The action of this routine depends on the
   ! setting of the Fill-Area Style attribute (see PGSFS).
   ! The polygon is clipped or not at the edge of the window, according
   ! to the value of the optional argument. The pen position is changed
   ! to (XPTS(1),YPTS(1)) in world coordinates (if N > 1).
   !
   ! Arguments:
   !  N      (input)  : number of points defining the polygon; the
   !                    line consists of N straight-line segments,
   !                    joining points 1 to 2, 2 to 3,... N-1 to N, N to 1.
   !                    N should be greater than 2 (if it is 2 or less,
   !                    nothing will be drawn).
   !  XPTS   (input)  : world x-coordinates of the vertices.
   !  YPTS   (input)  : world y-coordinates of the vertices.
   !                    Note: the dimension of arrays XPTS and YPTS must be
   !                    greater than or equal to N.
   ! CLIPPING (input) : boolean indicating if the clipping must be done or not.
   !--
   ! 21-Nov-1983 - [TJP].
   ! 16-Jul-1984 - Revised to shade polygon with GRFA [TJP].
   ! 21-Oct-1985 - Test PGFAS [TJP].
   ! 25-Nov-1994 - Implement clipping [TJP].
   ! 13-Jan-1994 - Fix bug in clipping [TJP].
   !  6-Mar-1995 - Add support for fill styles 3 and 4 [TJP].
   ! 12-Sep-1995 - Fix another bug in clipping [TJP].
   !  3-Jan-2020 - Added the optional arg. 'clipping' to authorize
   !               the drawing out of the window.
   !               Otherwise, replace multiple calls of pgbbuf since it
   !               already occurs.
   ! 20-Jan-2020 - Increase max nb of side for the clipped polygon, from 1000
   !               to 5000 (never found such a limitation, but "on ne sait
   !               jamais" !).
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   ! 14-Mar-2021 - The clipping is now done in the drivers.
   !  8-Apr-2021 - Fix the exception to clipping (not yet done).
   !-----------------------------------------------------------------------

   integer :: i
   logical :: clipped, clipping_disabled

   double precision :: rbuf(1)
   integer :: ibuf(1), lchr
   character(len=1) :: chr

   !------ end of declarations -- execution starts hereafter  ------

   if( n < 1 ) return

   if( present(clipping) ) then
      clipped = clipping
   else
      clipped = .true.
   end if

   clipping_disabled = .false.
   if( .not. clipped ) then
      if( CLIPPING_IN_AXES ) then
         ibuf(1) = 0 ! no clipping
         call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr ) ! set clipping
         clipping_disabled = .true.
      end if
   end if

   call pgbbuf()

   if( pgfas(pgid) == 2 .or. n < 3 ) then
      ! Outline style, or polygon of less than 3 vertices.
      call grmova( xpts(n), ypts(n) )
      do i = 1, n
         call grlina( xpts(i), ypts(i) )
      end do
   else if( pgfas(pgid) == 3 ) then
      ! Hatched style.
      call pghtch( n, xpts, ypts, 0.0d0 )
   else if( pgfas(pgid) == 4 ) then
      ! Hatched style.
      call pghtch( n, xpts, ypts, 0.0d0 )
      call pghtch( n, xpts, ypts, 90.0d0 )
   else
      ! Filled style
      call grfa( n, xpts, ypts )
   end if

   ! Set the current pen position.
   call grmova( xpts(1), ypts(1) )

   call pgebuf()

   if( clipping_disabled ) then
      ! restore the clipping, which should be the default
      ibuf(1) = 1 ! clipping at viewport
      call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr ) ! set clipping
   end if

end subroutine
