! PGOPEN -- Open a graphics device

integer function PGOPEN( device )

   character(len=*) :: device

   ! Open a graphics device for MFPLOT output (non X11 devices). If the
   ! device is opened successfully, it becomes the selected device to
   ! which graphics output is directed until another device is selected
   ! with PGSLCT or the device is closed with PGCLOS.
   !
   ! The value returned by PGOPEN should be tested to ensure that
   ! the device was opened successfully, e.g.,
   !
   !       ISTAT = PGOPEN('plot.ps/PS')
   !       IF( ISTAT <= 0 ) STOP
   !
   ! The DEVICE argument is a character constant or variable; its value
   ! should be one of the following:
   !
   ! (1) A complete device specification of the form 'device/type' or
   !     'file/type', where 'type' is one of the allowed MFPLOT device
   !     types (installation-dependent) and 'device' or 'file' is the
   !     name of a graphics device or disk file appropriate for this type.
   !     The 'device' or 'file' may contain '/' characters; the final
   !     '/' delimits the 'type'. If necessary to avoid ambiguity,
   !     the 'device' part of the string may be enclosed in double
   !     quotation marks.
   ! (2) A device specification of the form '/type', where 'type' is one
   !     of the allowed MFPLOT device types. MFPLOT supplies a default
   !     file or device name appropriate for this device type.
   ! (3) A device specification with '/type' omitted; in this case
   !     the type is taken from the environment variable MFPLOT_TYPE,
   !     if defined (e.g., setenv MFPLOT_TYPE PS). Because of possible
   !     confusion with '/' in file-names, omitting the device type
   !     in this way is not recommended.
   ! (4) A blank string (' '); in this case, PGOPEN will use the value
   !     of environment variable MFPLOT_DEV as the device specification,
   !     or '/NULL' if the environment variable is undefined.
   !
   ! The device type is case-insensitive (e.g., '/ps' and '/PS' are
   ! equivalent). The device or file name may be case-sensitive in some
   ! operating systems.
   !
   ! Examples of valid DEVICE arguments:
   !
   ! (1)  'plot.ps/ps', 'dir/plot.ps/ps', '"dir/plot.ps"/ps',
   !      'user:[tjp.plots]plot.ps/PS'
   ! (2)  '/ps'      (MFPLOT interprets this as 'pgplot.ps/ps')
   ! (3)  'plot.ps'  (if MFPLOT_TYPE is defined as 'ps', MFPLOT
   !                  interprets this as 'plot.ps/ps')
   ! (4)  '   '      (if MFPLOT_DEV is defined)
   ! (5)  '?  '
   ! (6)  '?Device specification for MFPLOT: '
   !
   ! [This routine was added to PGPLOT in Version 5.1.0. Older programs
   ! use PGBEG instead.]
   !
   ! Returns:
   !  PGOPEN          : returns either a positive value, the
   !                    identifier of the graphics device for use with
   !                    PGSLCT, or a 0 or negative value indicating an
   !                    error. In the event of error a message is
   !                    written on the standard error unit.
   ! Arguments:
   !  DEVICE  (input) : the 'device specification' for the plot device
   !                    (see above).
   !--
   ! 22-Dec-1995 - New routine [TJP].
   ! 14-May-1996 - Device '? ' should not give a blank prompt [TJP].
   ! 15-Jul-2007 - PGSLW requires now a real argument [EC]
   !  4-Jul-2015 - Call to PGSCLP is delayed after PGVSTD, else NaN values
   !               stop the program if the user chose to launch it with
   !               Floating-Point Exceptions enabled.
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   ! 10-Apr-2020 - Interactive prompt with the user is removed [EC].
   ! 11-Apr-2020 - Removed call to grsize: things have already been set in
   !               gropen_ec [EC].
   ! 16-Apr-2020 - Modified to take into account suppression of panels [EC].
   !--------------------------------------------------------------------

   integer :: deftyp, L, ic1
   integer :: ldefde, unit, istat
   character(len=512) :: defdev
   character(len=20) :: defstr

   !------ end of declarations -- execution starts hereafter  ------

   ! Initialize MFPLOT if necessary.
   call pginit

   ! Default device and type
   defdev = "/NULL"
   ldefde = 5
   deftyp = 0

   if( device == " " ) then
      ! Blank device string: use default device and type.
      istat = gropen( deftyp, unit, defdev(1:ldefde), pgid )
   else
      istat = gropen( deftyp, unit, device, pgid )
   end if

   ! Failed to open plot file?
   if( istat /= 1 ) then
      pgopen = - 1
      return
   end if

   ! Success: determine device characteristics.
   pgdevs(pgid) = 1
   pgxpin(pgid) = grpxpi(grcide)
   pgypin(pgid) = grpypi(grcide)

   pgblev(pgid) = 0

   call grgenv( "DEFLATE_A85", defstr, L )
   if( L > 0 ) then
      if( defstr(1:1) /= "1" ) then
         MF_DEFLATE_TO_A85 = .false.
      end if
   end if

   ! Set default attributes.
   call grsci(1)
   call grslw(1.0d0)
   call grsls(1)
   call pgsch(1.0d0)
   call pgsfs(1)
   call pgsah(1, 45.0d0, 0.3d0)
   call pgstbg(-1)
   call pgshs(45.0d0, 1.0d0, 0.0d0)

   ! Set the default range of color indices available for images (16 to
   ! device maximum, if device maximum >= 16; otherwise not possible).
   call grqcol(ic1, pgmxci(pgid))
   pgmnci(pgid) = 16
   if( pgmxci(pgid) < 16 ) pgmxci(pgid) = 0

   ! Select linear transfer function.
   pgitf(pgid) = 0

   pgopen = pgid

end function
