! PGNUMB -- Convert a number into a plottable character string

subroutine PGNUMB( mm, pp, form, string, nc )

   integer :: mm, pp, form
   character(len=*) :: string
   integer :: nc

   ! This routine converts a number into a decimal character
   ! representation. To avoid problems of floating-point roundoff, the
   ! number must be provided as an integer (MM) multiplied by a power of 10
   ! (10**PP).  The output string retains only significant digits of MM,
   ! and will be in either integer format (123), decimal format (0.0123),
   ! or exponential format (1.23x10**5). Standard escape sequences \u, \d
   ! raise the exponent and \x is used for the multiplication sign.
   ! This routine is used by PGBOX to create numeric labels for a plot.
   !
   ! Formatting rules:
   !   (a) Decimal notation (FORM=1):
   !       - Trailing zeros to the right of the decimal sign are
   !         omitted
   !       - The decimal sign is omitted if there are no digits
   !         to the right of it
   !       - When the decimal sign is placed before the first digit
   !         of the number, a zero is placed before the decimal sign
   !       - The decimal sign is a period (.)
   !       - No spaces are placed between digits (ie digits are not
   !         grouped in threes as they should be)
   !       - A leading minus (-) is added if the number is negative
   !   (b) Exponential notation (FORM=2):
   !       - The exponent is adjusted to put just one (non-zero)
   !         digit before the decimal sign
   !       - The mantissa is formatted as in (a), unless its value is
   !         1 in which case it and the multiplication sign are omitted
   !       - If the power of 10 is not zero and the mantissa is not
   !         zero, an exponent of the form \x10\u[-]nnn is appended,
   !         where \x is a multiplication sign (cross), \u is an escape
   !         sequence to raise the exponent, and as many digits nnn
   !         are used as needed
   !   (c) Automatic choice (FORM=0):
   !         Decimal notation is used if the absolute value of the
   !         number is less than 10000 or greater than or equal to
   !         0.01. Otherwise exponential notation is used.
   !
   ! Arguments:
   !  MM     (input)
   !  PP     (input)  : the value to be formatted is MM*10**PP.
   !  FORM   (input)  : controls how the number is formatted:
   !                    FORM = 0 -- use either decimal or exponential
   !                    FORM = 1 -- use decimal notation
   !                    FORM = 2 -- use exponential notation
   !  STRING (output) : the formatted character string, left justified.
   !                    If the length of STRING is insufficient, a single
   !                    asterisk is returned, and NC=1.
   !  NC     (output) : the number of characters used in STRING:
   !                    the string to be printed is STRING(1:NC).
   !--
   ! 23-Nov-1983 - Creation [?].
   !  9-Feb-1988 - Use temporary variable to avoid illegal character
   !               assignments; remove non-standard DO loops [TJP].
   ! 15-Dec-1988 - More corrections of the same sort [TJP].
   ! 27-Nov-1991 - Change code for multiplication sign [TJP].
   ! 23-Jun-1994 - Partial implementation of FORM=1 and 2 [TJP].
   ! 18-Jan-2010 - Increase char length of work string from 20 to 40 [EC].
   !-----------------------------------------------------------------------

   character :: bslash
   character(len=2) :: times, up, down
   character(len=40) :: wexp, temp
   character(len=42) :: work
   integer :: m, p, nd, i, j, k, nbp
   logical :: minus

   !------ end of declarations -- execution starts hereafter  ------

   ! Define backslash (escape) character and escape sequences.
   bslash = char(92)
   times  = bslash // 'x'
   up     = bslash // 'u'
   down   = bslash // 'd'

   ! Zero is always printed as "0".
   if( mm == 0 ) then
      string = '0'
      nc = 1
      return
   end if

   ! If negative, make a note of that fact.
   minus = mm < 0
   m = abs(mm)
   p = pp

   ! Convert M to a left-justified digit string in WORK. As M is a
   ! positive integer, it cannot use more than 10 digits (2147483647).
   j = 10
10 if( m /= 0 ) then
      k = mod(m,10)
      m = m/10
      work(j:j) = char(ichar('0')+k)
      j = j-1
      goto 10
   end if
   temp = work(j+1:)
   work = temp
   nd = 10 - j

   ! Remove right-hand zeros, and increment P for each one removed.
   ! ND is the final number of significant digits in WORK, and P the
   ! power of 10 to be applied. Number of digits before decimal point
   ! is NBP.
20 if( work(nd:nd) == '0' ) then
      nd = nd - 1
      p = p + 1
      goto 20
   end if
   nbp = nd + min(p,0)

   if( (p>=0) .and. ((form==0 .and. p+nd<=4) .or.                       &
                     (form==1 .and. p+nd<=10))) then
      ! Integral numbers of 4 or less digits are formatted as such.
      do i = 1, p
         nd = nd+1
         work(nd:nd) = '0'
      end do
      p = 0

   else if( form/=2 .and. nbp>=1 .and. nbp<=4 .and. nbp<nd ) then
      ! If NBP is 4 or less, simply insert a decimal point in the right place.
      temp = work(nbp+1:nd)
      work(nbp+2:nd+1) = temp
      work(nbp+1:nbp+1) = '.'
      nd = nd + 1
      p = 0

   else
      ! Otherwise insert a decimal point after the first digit, and adjust P.
      p = p + nd - 1
      if( form/=2 .and. p==-1 ) then
         temp = work
         work = '0'//temp
         nd = nd + 1
         p = 0
      else if( form/=2 .and. p==-2 ) then
         temp = work
         work = '00'//temp
         nd = nd + 2
         p = 0
      end if
      if( nd > 1 ) then
         temp = work(2:nd)
         work(3:nd+1) = temp
         work(2:2) = '.'
         nd = nd + 1
      end if
   end if

   ! Add exponent if necessary.
   if( p /= 0 ) then
      work(nd+1:nd+6) = times//'10'//up
      nd = nd + 6
      if( p < 0 ) then
         p = -p
         nd = nd + 1
         work(nd:nd) = '-'
      end if
      j = 10
40    if( p /= 0 ) then
         k = mod(p,10)
         p = p/10
         wexp(j:j) = char(ichar('0')+k)
         j = j - 1
         goto 40
      end if
      work(nd+1:) = wexp(j+1:10)
      nd = nd + 10 - j
      if( work(1:3)=='1'//times ) then
         temp = work(4:)
         work = temp
         nd = nd - 3
      end if
      work(nd+1:nd+2) = down
      nd = nd + 2
   end if

   ! Add minus sign if necessary and move result to output.
   if( minus ) then
     temp = work(1:nd)
     string = '-'//temp
     nc = nd + 1
   else
     string = work(1:nd)
     nc = nd
   end if

   ! Check result fits.
   if( nc > len(string) ) then
      string = '*'
      nc = 1
   end if

end subroutine
