! PGMTXT -- Write text at position relative to viewport

subroutine PGMTXT( side, disp, coord, hjust, text )

   character(len=*), intent(in) :: side, text
   double precision, intent(in) :: disp, coord, hjust

   ! Write text at a position specified relative to the viewport (outside
   ! or inside).  This routine is useful for annotating graphs. It is used
   ! by routine PGLAB.  The text is written using the current values of
   ! attributes color-index, line-width, character-height, and
   ! character-font.
   !
   ! Arguments:
   !  SIDE   (input)  : must include one of the characters 'B', 'L', 'T',
   !                    or 'R' signifying the Bottom, Left, Top, or Right
   !                    margin of the viewport. If it includes 'LV' or
   !                    'RV', the string is written perpendicular to the
   !                    frame rather than parallel to it.
   !  DISP   (input)  : the displacement of the character string from the
   !                    specified edge of the viewport, measured outwards
   !                    from the viewport in units of the character
   !                    height. Use a negative value to write inside the
   !                    viewport, a positive value to write outside.
   !  COORD  (input)  : the location of the character string along the
   !                    specified edge of the viewport, as a fraction of
   !                    the length of the edge.
   !  FJUST  (input)  : controls justification of the string parallel to
   !                    the specified edge of the viewport. If
   !                    FJUST = 0.0, the left-hand end of the string will
   !                    be placed at COORD; if JUST = 0.5, the center of
   !                    the string will be placed at COORD; if JUST = 1.0,
   !                    the right-hand end of the string will be placed at
   !                    at COORD. Other values between 0 and 1 give inter-
   !                    mediate placing, but they are not very useful.
   !  TEXT   (input) :  the text string to be plotted. Trailing spaces are
   !                    ignored when justifying the string, but leading
   !                    spaces are significant.
   !
   !--
   ! 18-Apr-1983
   ! 15-Aug-1987 - Fix BBUF/EBUF error.
   ! 27-Aug-1987 - Fix justification error if XPERIN /= YPERIN.
   ! 05-Sep-1989 - Change so that DISP has some effect for 'RV' and
   !               'LV' options [nebk]
   ! 16-Oct-1993 - Erase background of opaque text.
   !-- E.C. modif.
   ! 13-Jul-2010 - Added support of PS Font (calling another version
   !               of GRLEN and GRTEXT).
   ! 16-Aug-2018 - Optimize the computation of expressions involved when
   !               rotation is required. Never computes cos or sin, since
   !               angle is always 0 or 90 degrees.
   ! 17-Aug-2018 - grparse_ps_font is called only one time.
   ! 18-Aug-2018 - grlen call is removed, and replaced by a call to
   !               grparse_ps_font.
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   ! 16-Apr-2020 - Modified to take into account suppression of panels.
   !  9-Nov-2021 - Background drawing disabled (useless and leads to bugs).
   !-----------------------------------------------------------------------

   character(len=12) :: type
   integer :: length

   integer :: ci, i, L
   double precision :: angle, d_len, d, x, y, factor, ratio
   double precision :: xbox(4), ybox(4), bbox(4)
   character(len=20) :: test
   double precision :: h_min, h_max
   logical :: grparse_ps_font_ok

   !------ end of declarations -- execution starts hereafter  ------

   L = len_trim( text )
   if( L == 0 ) return

   call pgqinf( "TYPE", type, length )

!### TODO: pas un très bon emplacement, mais bon... il faudrait le mettre
!          plus haut, puisqu'on DOIT absolument lire ces métriques (X11,
!          EPS, PDF, tous les devices possibles quoi !)
   ! Read the AFM files (Adobe Font Metrics) of the eight fonts
   ! used for text (tables limited to 22:255 subrange).
   if( .not. MF_AFM_READ ) then
      ! Filling the tables of 'grpckg1.inc'.
      call gr_read_AFM_files()
      MF_AFM_READ = .true.
   end if
   ! Closing AFM files will be done in 'msExitFgl()'.

   grparse_ps_font_ok = .false.

   d = 0.
   if( hjust /= 0. ) then
      call grparse_ps_font( text(1:L), d_len, h_min, h_max, draw=.false. )
      grparse_ps_font_ok = .true.
   end if
   d = d_len*hjust
   ratio = pgypin(pgid)/pgxpin(pgid)
   call grtoup( test, side )
   if( index(test,'B') /= 0 ) then
      angle = 0.
      x = pgxvp(pgid) + coord*pgxlen(pgid) - d
      y = pgyvp(pgid) - pgysp(pgid)*disp
   else if( index(test,'LV') /= 0 ) then
      angle = 0.
      x = pgxvp(pgid) - pgysp(pgid)*disp - d
      y = pgyvp(pgid) + coord*pgylen(pgid) - 0.3*pgysp(pgid)
   else if( index(test,'L') /= 0 ) then
      angle = 90.
      x = pgxvp(pgid) - pgysp(pgid)*disp
      y = pgyvp(pgid) + coord*pgylen(pgid) - d*ratio
   else if( index(test,'T') /= 0 ) then
      angle = 0.
      x = pgxvp(pgid) + coord*pgxlen(pgid) - d
      y = pgyvp(pgid) + pgylen(pgid) + pgysp(pgid)*disp
   else if( index(test,'RV') /= 0 ) then
      angle = 0.
      x = pgxvp(pgid) + pgxlen(pgid) + pgysp(pgid)*disp - d
      y = pgyvp(pgid) + coord*pgylen(pgid) - 0.3*pgysp(pgid)
   else if( index(test,'R') /= 0 ) then
      angle = 90.
      x = pgxvp(pgid) + pgxlen(pgid) + pgysp(pgid)*disp
      y = pgyvp(pgid) + coord*pgylen(pgid) - d*ratio
   else
      call grwarn( 'Invalid "SIDE" argument in PGMTXT.' )
      return
   end if

   call pgbbuf()

   !### To avoid confusion with the background color (possibly different
   ! from black or white) selected by pgptxt, here the drawing of the
   ! background under is disabled.
   ! Text background should always set with color index zero (i.e. the
   ! default background color), but during Pan and Zoom, we obtain some
   ! strange results: the framebox is sometimes overwritten by very small
   ! white rectangular zones. Ok, the origin (PGXORG, PGYORG) should be
   ! changed during the scroll in the window (because the written pixmap
   ! is bigger), but "ça ne changera rien à l'affaire!".
   ! Anyway, I don't see any circumstances where the drawing of a white
   ! background would be useful...
!    ! Text background color
!    if( .not. grparse_ps_font_ok ) then
!       call grparse_ps_font( text(1:L), d, h_min, h_max, draw=.false. )
!    end if
!    bbox(1) = 0.
!    bbox(2) = h_min
!    bbox(3) = d
!    bbox(4) = h_max
!    if( angle == 90. ) then
!       ! Rotate bounding box.
!       ratio = grpxpi(grcide)/grpypi(grcide)
!       xbox(1) = x - bbox(2)*ratio
!       ybox(1) = y + bbox(1)
!       xbox(2) = x - bbox(4)*ratio
!       ybox(2) = y + bbox(1)
!       xbox(3) = x - bbox(4)*ratio
!       ybox(3) = y + bbox(3)
!       xbox(4) = x - bbox(2)*ratio
!       ybox(4) = y + bbox(3)
!    else if( angle == 0.0 ) then
!       xbox(1) = x + bbox(1)*ratio
!       ybox(1) = y + bbox(2)
!       xbox(2) = x + bbox(1)*ratio
!       ybox(2) = y + bbox(4)
!       xbox(3) = x + bbox(3)*ratio
!       ybox(3) = y + bbox(4)
!       xbox(4) = x + bbox(3)*ratio
!       ybox(4) = y + bbox(2)
!    else
!       call grwarn( 'Invalid "ANGLE" argument in PGMTXT. ' //         &
!                    'Only 0 and 90 degrees are possible!' )
!       pause "only for debugging purpose"
!       stop
!    end if
!    ! Convert bounding box to world coordinates.
!    do i = 1, 4
!       xbox(i) = (xbox(i)-PGXORG(pgid))/PGXSCL(pgid)
!       ybox(i) = (ybox(i)-PGYORG(pgid))/PGYSCL(pgid)
!    end do
!    call grqci( ci )
!    call grsci( 0 ) ! old was pgtbci(pgid)
!    call grfa( 4, xbox, ybox )
!    call grsci( ci )

   if( type == 'EPS' ) then
      call grtext_ps_font( angle, x, y, text(1:L) )
   else if( type == 'PDF' ) then
      call grtext_pdf_font( angle, x, y, text(1:L) )
   else
      call grtext_xft( angle, x, y, text(1:L) )
   end if

   call pgebuf()

end
