! PGLINE -- Draw a polyline (curve defined by line-segments)

subroutine PGLINE( n, xpts, ypts )

   integer,          intent(in) :: n
   double precision, intent(in) :: xpts(*), ypts(*)

   ! Primitive routine to draw a Polyline. A polyline is one or more
   ! connected straight-line segments.  The polyline is drawn using the
   ! current setting of attributes color-index, line-style, and line-width.
   !
   ! Arguments:
   !  N         : number of points defining the line; the line consists
   !              of (N-1) straight-line segments. N should be greater
   !              than 1 (if it is 1 or less, nothing will be drawn).
   !  XPTS      : world x-coordinates of the points.
   !  YPTS      : world y-coordinates of the points.
   !
   ! The dimension of arrays X and Y must be greater than or equal to N.
   ! The "pen position" is changed to (X(N),Y(N)) in world coordinates
   ! (if N > 1).
   !--
   ! 27-Nov-1986
   ! ??-???-200? - Avoid drawing points having NaN coordinates.
   ! 14-Sep-2018 - Take care of dash pattern offset (linestyle arg added)
   !               for EPS and PDF.
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !  5-Apr-2021 - Clipping is now monitored by a higher calling routine.
   !  1-Dec-2021 - Remove linestyle argument, because dashing regularity
   !               is automatically done by the printer device.
   !               Clipping is now done by a lower routine.
   ! 17-Mar-2023 - Fix size of valid_ranges.
   !-----------------------------------------------------------------------

   integer :: i

   logical :: move_ok
   logical, external :: grisnan

   integer, allocatable :: valid_ranges(:)
   logical :: in_range
   integer :: k, k_max, k1, k2, j, n_sub, npts
   character(len=32) :: dev_typ
   logical :: ldummy

   double precision :: rbuf(1)
   integer :: ibuf(5), lchr
   character(len=14) :: chr

   !------ end of declarations -- execution starts hereafter  ------

   if( n < 2 ) return

   if( PRINTING_EPS .or. PRINTING_PDF ) then

      chr = "s" ! short for polyline_start
      lchr = 1
      call grexec( grgtyp, DRAW_POLYLINE, rbuf, ibuf, chr, lchr )

      move_ok = .false.
      do i = 1, n
         if( grisnan(xpts(i)) .or. grisnan(ypts(i)) ) then
            move_ok = .false.
         else
            if( move_ok ) then
               call grlina( xpts(i), ypts(i) )
            else
               call grmova( xpts(i), ypts(i) )
               move_ok = .true.
            end if
         end if
      end do

      chr = "e" ! short for polyline_end
      lchr = 1
      call grexec( grgtyp, DRAW_POLYLINE, rbuf, ibuf, chr, lchr )

   else ! X11 or NULL device

      ! Quick return for the NULL device
      call grqtyp( dev_typ, ldummy )
      if( trim(dev_typ) == "NULL" ) return

      allocate( valid_ranges(n+1) )
      k = 0

      ! First pass to detect valid intervals (i.e. contiguous)
      in_range = .false.
      do i = 1, n
         if( grisnan(xpts(i)) .or. grisnan(ypts(i)) ) then
            if( in_range ) then
               k = k + 1
               valid_ranges(k) = i - 1
               in_range = .false.
            end if
         else
            if( .not. in_range ) then
               k = k + 1
               valid_ranges(k) = i
               in_range = .true.
            end if
         end if
      end do
      if( in_range ) then
         k = k + 1
         valid_ranges(k) = n
      end if
      k_max = k

      if( mod(k_max,2) /= 0 ) then
         print "(/,A)", "(MUESLI pgline:) internal error ***"
         print *, "    -> k_max should be even."
         return
      end if

      if( k_max == 0 ) return

      n_sub = k_max / 2

      call pgbbuf()

      do j = 1, n_sub
         k1 = valid_ranges(2*j-1)
         k2 = valid_ranges(2*j)
         npts = k2 - k1 + 1
         if( npts < 2 ) then
            cycle
         else if( npts == 2 ) then
            call grmova( xpts(k1), ypts(k1) )
            call grlina( xpts(k2), ypts(k2) )
         else
            call pgline2( npts, xpts(k1), ypts(k1) )
         end if
      end do

      call pgebuf()

   end if

end subroutine
