! PGLAB -- Write labels for x-axis, y-axis, and top of plot

subroutine PGLAB( xlbl, xlbl_pos, ylbl, toplbl )

   character(len=*), intent(in) :: xlbl, ylbl, toplbl
   integer,          intent(in) :: xlbl_pos

   ! Write labels outside the viewport.
   !
   ! Arguments:
   !  XLBL     (input) : a label for the x-axis (centered below the
   !                     viewport).
   !  XLBL_POS (input) : position of the above label for the x-axis:
   !                     1=below the box (default); 2=above the box.
   !                     XLBL_POS is not referenced for YLBL only.
   !  YLBL     (input) : a label for the y-axis (centered to the left
   !                     of the viewport, drawn vertically).
   !  TOPLBL   (input) : a label for the entire plot (centered above the
   !                     viewport).
   !--
   ! 11-May-1990 - Remove unnecessary include [TJP].
   ! 10-Apr-2021 - Improve position of strings, according to the character
   !               height of the numerical labels of the axes, which may
   !               be different from that of the descriptive labelling.
   !  6-May-2021 - Modification to take into account the title font size,
   !               which may be now different from labels font size.
   ! 16-Jun-2021 - Light modification to take into account the position
   !               of X Label (either below the box -- the default --, or
   !               above the box).
   ! 17-Jun-2021 - Change in accessing the different font factors.
   !-----------------------------------------------------------------------

   double precision :: factor_a, factor_l

   !------ end of declarations -- execution starts hereafter  ------

   ! Layout correction (difficult to proceed differently...)
   ! (other saved variables, as pgysp, pgchsz or grfac, doesn't contains
   !  the information about a difference between the axis font size and
   !  that of labels -- moreover, win%axis_font_size, win%label_font_size
   !  and win%title_font_size are not reachable here!)
   !
   ! Before calling this routine for drawing X or Y label:
   !        PGSCH is called with the Label Font Factor
   ! but before calling this routine for drawing Title label:
   !        PGSCH is called with the Title Font Factor

   call pgbbuf()

   ! Caution: here pgysp(pgid) is the character height of the label font size.
   ! We must know the axis font size to define correctly the layout.
   ! => Label Font Size = pgysp(pgid)
   ! =>  Axis Font Size = pgysp(pgid)*factor_a
   ! (see also PGENV)
   factor_a = pg_axe_fnt_siz_fct(pgid)/pg_lab_fnt_siz_fct(pgid)

   if( len_trim(xlbl) > 0 ) then
      ! wanted distance:  h_pix = 2*Height_Axis_Font + 0.5*remaining_space
      !                           + 0.5*Y_space
      !
      !  => h_pix = 2*pgysp(pgid)*factor_a + pgysp(pgid) + 0.5*pgysp(pgid)
      !
      !  =>  disp = 1.5 + 2*factor_a    (in terms of char height)
      !
      if( xlbl_pos == 1 ) then
         call pgmtxt( side='B', disp=1.5d0+2.0d0*factor_a,              &
                      coord=0.5d0, hjust=0.5d0, text=xlbl )
      else if( xlbl_pos == 2 ) then
         call pgmtxt( side='T', disp=0.5d0+2.0d0*factor_a,              &
                      coord=0.5d0, hjust=0.5d0, text=xlbl )
      else
         print *, "(pglab:) Internal ERROR: xlbl_pos must not be zero!"
         pause "only for debugging purpose"
         stop
      end if
   end if

   if( len_trim(ylbl) > 0 ) then
      ! wanted distance:  h_pix = 2*Height_Axis_Font + 0.5*remaining_space
      !                           - 0.5*Y_space
      !
      !  => h_pix = 2*pgysp(pgid)*factor_a + pgysp(pgid) - 0.5*pgysp(pgid)
      !
      !  =>  disp = 0.5 + 2*factor_a    (in terms of char height)
      !
      call pgmtxt( side='L', disp=0.5d0+2.0d0*factor_a,                 &
                   coord=0.5d0, hjust=0.5d0, text=ylbl )
   end if

   ! Similar way for the title (during another call)
   ! => Title Font Size = pgysp(pgid)
   ! =>  Axis Font Size = pgysp(pgid)*factor_a
   factor_a = pg_axe_fnt_siz_fct(pgid)/pg_tit_fnt_siz_fct(pgid)
   ! => Label Font Size = pgysp(pgid)*factor_l
   factor_l = pg_lab_fnt_siz_fct(pgid)/pg_tit_fnt_siz_fct(pgid)

   if( len_trim(toplbl) > 0 ) then
      if( xlbl_pos == 1 ) then
         call pgmtxt( side='T', disp=0.5d0+2.0d0*factor_a,              &
                      coord=0.5d0, hjust=0.5d0, text=toplbl )
      else if( xlbl_pos == 2 ) then
         call pgmtxt( side='T', disp=0.5d0+2.0d0*factor_a+2.0d0*factor_l, &
                      coord=0.5d0, hjust=0.5d0, text=toplbl )
      else
         print *, "(pglab:) Internal ERROR: xlbl_pos must not be zero!"
         pause "only for debugging purpose"
         stop
      end if
   end if

   call pgebuf()

end subroutine
