! PGIMAG -- Color image from a 2D data array

subroutine pgimag( a, idim, jdim, i1, i2, j1, j2, a1, a2, tr )

   integer          :: idim, jdim, i1, i2, j1, j2
   double precision :: a(idim,jdim), a1, a2, tr(6)

   ! Draw a color image of an array in current window. The subsection
   ! of the array A defined by indices (I1:I2, J1:J2) is mapped onto
   ! the view surface world-coordinate system by the transformation
   ! matrix TR. The resulting quadrilateral region is clipped at the edge
   ! of the window. Each element of the array is represented in the image
   ! by a small quadrilateral, which is filled with a color specified by
   ! the corresponding array value.
   !
   ! The subroutine uses color indices in the range C1 to C2, which can
   ! be specified by calling PGSCIR before PGIMAG. The default values
   ! for C1 and C2 are device-dependent; these values can be determined by
   ! calling PGQCIR. Note that color representations should be assigned to
   ! color indices C1 to C2 by calling PGSCR before calling PGIMAG. On some
   ! devices (but not all), the color representation can be changed after
   ! the call to PGIMAG by calling PGSCR again.
   !
   ! Array values in the range A1 to A2 are mapped on to the range of
   ! color indices C1 to C2, with array values <= A1 being given color
   ! index C1 and values >= A2 being given color index C2. The mapping
   ! function for intermediate array values can be specified by
   ! calling routine PGSITF before PGIMAG; the default is linear.
   !
   ! On devices which have no available color indices (C1 > C2),
   ! PGIMAG will return without doing anything. On devices with only
   ! one color index (C1=C2), all array values map to the same color
   ! which is rather uninteresting. An image is always "opaque",
   ! i.e., it obscures all graphical elements previously drawn in
   ! the region.
   !
   ! The transformation matrix TR is used to calculate the world
   ! coordinates of the center of the "cell" that represents each
   ! array element. The world coordinates of the center of the cell
   ! corresponding to array element A(I,J) are given by:
   !
   !          X = TR(1) + TR(2)*I + TR(3)*J
   !          Y = TR(4) + TR(5)*I + TR(6)*J
   !
   ! Usually TR(3) and TR(5) are zero -- unless the coordinate
   ! transformation involves a rotation or shear.  The corners of the
   ! quadrilateral region that is shaded by PGIMAG are given by
   ! applying this transformation to (I1-0.5,J1-0.5), (I2+0.5, J2+0.5).
   !
   ! Arguments:
   !  A      (input)  : the array to be plotted.
   !  IDIM   (input)  : the first dimension of array A.
   !  JDIM   (input)  : the second dimension of array A.
   !  I1, I2 (input)  : the inclusive range of the first index
   !                    (I) to be plotted.
   !  J1, J2 (input)  : the inclusive range of the second
   !                    index (J) to be plotted.
   !  A1     (input)  : the array value which is to appear with shade C1.
   !  A2     (input)  : the array value which is to appear with shade C2.
   !  TR     (input)  : transformation matrix between array grid and
   !                    world coordinates.
   !--
   ! 15-Sep-1994 - New routine [TJP].
   ! 21-Jun-1995 - Minor change to header comments [TJP].
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------

   double precision :: PA(6)

   !------ end of declarations -- execution starts hereafter  ------

   ! Check inputs.
   if( i1 < 1 .or. i2 > idim .or. i1 > i2 .or.                          &
       j1 < 1 .or. j2 > jdim .or. j1 > j2) then
      call grwarn('PGIMAG: invalid range I1:I2, J1:J2')
   else if( a1 == a2 ) then
      call grwarn('PGIMAG: foreground level = background level')
   else if( pgmnci(pgid) > pgmxci(pgid) ) then
      call grwarn('PGIMAG: not enough colors available')
   else

      ! Call lower-level routine to do the work.
      call pgbbuf()
      pa(1) = tr(1)*pgxscl(pgid) + pgxorg(pgid)
      pa(2) = tr(2)*pgxscl(pgid)
      pa(3) = tr(3)*pgxscl(pgid)
      pa(4) = tr(4)*pgyscl(pgid) + pgyorg(pgid)
      pa(5) = tr(5)*pgyscl(pgid)
      pa(6) = tr(6)*pgyscl(pgid)
      call grimg0( a, idim, jdim, i1, i2, j1, j2, a1, a2, pa,           &
                   pgmnci(pgid), pgmxci(pgid), pgitf(pgid) )
      call pgebuf()

   end if

end subroutine
