! PGHTCH -- Hatch a polygonal area (internal routine)

subroutine PGHTCH( n, x, y, da )

   integer,          intent(in) :: n
   double precision, intent(in) :: x(*), y(*), da

   ! Hatch a polygonal area using equi-spaced parallel lines. The lines
   ! are drawn using the current line attributes: line style, line width,
   ! and color index. Cross-hatching can be achieved by calling this
   ! routine twice.
   !
   ! Limitations: the hatching will not be done correctly if the
   ! polygon is so complex that a hatch line intersects more than
   ! 32 of its sides.
   !
   ! Arguments:
   !  N      (input)  : the number of vertices of the polygonal.
   !  X,Y    (input)  : the (x,y) world-coordinates of the vertices
   !                    (in order).
   !  DA      (input) : 0.0 for normal hatching, 90.0 for perpendicular
   !                    hatching.
   !--
   ! Reference: I.O. Angel and G. Griffith "High-resolution computer
   ! graphics using Fortran 77", Halsted Press, 1987.
   !
   ! 18-Feb-1995 [TJP].
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------

   ! MAXP is the maximum number of intersections any hatch line may make
   ! with the sides of the polygon.
   integer, parameter :: maxp=32
   integer np(maxp), i,j, ii,jj, nmin,nmax, nx, ni, nnp
   double precision angle, sepn, phase
   double precision rmu(maxp), dx,dy, c, cmid,cmin,cmax, sx,sy, ex,ey, delta
   double precision qx,qy, r, rmu1, rmu2, xi,yi, bx,by
   double precision dh, xs1, xs2, ys1, ys2, xl, xr, yt, yb, dindx, dindy

   !------ end of declarations -- execution starts hereafter  ------

   ! Check arguments.
   if( n < 3 ) return
   call pgqhs( angle, sepn, phase )
   angle = angle + da
   if( sepn == 0.0d0 ) return

   ! The unit spacing is 1 percent of the smaller of the height or
   ! width of the view surface. The line-spacing (DH), in inches, is
   ! obtained by multiplying this by argument SEPN.
   call pgqvsz( 1, xs1, xs2, ys1, ys2 )
   dh = sepn*min(abs(xs2-xs1),abs(ys2-ys1))/100.0d0

   ! DINDX and DINDY are the scales in inches per world-coordinate unit.
   call pgqvp( 1, xs1, xs2, ys1, ys2 )
   call pgqwin( xl, xr, yb, yt )
   if( xr /= xl .and. yt /= yb ) then
      dindx = (xs2 - xs1)/(xr - xl)
      dindy = (ys2 - ys1)/(yt - yb)
   else
      return
   end if

   call pgbbuf()

   ! The vector (SX,SY) is a vector length DH perpendicular to
   ! the hatching lines, which have vector (DX,DY).
   dx = cos(angle*deg_to_rad)
   dy = sin(angle*deg_to_rad)
   sx = (-dh)*dy
   sy = dh*dx

   ! The hatch lines are labelled by a parameter C, the distance from
   ! the coordinate origin. Calculate CMID, the C-value of the line
   ! that passes through the hatching reference point (BX,BY), and
   ! CMIN and CMAX, the range of C-values spanned by lines that intersect
   ! the polygon.
   bx = phase*sx
   by = phase*sy
   cmid = dx*by - dy*bx
   cmin = dx*y(1)*dindy - dy*x(1)*dindx
   cmax = cmin
   do i = 2, n
      c = dx*y(i)*dindy - dy*x(i)*dindx
      cmin = min(c,cmin)
      cmax = max(c,cmax)
   end do

   ! Compute integer labels for the hatch lines; N=0 is the line
   ! which passes through the reference point; NMIN and NMAX define
   ! the range of labels for lines that intersect the polygon.
   ! [Note that INT truncates towards zero; we need FLOOR and CEIL
   ! functions.]
   cmin = (cmin-cmid)/dh
   cmax = (cmax-cmid)/dh
   nmin = int(cmin)
   if( dble(nmin) < cmin ) nmin = nmin + 1
   nmax = int(cmax)
   if( dble(nmax) > cmax ) nmax = nmax - 1

   ! Each iteration of the following loop draws one hatch line.
   do j = nmin, nmax

      ! The parametric representation of this hatch line is
      ! (X,Y) = (QX,QY) + RMU*(DX,DY).
      qx = bx + dble(j)*sx
      qy = by + dble(j)*sy

      ! Find the NX intersections of this line with the edges of the polygon.
      nx = 0
      ni = n
      do i = 1, n
         ex = (x(i) - x(ni))*dindx
         ey = (y(i) - y(ni))*dindy
         delta = ex*dy - ey*dx
         if( abs(delta) < 1.0d-5 ) then
            ! Lines are parallel
         else
            ! Lines intersect in (XI,YI)
            r = ((qx-x(ni)*dindx)*dy - (qy-y(ni)*dindy)*dx)/delta
            if( r > 0.0d0 .and. r <= 1.0d0 ) then
               if( nx < maxp ) nx = nx + 1
               np(nx) = nx
               if( abs(dx) > 0.5d0 ) then
                  xi = x(ni)*dindx + r*ex
                  rmu(nx) = (xi-qx)/dx
               else
                  yi = y(ni)*dindy + r*ey
                  rmu(nx) = (yi-qy)/dy
               end if
            end if
         end if
         ni = i
      end do

      ! The RMU array now contains the intersections. Sort them into order.
      do ii = 1, nx-1
         do jj = ii+1, nx
            if( rmu(np(ii)) < rmu(np(jj)) ) then
               nnp = np(ii)
               np(ii) = np(jj)
               np(jj) = nnp
            end if
         end do
      end do

      ! Join the intersections in pairs.
      ni = 1
      do while( ni < nx )
         rmu1 = rmu(np(ni))
         rmu2 = rmu(np(ni+1))
         call grmova( (qx+rmu1*dx)/dindx, (qy+rmu1*dy)/dindy )
         call grlina( (qx+rmu2*dx)/dindx, (qy+rmu2*dy)/dindy )
         ni = ni + 2
      end do
   end do

   call pgebuf()

end subroutine
