! PGENV -- Set window and viewport

subroutine PGENV( xmin, xmax, ymin, ymax, just, colbar, xlabel_pos, clear )

   double precision, intent(in) :: xmin, xmax, ymin, ymax
   integer,          intent(in) :: just, colbar, xlabel_pos
   logical, optional :: clear

   ! Set MFPLOT "Plotter Environment".  PGENV establishes the scaling
   ! for subsequent drawing calls.
   !
   ! Arguments:
   !  XMIN   (input): the world x-coordinate at the bottom left corner
   !                  of the viewport.
   !  XMAX   (input): the world x-coordinate at the top right corner
   !                  of the viewport (note XMAX may be less than XMIN).
   !  YMIN   (input): the world y-coordinate at the bottom left corner
   !                  of the viewport.
   !  YMAX   (input): the world y-coordinate at the top right corner
   !                  of the viewport (note YMAX may be less than YMIN).
   !  JUST   (input): if JUST=1, the scales of the x and y axes (in world
   !                  coordinates per inch) will be equal, otherwise they
   !                  will be scaled independently.
   ! COLBAR  (input): By default, there is no colorbar. If COLBAR=1, a
   !                  colorbar will be inserted on the right side; if
   !                  COLBAR=2, a colorbar will be inserted on the bottom
   !                  side.
   ! XLABEL_POS (input): If XLABEL_POS=1 (the default), numbering and
   !                  descriptive labelling are located below the box. If
   !                  XLABEL_POS=2, the same labelling is located above the
   !                  box.
   !--
   !  1-May-1983
   ! 25-Sep-1985 - Change to use PGWNAD [TJP].
   ! 23-Nov-1985 - Add PGPLOT_ENVOPT option [TJP].
   ! 31-Dec-1985 - Remove automatic PGBEG call [TJP].
   ! 29-Aug-1989 - Remove common block; no longer needed [TJP].
   !---------------------------------------------------------------------------
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !  7-Mar-2020 - Remove completely the PGBOX call: in Muesli, it is always
   !               called at another time. Last argument suppressed.
   ! 17-Apr-2020 - Removed PGVSTD call to avoid calling PGVW twice.
   ! 11-Mar-2021 - Removed clipping setting. Clipping in now enabled by
   !               default.
   ! 17-Mar-2021 - Added the argument COLBAR to specify more room around
   !               the viewport for the colorbar and its legend.
   !  6-May-2021 - Modification to take into account the title font size,
   !               which may be now different from labels font size.
   ! 16-Jun-2021 - Light modification to take care of the location of the
   !               numerical labelling for the X-axis (may be below the box,
   !               which is the default, or above the box). This introduces
   !               the new XLABEL_POS argument.
   !---------------------------------------------------------------------------

   double precision :: xleft, xright, ybot, ytop, ref_height, r
   double precision :: factor, axis_label_height, desc_label_height, title_height
   double precision :: left_margin, right_margin, bottom_margin, top_margin
   double precision :: margins_sum
   logical :: must_be_cleared

   character(len=32) :: dev_typ
   logical :: ldummy

   double precision :: rbuf(1)
   integer :: ibuf(1), lchr
   character(len=1) :: chr

   !------ end of declarations -- execution starts hereafter  ------

   ! Clear the surface (it is sufficient; pgpage call has been removed)
   if( present(clear) ) then
      must_be_cleared = clear
   else
      must_be_cleared = .true.
   end if
   if( must_be_cleared ) then
      call grepic
      call grbpic
   end if

   ! Caution: as pgenv is called just before the drawing of the box,
   !          here pgysp(pgid) is the Axis Font Size.
   ! => Label Font Size = pgysp(pgid)/factor
   ! with:
   factor = pg_axe_fnt_siz_fct(pgid)/pg_lab_fnt_siz_fct(pgid)
   ! (see also PGLAB)

   if( pg_axe_fnt_siz_fct(pgid) == 0.0d0 ) then
      call grwarn('PGENV: internal error in setting ref_height.')
   end if

   ! Required height for putting ticks and their numerical labels
   axis_label_height = 2.0d0*pgysp(pgid)

   call grqtyp( dev_typ, ldummy )
   if( trim(dev_typ) == "XWINDOW" ) then
      ! Save the above height in the X11 driver
      ibuf(1) = nint(axis_label_height)
      ! Make this value even
      if( mod(ibuf(1),2) /= 0 ) ibuf(1) = ibuf(1) + 1
      call grexec( grgtyp, GET_LABEL_AREA_AROUND, rbuf, ibuf, chr, lchr )
   end if

   ! Required height for putting descriptive labels
   desc_label_height = 2.0d0*(pgysp(pgid)/factor)

   ref_height = axis_label_height + desc_label_height

   left_margin = ref_height
   right_margin = axis_label_height
   if( colbar == 1 ) then
      right_margin = right_margin + 2.0d0*axis_label_height
   end if

   ! in inches
   xleft = left_margin/pgxpin(pgid)

   margins_sum = left_margin + right_margin
   if( grxmxa(grcide) > margins_sum ) then
      xright = xleft + (grxmxa(grcide)-margins_sum)/pgxpin(pgid)
   else
      print *, "(MUESLI FGL:) PGENV: cannot get a positive viewport width (too small view surface?)"
      pause "for debugging purpose"
      return
   end if

   !-------------------------------------------------------
   if( xlabel_pos == 1 ) then
      bottom_margin = ref_height
   else ! xlabel_pos = 2
      bottom_margin = axis_label_height
   end if
   if( colbar == 2 ) then
      bottom_margin = bottom_margin + 2.0d0*axis_label_height
   end if

   ! The top margin is somewhat different because the title font size may
   ! differ from that of descriptive labels...
   factor = pg_axe_fnt_siz_fct(pgid)/pg_tit_fnt_siz_fct(pgid)
   title_height = 2.0d0*(pgysp(pgid)/factor)
   top_margin = axis_label_height + title_height
   if( xlabel_pos == 2 ) then
      top_margin = top_margin + desc_label_height
   end if

   ! in inches
   ybot = bottom_margin/pgypin(pgid)

   margins_sum = bottom_margin + top_margin
   if( grymxa(grcide) > margins_sum ) then
      ytop = ybot + (grymxa(grcide)-margins_sum)/pgypin(pgid)
   else
      print *, "(MUESLI FGL:) PGENV: cannot get a positive viewport height (too small view surface?)"
      pause "for debugging purpose"
      return
   end if

   pgxlen(pgid) = (xright-xleft)*pgxpin(pgid)
   pgylen(pgid) = (ytop-ybot)*pgypin(pgid)

   pgxvp(pgid)  = xleft*pgxpin(pgid)
   pgyvp(pgid)  = ybot*pgypin(pgid)

   ! If invalid arguments are specified, issue warning and leave window
   ! unchanged.
   if( xmin == xmax ) then
      call grwarn('Invalid x limits in PGENV: xmin = xmax.')
      return
   else if( ymin == ymax ) then
      call grwarn('Invalid y limits in PGENV: ymin = ymax.')
      return
   end if

   ! Call PGSWIN to define the window.
   ! If equal-scales requested, adjust viewport.
   if( just == 1 ) then
      call pgwnad( xmin, xmax, ymin, ymax )
   else
      call pgswin( xmin, xmax, ymin, ymax )
   end if

end subroutine
