! PGCTAB -- Install the color table to be used by PGIMAG

subroutine PGCTAB( l, r, g, b, nc, contra, bright )

   integer :: nc
   double precision :: l(nc), r(nc), g(nc), b(nc), contra, bright

   ! Use the given color table to change the color representations of
   ! all color indexes marked for use by PGIMAG. To change which
   ! color indexes are thus marked, call PGSCIR before calling PGCTAB
   ! or PGIMAG. On devices that can change the color representations
   ! of previously plotted graphics, PGCTAB will also change the colors
   ! of existing graphics that were plotted with the marked color
   ! indexes. This feature can then be combined with PGBAND to
   ! interactively manipulate the displayed colors of data previously
   ! plotted with PGIMAG.
   !
   ! Limitations:
   !  1. Some devices do not propagate color representation changes
   !     to previously drawn graphics.
   !  2. Some devices ignore requests to change color representations.
   !  3. The appearance of specific color representations on grey-scale
   !     devices is device-dependent.
   !
   ! Notes:
   !  To reverse the sense of a color table, change the chosen contrast
   !  and brightness to -CONTRA and 1-BRIGHT.
   !
   !  In the following, the term 'color table' refers to the input
   !  L,R,G,B arrays, whereas 'color ramp' refers to the resulting
   !  ramp of colors that would be seen with PGWEDG.
   !
   ! Arguments:
   !  L      (input)  : An array of NC normalized ramp-intensity levels
   !                    corresponding to the RGB primary color intensities
   !                    in R(),G(),B(). Colors on the ramp are linearly
   !                    interpolated from neighbouring levels.
   !                    Levels must be sorted in increasing order.
   !                     0.0 places a color at the beginning of the ramp.
   !                     1.0 places a color at the end of the ramp.
   !                    Colors outside these limits are legal, but will
   !                    not be visible if CONTRA=1.0 and BRIGHT=0.5.
   !  R      (input)  : An array of NC normalized red intensities.
   !  G      (input)  : An array of NC normalized green intensities.
   !  B      (input)  : An array of NC normalized blue intensities.
   !  NC     (input)  : The number of color table entries.
   !  CONTRA (input)  : The contrast of the color ramp (normally 1.0).
   !                    Negative values reverse the direction of the ramp.
   !  BRIGHT (input)  : The brightness of the color ramp. This is normally
   !                    0.5, but can sensibly hold any value between 0.0
   !                    and 1.0. Values at or beyond the latter two
   !                    extremes, saturate the color ramp with the colors
   !                    of the respective end of the color table.
   !--
   ! 17-Sep-1994 - New routine [MCS].
   ! 14-Apr-1997 - Modified to implement a more conventional
   !                interpretation of contrast and brightness [MCS].
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------

   integer :: minind, maxind, ci
   integer :: ntotal, nspan
   integer :: below, above
   logical :: forwrd
   double precision :: ca, cb, cifrac, span
   double precision :: level
   double precision :: ldiff, lfrac
   double precision :: red, green, blue

   ! Set the minimum absolute contrast - this prevents a divide by zero.
   double precision, parameter :: minctr = 1.0d0/256

   !------ end of declarations -- execution starts hereafter  ------

   ! No colormap entries?
   if( nc == 0 ) return

   ! Determine the range of color indexes to be used.
   call pgqcir( minind, maxind )

   ! Count the total number of color indexes to be processed.
   ntotal = maxind - minind + 1

   ! No definable colors?
   if( ntotal < 1 .or. minind < 0 ) return

   ! Prevent a divide by zero later by ensuring that CONTRA >= ABS(MINCTR).
   if( abs(contra) < minctr ) then
      contra = sign(minctr, contra)
   end if

   ! Convert contrast to the normalized stretch of the
   ! color table across the available color index range.
   span = 1.0d0 / abs(contra)

   ! Translate from brightness and contrast to the normalized color index
   ! coordinates, CA and CB, at which to place the start and end of the
   ! color table.
   if( contra >= 0.0d0 ) then
      ca = 1.0d0 - bright*(1.0d0 + span)
      cb = ca + span
   else
      ca = bright*(1.0d0 + span)
      cb = ca - span
   end if

   ! Determine the number of color indexes spanned by the color table.
   nspan = int(span * ntotal)

   ! Determine the direction in which the color table should be traversed.
   forwrd = ca <= cb

   ! Initialize the indexes at which to start searching the color table.
   !
   ! Set the start index for traversing the table from NC to 1.
   below = nc

   ! Set the start index for traversing the table from 1 to NC.
   above = 1

   ! Buffer drawing commands until the color map has been completely
   ! installed.
   call pgbbuf()

   ! Linearly interpolate the color table RGB values onto each color index.
   do ci = minind, maxind

      ! Turn the color index into a fraction of the range MININD..MAXIND.
      cifrac = dble(ci-minind) / dble(maxind-minind)

      ! Determine the color table position that corresponds to color index,
      ! CI.
      if( nspan > 0 ) then
         level = (cifrac-ca)/(cb-ca)
      else
         if( cifrac <= ca ) then
            level = 0.0d0
         else
            level = 1.0d0
         end if
      end if

      ! Search for the indexes of the two color table entries that straddle
      ! LEVEL. The search algorithm assumes that values in L() are arranged
      ! in increasing order. This allows us to search the color table from
      ! the point at which the last search left off, rather than having to
      ! search the whole color table each time.
      if( forwrd ) then
 2       if( above <= nc .and. l(above) > level ) then
            above = above + 1
            goto 2
         end if
         below = above - 1
      else
 3       if( below >= 1 .and. l(below) > level ) then
            below = below - 1
            goto 3
         end if
         above = below + 1
      end if

      ! If the indexes lie outside the table, substitute the index of the
      ! nearest edge of the table.
      if( below < 1 ) then
         level = 0.0d0
         below = 1
         above = 1
      else if( above > nc ) then
         level = 1.0d0
         below = nc
         above = nc
      end if

      ! Linearly interpolate the primary color intensities from color table
      ! entries, BELOW and ABOVE.
      ldiff = l(above) - l(below)
      if( ldiff > minctr ) then
         lfrac = (level - l(below)) / ldiff
      else
         lfrac = 0.0d0
      end if
      red   = r(below) + (r(above) - r(below))*lfrac
      green = g(below) + (g(above) - g(below))*lfrac
      blue  = b(below) + (b(above) - b(below))*lfrac

      ! Intensities are only defined between 0 and 1.
      if( red   < 0.0d0 )   red = 0.0d0
      if( red   > 1.0d0 )   red = 1.0d0
      if( green < 0.0d0 ) green = 0.0d0
      if( green > 1.0d0 ) green = 1.0d0
      if( blue  < 0.0d0 )  blue = 0.0d0
      if( blue  > 1.0d0 )  blue = 1.0d0

      ! Install the new color representation.
      call grscr( ci, red, green, blue )
   end do

   ! Reveal the changed color map.
   call pgebuf()

end subroutine
