! PGCONX -- Contour map of a 2D data array (non rectangular)

subroutine PGCONX( a, idim, jdim, i1, i2, j1, j2, c, nc, plot )

   integer :: idim, jdim, i1, j1, i2, j2, nc
   double precision :: a(idim,jdim), c(*)
   external plot

   ! Draw a contour map of an array using a user-supplied plotting
   ! routine.  This routine should be used instead of PGCONT when the
   ! data are defined on a non-rectangular grid.  PGCONT permits only
   ! a linear transformation between the (I,J) grid of the array
   ! and the world coordinate system (x,y), but PGCONX permits any
   ! transformation to be used, the transformation being defined by a
   ! user-supplied subroutine. The nature of the contouring algorithm,
   ! however, dictates that the transformation should maintain the
   ! rectangular topology of the grid, although grid-points may be
   ! allowed to coalesce.  As an example of a deformed rectangular
   ! grid, consider data given on the polar grid theta=0.1n(pi/2),
   ! for n=0,1,...,10, and r=0.25m, for m=0,1,..,4. This grid
   ! contains 55 points, of which 11 are coincident at the origin.
   ! The input array for PGCONX should be dimensioned (11,5), and
   ! data values should be provided for all 55 elements.  PGCONX can
   ! also be used for special applications in which the height of the
   ! contour affects its appearance, e.g., stereoscopic views.
   !
   ! The map is truncated if necessary at the boundaries of the viewport.
   ! Each contour line is drawn with the current line attributes (color
   ! index, style, and width); except that if argument NC is positive
   ! (see below), the line style is set by PGCONX to 1 (solid) for
   ! positive contours or 2 (dashed) for negative contours. Attributes
   ! for the contour lines can also be set in the user-supplied
   ! subroutine, if desired.
   !
   ! Arguments:
   !  A      (input) : data array.
   !  IDIM   (input) : first dimension of A.
   !  JDIM   (input) : second dimension of A.
   !  I1, I2 (input) : range of first index to be contoured (inclusive).
   !  J1, J2 (input) : range of second index to be contoured (inclusive).
   !  C      (input) : array of NC contour levels; dimension at least NC.
   !  NC     (input) : +/- number of contour levels (less than or equal
   !                   to dimension of C). If NC is positive, it is the
   !                   number of contour levels, and the line-style is
   !                   chosen automatically as described above. If NC is
   !                   negative, it is minus the number of contour
   !                   levels, and the current setting of line-style is
   !                   used for all the contours.
   !  PLOT   (input) : the address (name) of a subroutine supplied by
   !                   the user, which will be called by PGCONX to do
   !                   the actual plotting. This must be declared
   !                   EXTERNAL in the program unit calling PGCONX.
   !
   ! The subroutine PLOT will be called with four arguments:
   !      CALL PLOT(VISBLE,X,Y,Z)
   ! where X,Y (input) are real variables corresponding to
   ! I,J indices of the array A. If  VISBLE (input, integer) is 1,
   ! PLOT should draw a visible line from the current pen
   ! position to the world coordinate point corresponding to (X,Y);
   ! if it is 0, it should move the pen to (X,Y). Z is the value
   ! of the current contour level, and may be used by PLOT if desired.
   ! Example:
   !           subroutine plot( visble, x, y, z )
   !              double precision :: x, y, z, xworld, yworld
   !              integer :: visble
   !              xworld = x*cos(y) ! this is the user-defined
   !              yworld = x*sin(y) ! transformation
   !              if( visble == 0 ) then
   !                 call grmova( xworld, yworld )
   !              else
   !                 call grlina( xworld, yworld )
   !              end if
   !           end
   !--
   ! 14-Nov-1985 - new routine [TJP].
   ! 12-Sep-1989 - correct documentation error [TJP].
   ! 22-Apr-1990 - corrected bug in panelling algorithm [TJP].
   ! 13-Dec-1990 - make errors non-fatal [TJP].
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------

   integer :: i
   integer :: nnx,nny, kx,ky, ki,kj, ia,ib, ja,jb, ls, px, py
   logical :: style

   !------ end of declarations -- execution starts hereafter  ------

   ! Check arguments.
   if( i1<1 .or. i2>idim .or. i1>=i2 .or.                               &
       j1<1 .or. j2>jdim .or. j1>=j2) then
      call grwarn('PGCONX: invalid range i1:i2, j1:j2')
      return
   end if
   if( nc == 0 ) return

   call pgbbuf()

   style = nc > 0
   call grqls(ls) ! Save style

   ! Divide arrays into panels not exceeding MAXEMX by MAXEMY for
   ! contouring by PGCNSC.
   nnx = i2-i1+1
   nny = j2-j1+1
   kx = max(1,(nnx+maxemx-2)/(maxemx-1))
   ky = max(1,(nny+maxemy-2)/(maxemy-1))
   px = (nnx+kx-1)/kx
   py = (nny+ky-1)/ky
   do ki = 1, kx
      ia = i1 + (ki-1)*px
      ib = min(i2, ia + px)
      do kj = 1, ky
         ja = j1 + (kj-1)*py
         jb = min(j2, ja + py)

         ! Draw the contours in one panel.
         ! (no need to take care of linestyle, it is not used
         !  by PostScript fonts)
         if( .not. printing_eps .and. .not. printing_pdf ) then
            if( style ) call grsls(1)
         end if
         do i = 1, abs(nc)
            if( .not. printing_eps .and. .not. printing_pdf ) then
               if( style .and. (c(i)<0.0) ) call grsls(2)
            end if
            call pgcnsc(a,idim,jdim,ia,ib,ja,jb,c(i),plot)
            if( .not. printing_eps .and. .not. printing_pdf ) then
               if( style ) call grsls(1)
            end if
         end do
      end do
   end do

   call grsls(ls) ! Restore style

   call pgebuf()

end subroutine
