! PGCONF -- Fill between two contours

subroutine PGCONF( a, idim, jdim, i1, i2, j1, j2, c1, c2, tr )

   integer :: idim, jdim, i1, i2, j1, j2
   double precision :: a(idim,jdim), c1, c2, tr(6)

   ! Shade the region between two contour levels of a function defined on
   ! the nodes of a rectangular grid. The routine uses the current fill
   ! attributes, hatching style (if appropriate), and color index.
   !
   ! If you want to both shade between contours and draw the contour
   ! lines, call this routine first (once for each pair of levels) and
   ! then CALL PGCONT (or PGCONS) to draw the contour lines on top of the
   ! shading.
   !
   ! Note 1: This routine is not very efficient: it generates a polygon
   ! fill command for each cell of the mesh that intersects the desired
   ! area, rather than consolidating adjacent cells into a single polygon.
   !
   ! Note 2: If both contours intersect all four edges of a particular
   ! mesh cell, the program behaves badly and may consider some parts
   ! of the cell to lie in more than one contour range.
   !
   ! Note 3: If a contour crosses all four edges of a cell, this
   ! routine may not generate the same contours as PGCONT or PGCONS
   ! (these two routines may not agree either). Such cases are always
   ! ambiguous and the routines use different approaches to resolving
   ! the ambiguity.
   !
   ! Arguments:
   !  A      (input)  : data array.
   !  IDIM   (input)  : first dimension of A.
   !  JDIM   (input)  : second dimension of A.
   !  I1,I2  (input)  : range of first index to be contoured (inclusive).
   !  J1,J2  (input)  : range of second index to be contoured (inclusive).
   !  C1, C2 (input)  : contour levels; note that C1 must be less than C2.
   !  TR     (input)  : array defining a transformation between the I,J
   !                    grid of the array and the world coordinates. The
   !                    world coordinates of the array point A(I,J) are
   !                    given by:
   !                      X = TR(1) + TR(2)*I + TR(3)*J
   !                      Y = TR(4) + TR(5)*I + TR(6)*J
   !                    Usually TR(3) and TR(5) are zero - unless the
   !                    coordinate transformation involves a rotation
   !                    or shear.
   !--
   ! 03-Oct-1996 - New routine [TJP].
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------

   integer :: i, j, ic, npt, lev

   double precision :: dval(5), x(8), y(8), delta, xx, yy, c, r
   integer :: idelt(6)
   data idelt/0,-1,-1,0,0,-1/

   !------ end of declarations -- execution starts hereafter  ------

   ! Check arguments.
   if( i1 < 1 .or. i2 > idim .or. i1 >= i2 .or.                         &
       j1 < 1 .or. j2 > jdim .or. j1 >= j2 ) return
   if( c1 >= c2 ) return

   call pgbbuf()

   do j = j1+1, j2
      do i = i1+1, i2
         dval(1) = a(i-1,j)
         dval(2) = a(i-1,j-1)
         dval(3) = a(i,j-1)
         dval(4) = a(i,j)
         dval(5) = dval(1)

         npt = 0
         do ic = 1, 4
            if( dval(ic) >= c1 .and. dval(ic) < c2 ) then
               npt = npt + 1
               xx = i  +idelt(ic+1)
               yy = j + idelt(ic)
               x(npt) = tr(1) + tr(2)*xx + tr(3)*yy
               y(npt) = tr(4) + tr(5)*xx + tr(6)*yy
            end if
            r = dval(ic+1) - dval(ic)
            if( r == 0.0d0 ) cycle
            do lev = 1, 2
               if( r > 0.0d0 ) then
                  c = c1
                  if( lev == 2 ) c = c2
               else
                  c = c2
                  if( lev == 2 ) c = c1
               end if
               delta = (c-dval(ic))/r
               if( delta > 0.0d0 .and. delta < 1.0d0 ) then
                  if( ic == 1 .or. ic == 3 ) then
                     xx = i + idelt(ic+1)
                     yy = dble(j+idelt(ic)) +                           &
                          delta*dble(idelt(ic+1)-idelt(ic))
                  else
                     xx = dble(i+idelt(ic+1)) +                         &
                          delta*dble(idelt(ic+2)-idelt(ic+1))
                     yy = j + idelt(ic)
                  end if
                  npt = npt + 1
                  x(npt) = tr(1) + tr(2)*xx + tr(3)*yy
                  y(npt) = tr(4) + tr(5)*xx + tr(6)*yy
               end if
            end do
         end do
         if( npt >= 3 ) call pgpoly( npt, x, y )
      end do
   end do

   call pgebuf()

end subroutine
