! PGCONB -- Contour map of a 2D data array, with blanking

subroutine PGCONB( a, idim, jdim, i1, i2, j1, j2, c, nc, tr, blank )

   integer :: idim, jdim, i1, i2, j1, j2, nc
   double precision :: a(idim,jdim), c(*), tr(6), blank

   ! Draw a contour map of an array. This routine is the same as PGCONS,
   ! except that array elements that have the "magic value" defined by
   ! argument BLANK are ignored, making gaps in the contour map. The
   ! routine may be useful for data measured on most but not all of the
   ! points of a grid.
   !
   ! Arguments:
   !  A      (input)  : data array.
   !  IDIM   (input)  : first dimension of A.
   !  JDIM   (input)  : second dimension of A.
   !  I1,I2  (input)  : range of first index to be contoured (inclusive).
   !  J1,J2  (input)  : range of second index to be contoured (inclusive).
   !  C      (input)  : array of contour levels (in the same units as the
   !                    data in array A); dimension at least NC.
   !  NC     (input)  : number of contour levels (less than or equal to
   !                    dimension of C). The absolute value of this
   !                    argument is used (for compatibility with PGCONT,
   !                    where the sign of NC is significant).
   !  TR     (input)  : array defining a transformation between the I,J
   !                    grid of the array and the world coordinates. The
   !                    world coordinates of the array point A(I,J) are
   !                    given by:
   !                      X = TR(1) + TR(2)*I + TR(3)*J
   !                      Y = TR(4) + TR(5)*I + TR(6)*J
   !                    Usually TR(3) and TR(5) are zero - unless the
   !                    coordinate transformation involves a rotation
   !                    or shear.
   !  BLANK   (input) : elements of array A that are exactly equal to
   !                    this value are ignored (blanked).
   !--
   ! 21-Sep-1989 - Derived from PGCONS [TJP].
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------

   integer :: i, ic, icorn, idelt(6), j, k, npt
   integer :: ioff(8), joff(8), ienc, itmp, jtmp, ilo, itot

   double precision :: ctr, delta, dval(5), xx, yy, x(4), y(4)
   data idelt/0,-1,-1,0,0,-1/
   data ioff/-2,-2,-1,-1, 0, 0, 1, 1/
   data joff/ 0,-1,-2, 1,-2, 1,-1, 0/

   !------ end of declarations -- execution starts hereafter  ------

   ! Check arguments.
   if( i1 < 1 .or. i2 > idim .or. i1 >= i2 .or.                          &
       j1 < 1 .or. j2 > jdim .or. j1 >= j2 ) return
   if( nc == 0 ) return

   call pgbbuf()

   do j = j1+1, j2
      do i = i1+1, i2
         dval(1) = a(i-1,j)
         dval(2) = a(i-1,j-1)
         dval(3) = a(i,j-1)
         dval(4) = a(i,j)
         dval(5) = dval(1)
         if( any(dval(1:4) == blank) ) cycle
         do ic = 1, abs(nc)
            ctr = c(ic)
            npt = 0
            do icorn = 1, 4
               if( (dval(icorn) < ctr .and. dval(icorn+1) < ctr)        &
                   .or. (dval(icorn) >= ctr .and. dval(icorn+1) >= ctr) ) cycle
               npt = npt+1
               delta = (ctr-dval(icorn))/(dval(icorn+1)-dval(icorn))
               goto (60,70,60,70), icorn

 60            xx = i+idelt(icorn+1)
               yy = dble(j+idelt(icorn)) +                              &
                    delta*dble(idelt(icorn+1)-idelt(icorn))
               goto 80

 70            xx = dble(i+idelt(icorn+1)) +                            &
                    delta*dble(idelt(icorn+2)-idelt(icorn+1))
               yy  = j+idelt(icorn)

 80            x(npt) = tr(1) + tr(2)*xx + tr(3)*yy
               y(npt) = tr(4) + tr(5)*xx + tr(6)*yy

            end do
            if( npt == 2 ) then
               ! Contour crosses two sides of cell. Draw line-segment.
               call grmova( x(1), y(1) )
               call grlina( x(2), y(2) )
            else if( npt == 4 ) then
               ! The 'ambiguous' case. The routine must draw two line segments
               ! here and there are two ways to do so. The following four lines
               ! would implement the original MFPLOT method:
               !     CALL PGCP(0,X(1),Y(1),CTR)
               !     CALL PGCP(1,X(2),Y(2),CTR)
               !     CALL PGCP(0,X(3),Y(3),CTR)
               !     CALL PGCP(1,X(4),Y(4),CTR)
               ! Choose between \\ and // based on the 8 points just outside
               ! the current box. If half or more of these points lie below
               ! the contour level, then draw the lines such that the high
               ! corners lie between the lines, otherwise, draw the lines such
               ! that the low corners are enclosed.  Care is taken to avoid
               ! going off the edge.
               itot = 0
               ilo = 0
               do k = 1, 8
                  itmp = i + ioff(k)
                  jtmp = j + joff(k)
                  if( itmp < i1 .or. i2 < itmp ) cycle
                  if( jtmp < j1 .or. j2 < jtmp ) cycle
                  if( a(itmp,jtmp) == blank ) cycle
                  itot = itot + 1
                  if( a(itmp,jtmp) < ctr ) ilo = ilo + 1
               end do
               ienc = +1
               if( ilo < itot/2) ienc = -1
               if( ienc < 0 .and. dval(1) < ctr .or.                    &
                   ienc > 0 .and. dval(1) >= ctr ) then
                  call grmova( x(1), y(1) )
                  call grlina( x(2), y(2) )
                  call grmova( x(3), y(3) )
                  call grlina( x(4), y(4) )
               else
                  call grmova( x(1), y(1) )
                  call grlina( x(4), y(4) )
                  call grmova( x(3), y(3) )
                  call grlina( x(2), y(2) )
               end if
            end if
         end do
      end do
   end do

   call pgebuf()

end subroutine
