! PGCNSC -- Auxiliary routine used by PGCONX

subroutine PGCNSC( z, mx, my, ia, ib, ja, jb, z0, plot )

use fml, only : mf, msDisplay
   integer :: mx, my, ia, ib, ja, jb
   double precision :: z(mx,*)
   double precision :: z0
   external plot

   ! MFPLOT (internal routine): Draw a single contour.  This routine is
   ! called by PGCONX, but may be called directly by the user.
   !
   ! Arguments:
   !
   ! Z (real array dimension MX,MY, input): the array of function values.
   ! MX,MY (integer, input): actual declared dimension of Z(*,*).
   ! IA,IB (integer, input): inclusive range of the first index of Z to be
   !       contoured.
   ! JA,JB (integer, input): inclusive range of the second index of Z to
   !       be contoured.
   ! Z0 (real, input): the contour level sought.
   ! PLOT (the name of a subroutine declared EXTERNAL in the calling
   !       routine): this routine is called by PGCNSC to do all graphical
   !       output. The calling sequence is CALL PLOT(K,X,Y,Z) where Z is
   !       the contour level, (X,Y) are the coordinates of a point (in the
   !       inclusive range I1<X<I2, J1<Y<J2, and if K is 0, the routine is
   !       to move the pen to (X,Y); if K is 1, it is to draw a line from
   !       the current position to (X,Y).
   !
   ! NOTE:  the intervals (IA,IB) and (JA,JB) must not exceed the
   ! dimensions of an internal array whose dimensions are set as parameters
   ! in the mod_pgplot module.
   !--
   ! 17-Sep-1989 - Completely rewritten [TJP]. The algorithm is my own,
   !               but it is probably not original. It could probably be
   !               coded more briefly, if not as clearly.
   !  1-May-1994 - Modified to draw contours anticlockwise about maxima,
   !               to prevent contours at different levels from
   !               crossing in degenerate cells [TJP].
   !  6-Jun-2006 - Convert the statement function 'XRANGE' in an internal
   !               procedure (after 'contains') [valid only in F90]
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------
   integer, parameter :: up = 1, down = 2, left = 3, right = 4

   logical :: flags(maxemx,maxemy,2)

   integer :: i, j, ii, jj, dir
   double precision :: z1, z2, z3

   !------ end of declarations -- execution starts hereafter  ------

   ! Check for errors.
   if( (ib-ia+1) > maxemx .or. (jb-ja+1) > maxemy ) then
      call grwarn( 'PGCNSC - array index range exceeds' //              &
                   ' built-in limit: see maxemx and maxemy parameters in mod_pgplot')
      return
   end if

   ! Initialize the flags. The first flag for a gridpoint is set if the
   ! contour crosses the line segment to the right of the gridpoint
   ! (joining [I,J] to [I+1,J]); the second flag is set if it crosses the
   ! line segment above the gridpoint (joining [I,J] to [I,J+1]). The top
   ! and right edges require special treatment. (For purposes of description
   ! only, we assume I increases horizontally to the right and J increases
   ! vertically upwards.)
   do i = ia, ib
      ii = i-ia+1
      do j = ja, jb
         jj = j-ja+1
         z1 = z(i,j)
         flags(ii,jj,1) = .false.
         flags(ii,jj,2) = .false.
         if( i < ib ) then
            z2 = z(i+1,j)
            if( xrange(z0,z1,z2) ) flags(ii,jj,1) = .true.
         end if
         if( j < jb ) then
            z3 = z(i,j+1)
            if( xrange(z0,z1,z3) ) flags(ii,jj,2) = .true.
         end if
      end do
   end do

   ! Search the edges of the array for the start of an unclosed contour.
   ! Note that (if the algorithm is implemented correctly) all unclosed
   ! contours must begin and end at the edge of the array. When one is
   ! found, call PGCN01 to draw the contour, telling it the correct
   ! starting direction so that it follows the contour into the array
   ! instead of out of it. A contour is only started if the higher
   ! ground lies to the left: this is to enforce the direction convention
   ! that contours are drawn anticlockwise around maxima. If the high
   ! ground lies to the right, we will find the other end of the contour
   ! and start there.

   ! Bottom edge.
   j = ja
   jj = j-ja+1
   do i = ia, ib-1
      ii = i-ia+1
      if( flags(ii,jj,1) .and. (z(i,j)>z(i+1,j)) ) then
          call pgcn01( z, mx, my, ia, ib, ja, jb,                       &
                       z0, plot, flags, i, j, up )
      end if
   end do

   ! Right edge.
   i = ib
   ii = i-ia+1
   do j = ja, jb-1
      jj = j-ja+1
      if( flags(ii,jj,2) .and. (z(i,j)>z(i,j+1)) ) then
          call pgcn01( z, mx, my, ia, ib, ja, jb,                       &
                       z0, plot, flags, i, j, left )
      end if
   end do

   ! Top edge.
   j = jb
   jj = j-ja+1
   do i = ib-1, ia,-1
      ii = i-ia+1
      if( flags(ii,jj,1) .and. (z(i+1,j)>z(i,j)) ) then
          call pgcn01( z, mx, my, ia, ib, ja, jb,                       &
                       z0, plot, flags, i, j, down )
      end if
   end do

   ! Left edge.
   i = ia
   ii = i-ia+1
   do j = jb-1, ja,-1
      jj = j-ja+1
      if( flags(ii,jj,2)  .and. (z(i,j+1)>z(i,j)) ) then
          call pgcn01( z, mx, my, ia, ib, ja, jb,                       &
                       z0, plot, flags, i, j, right )
      end if
   end do

   ! Now search the interior of the array for a crossing point, which will
   ! lie on a closed contour (because all unclosed contours have been
   ! eliminated). It is sufficient to search just the horizontal crossings
   ! (or the vertical ones); any closed contour must cross a horizontal
   ! and a vertical gridline. PGCN01 assumes that when it cannot proceed
   ! any further, it has reached the end of a closed contour. Thus all
   ! unclosed contours must be eliminated first.
   do i = ia+1, ib-1
      ii = i-ia+1
      do j = ja+1, jb-1
         jj = j-ja+1
         if( flags(ii,jj,1) ) then
            dir = up
            if( z(i+1,j) > z(i,j) ) dir = down
            call pgcn01( z, mx, my, ia, ib, ja, jb,                     &
                         z0, plot, flags, i, j, dir )
         end if
      end do
   end do

   ! We didn't find any more crossing points: we've finished.

   contains

      logical function xrange( p, p1, p2 )

         ! The function XRANGE decides whether a contour at level P
         ! crosses the line between two gridpoints with values P1 and P2.
         ! It is important that a contour cannot cross a line with equal
         ! endpoints.

         double precision, intent(in) :: p, p1, p2
         xrange = ( p > min(p1,p2) ) .and. ( p <= max(p1,p2) )          &
                    .and. ( p1 /= p2 )
      end function

end subroutine
