! PGBOX1 -- Support routines for PGBOX

subroutine PGBOX1( a, b, intv, i1, i2 )

   double precision, intent(in)  :: a, b, intv
   integer,          intent(out) :: i1, i2

   ! This routine is used to determine where to draw the tick marks on
   ! an axis. The input arguments A and B are the world-coordinate
   ! end points of an axis; INTV is the tick interval. PGBOX1 returns
   ! two integers, I1 and I2, such that the required tick marks are
   ! to be placed at world-coordinates (I*INTV), for I=I1,...,I2.
   ! Normally I2 is greater than or equal to I1, but if there are no
   ! values of I such that I*XD lies in the inclusive range (A, B),
   ! then I2 will be 1 less than I1.
   !
   ! Arguments:
   !  A, B   (input)  : world-coordinate end points of the axis. A must
   !                    not be equal to B; otherwise, there are no restrictions.
   !  INTV   (input)  : world-coordinate tick interval. INTV may be positive
   !                    or negative, but may not be zero.
   !  I1, I2 (output) : tick marks should be drawn at world
   !                    coordinates I*INTV for I in the inclusive range
   !                    I1...I2 (see above).
   !
   ! 14-Jan-1986 - new routine [TJP].
   ! 13-Dec-1990 - remove error check [TJP].
   !-----------------------------------------------------------------------
   ! 18-oct-2011 - Returning other real values, to avoid roundoff errors
   !               when A and B are very close together.
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !  4-Jun-2021 - Reduce flops number to compute xlo and xhi.
   !-----------------------------------------------------------------------

   double precision :: a2, b2, lo, hi

   !------ end of declarations -- execution starts hereafter  ------

   a2 = a/intv
   b2 = b/intv
   if( a2 < b2 ) then
      lo = a2
      hi = b2
   else
      lo = b2
      hi = a2
   end if

   i1 = nint(lo)
   if( dble(i1) < lo ) i1 = i1 + 1

   i2 = nint(hi)
   if( dble(i2) > hi ) i2 = i2 - 1

end subroutine PGBOX1
!_______________________________________________________________________
!
subroutine tiny_shift_range( a, b, intv )

   double precision, intent(in out) :: a, b
   double precision, intent(in)     :: intv

   ! This routine makes a tiny modifications of A and/or B, which are the
   ! same arguments as used in PGBOX1.
   ! The aims of this modification is to avoid a surprising result when
   ! the user set the extremal values for an axis and don't see the
   ! corresponding tick drawn on the axis. This is due to binary conversion
   ! when reading decimal values in the source code and the fact that the
   ! number of floating-point values is finite.
   ! For ex., if the user set the range of an axis to [-0.3d0,0.3d0],
   !          the extremal ticks will not appear, because as soon as
   !          the reals -0.3d0 and 0.3d0 are read, the conversion in binary
   !          values lead respectively to -0.29999999999999999 and
   !          0.29999999999999999, so the expected ticks at extremal
   !          values cannot be drawn.
   !
   ! Arguments:
   !  A, B (in- output): world-coordinate end points of the axis. A must not
   !                     be equal to B; otherwise, there are no restrictions.
   !  INTV      (input): world-coordinate tick interval. INTV may be positive
   !                     or negative, but may not be zero.
   !
   !  4-Jun-2021 - new routine.
   ! 11-Jun-2021 - degrade epsilon check to its single precision, because it
   !               is not uncommon that the programmer writes [-0.3,0.3] in
   !               his/her code, i.e. without the 'd' exponent letter for
   !               double precision.
   !-----------------------------------------------------------------------

   double precision :: a2, b2, new
   integer :: i

   !------ end of declarations -- execution starts hereafter  ------

   ! Note: When using a double precision epsilon, the roundoff compensation
   !       should use 2*epsilon instead of one epsilon, because we are working
   !       on a ratio of two FP numbers (a2 = a/intv);
   !       However, for a single precision epsilon, the '2' factor is not
   !       necessary.

   a2 = a/intv
   b2 = b/intv
   if( a2 < b2 ) then
      i = nint(a2)
      if( a2 < 0.0d0 ) then
         new = a2*(1.0d0+epsilon(1.0))
         if( dble(i) > new ) a = new*intv
      else if( a2 > 0.0d0 ) then
         new = a2*(1.0d0-epsilon(1.0))
         if( dble(i) > new ) a = new*intv
      end if

      i = nint(b2)
      if( b2 < 0.0d0 ) then
         new = b2*(1.0d0-epsilon(1.0))
         if( dble(i) < new ) b = new*intv
      else if( b2 > 0.0d0 ) then
         new = b2*(1.0d0+epsilon(1.0))
         if( dble(i) < new ) b = new*intv
      end if
   else
      i = nint(b2)
      if( b2 < 0.0d0 ) then
         new = b2*(1.0d0+epsilon(1.0))
         if( dble(i) > new ) b = new*intv
      else if( b2 > 0.0d0 ) then
         new = b2*(1.0d0-epsilon(1.0))
         if( dble(i) > new ) b = new*intv
      end if

      i = nint(a2)
      if( a2 < 0.0d0 ) then
         new = a2*(1.0d0-epsilon(1.0))
         if( dble(i) < new ) a = new*intv
      else if( a2 > 0.0d0 ) then
         new = a2*(1.0d0+epsilon(1.0))
         if( dble(i) < new ) a = new*intv
      end if
   end if

end subroutine tiny_shift_range
