! PGBOX -- Draw labeled frame around viewport

subroutine PGBOX( xopt, xtick, nxsub, yopt, ytick, nysub )

   character(len=*), intent(in) :: xopt, yopt
   double precision, intent(in) :: xtick, ytick
   integer,          intent(in) :: nxsub, nysub

   ! Annotate the viewport with frame, axes, numeric labels, etc.
   ! PGBOX is called on the user's behalf by PGENV, but may also be
   ! called explicitly.
   !
   ! Arguments:
   !  XOPT  (input)  : string of options for X (horizontal) axis of
   !                   plot. Options are single letters, and may be in
   !                   any order (see below).
   !  XTICK (input)  : world coordinate interval between major tick marks
   !                   on X axis. If XTICK=0, the interval is chosen by
   !                   PGBOX, so that there will be at least 3 major tick
   !                   marks along the axis.
   !  NXSUB (input)  : the number of subintervals to divide the major
   !                   coordinate interval into. If XTICK=0 or NXSUB=0,
   !                   the number is chosen by PGBOX.
   !  YOPT  (input)  : string of options for Y (vertical) axis of plot.
   !                   Coding is the same as for XOPT.
   !  YTICK (input)  : like XTICK for the Y axis.
   !  NYSUB (input)  : like NXSUB for the Y axis.
   !
   ! Options (for parameters XOPT and YOPT):
   !  A : draw Axis (X axis is horizontal line Y=0, Y axis is vertical
   !      line X=0).
   !  B : draw bottom (X) or left (Y) edge of frame.
   !  C : draw top (X) or right (Y) edge of frame.
   !  I : invert the tick marks; i.e. draw them outside the viewport
   !      instead of inside.
   !  L : label axis logarithmically (see below).
   !  N : write numeric labels in the conventional location below the
   !      viewport (X) or to the left of the viewport (Y).
   !  P : extend ("project") major tick marks outside the box (ignored if
   !      option I is specified).
   !  M : write numeric labels in the unconventional location above the
   !      viewport (X) or to the right of the viewport (Y).
   !  T : draw major tick marks at the major coordinate interval.
   !  S : draw minor tick marks (subticks).
   !  V : orient numeric labels vertically. This is only applicable to Y.
   !      The default is to write Y-labels parallel to the axis.
   !  1 : force decimal labelling, instead of automatic choice (see PGNUMB)
   !  2 : force exponential labelling, instead of automatic.
   !  3 : force integer, strictly positive labelling (indices of a matrix).
   !
   ! To get a complete frame, specify BC in both XOPT and YOPT.
   ! Tick marks, if requested, are drawn on the axes or frame
   ! or both, depending which are requested. If none of ABC is specified,
   ! tick marks will not be drawn. When PGENV calls PGBOX, it sets both
   ! XOPT and YOPT according to the value of its parameter AXIS:
   !      -1: 'BC', 0: 'BCNST', 1: 'ABCNST', 2: 'ABCGNST'.
   !
   ! For a logarithmic axis, the major tick interval is always 1.0. The
   ! numeric label is 10**(x) where x is the world coordinate at the
   ! tick mark. If subticks are requested, 8 subticks are drawn between
   ! each major tick at equal logarithmic intervals.
   !
   ! To label an axis with time (days, hours, minutes, seconds) or
   ! angle (degrees, arcmin, arcsec), use routine PGTBOX.
   !--
   ! 19-Oct-1983
   ! 23-Sep-1984 - Fix bug in labelling reversed logarithmic axes.
   !  6-May-1985 - Improve behavior for pen plotters [TJP].
   ! 23-Nov-1985 - Add 'P' option [TJP].
   ! 14-Jan-1986 - Use new routine PGBOX1 to fix problem of missing
   !               labels at end of axis [TJP].
   !  8-Apr-1987 - Improve automatic choice of tick interval; improve
   !               erroneous rounding of tick interval to 1 digit [TJP].
   ! 23-Apr-1987 - Fix bug: limit max number of ticks to ~10 [TJP].
   !  7-Nov-1987 - Yet another change to algorithm for choosing tick
   !               interval; maximum tick interval is now 0.2*range of
   !               axis, which may round up to 0.5 [TJP].
   ! 15-Dec-1988 - Correct declaration of MAJOR [TJP].
   !  6-Sep-1989 - Use Fortran generic intrinsic functions [TJP].
   ! 18-Oct-1990 - Correctly initialize UTAB(1) [AFT].
   ! 19-Oct-1990 - Do all plotting in world coordinates [TJP].
   !  6-Nov-1991 - Label logarithmic subticks when necessary [TJP].
   !  4-Jul-1994 - Add '1' and '2' options [TJP].
   ! 20-Apr-1995 - Adjust position of labels slightly, and move out
   !               when ticks are inverted [TJP].
   ! 26-Feb-1997 - Use new routine pgclp [TJP].
   !-----------------------------------------------------------------------
   ! From around 2003: small fixes by É. Canot
   !    May-2007 - Increase precision of logarithm table (7 decimals
   !               instead of 5). Improved also log axes labelling.
   !    Oct-2008 - RANGE inlined function renamed xrange (because 'range'
   !               is an intrinsic procedure in Std F95).
   !    Jun-2016 - Simplify computation of UTAB(:), both for X and Y.
   !               Added on more level of subdivision (4 parts) for
   !               zooming in logarithmic axes.
   !    Oct-2018 - "G" code is removed since grid is drawn by another
   !               routine
   !    Feb-2020 - Added white rectangles before drawing axes and ticks,
   !               to be sure not to see a character string clipped at
   !               the viewport (4 calls to grrect). [but removed]
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   ! 11-Mar-2021 - Clipping is now treated only via the global variable
   !               CLIPPING_IN_AXES.
   ! 15-Mar-2021 - Option to label only integer values (useful for Pcolor
   !               and Spy routines). Take care also of Colorbar.
   !  5-Apr-2021 - Clipping is now monitored by a higher calling routine,
   !               i.e. mf_win_draw_box.
   ! 27-Nov-2021 - Clipping state is switched if necessary.
   ! 15-Dec-2021 - The frame box is ALWAYS drawn, even if corresponding
   !               options is not set (I don't like open axes).
   ! 11-Mar-2025 - Implement a way to get the number of ticks for X-axis
   !               (major ticks or all; linear axis), and also to change
   !               the label of the ticks, by substituting character
   !               strings provided by the user. Module variables are
   !               stored in 'mod_pgplot'.
   !-----------------------------------------------------------------------

   character*20 :: clbl
   character*64 :: opt
   logical :: xopta, xoptb, xoptc, xoptn, xoptm, xoptt, xopts
   logical :: yopta, yoptb, yoptc, yoptn, yoptm, yoptt, yopts
   logical :: xopti, yopti, yoptv, xoptl, yoptl, xoptp, yoptp
   logical :: xrange, major, xoptls, yoptls
   logical :: xoptls2, xoptls3, yoptls2, yoptls3
   integer :: i, i1, i2, j, nc, np, nv, ki, clip
   integer :: j2, j3
   integer :: nsubx, nsuby, jmax, xnform, ynform
   double precision :: tikl, tikl1, tikl2, xc, yc
   double precision :: xint, xint2, xval, yint, yint2, yval
   double precision :: a, b, c
   double precision :: xndsp, xmdsp, yndsp, ymdsp, ynvdsp, ymvdsp
   double precision :: xblc, xtrc, yblc, ytrc

   double precision :: xlo, xhi

   double precision :: v1, v2, vv ! avoid precision loss in log(exp(r))
   double precision :: ecart, ecart_ratio, factor
   integer :: mm, pp

   ! Table of logarithms 1..10 (log10)
   double precision :: tab(10), utab(10)
   DATA TAB / 0.000000000000000d0, 0.301029995663981d0,                 &
              0.477121254719662d0, 0.602059991327962d0,                 &
              0.698970004336019d0, 0.778151250383644d0,                 &
              0.845098040014257d0, 0.903089986991944d0,                 &
              0.954242509439325d0, 1.000000000000000d0 /

   double precision :: tikl_v, tikl_h

   logical :: to_print, clipping_disabled, clipping_save
   double precision :: num_val
   integer :: vec_tmp(1), n1, n2

   double precision :: rbuf(1)
   integer :: ibuf(2), lchr
   character(len=14) :: chr

   xrange(a,b,c) = ( a<b .and. b<c ) .or. ( c<b .and. b<a )

   !------ end of declarations -- execution starts hereafter  ------

   ! Means that these module variables are undefined!
   ! (see below for their conditional set)
   X_ALL_TICKS_IND_START(pgid) = -666 ; X_ALL_TICKS_IND_END(pgid) = -666
   X_MAJ_TICKS_IND_START(pgid) = -666 ; X_MAJ_TICKS_IND_END(pgid) = -666

   call pgbbuf()

   ! Get window in world coordinates
   call pgqwin( xblc, xtrc, yblc, ytrc )

   ! Decode options.
   call grtoup(opt,xopt)
   xopta = index(opt,'A') /= 0 .and. xrange(yblc,0.0d0,ytrc)
   xoptb = index(opt,'B') /= 0
   xoptc = index(opt,'C') /= 0
   xopti = index(opt,'I') /= 0
   xoptl = index(opt,'L') /= 0
   xoptm = index(opt,'M') /= 0
   xoptn = index(opt,'N') /= 0
   xopts = index(opt,'S') /= 0
   xoptt = index(opt,'T') /= 0
   xoptp = index(opt,'P') /= 0 .and. (.not.xopti)
   xnform = 0
   if( index(opt,'1') /= 0 ) xnform = 1
   if( index(opt,'2') /= 0 ) xnform = 2
   call grtoup(opt,yopt)
   yopta = index(opt,'A') /= 0 .and. xrange(xblc,0.0d0,xtrc)
   yoptb = index(opt,'B') /= 0
   yoptc = index(opt,'C') /= 0
   yopti = index(opt,'I') /= 0
   yoptl = index(opt,'L') /= 0
   yoptm = index(opt,'M') /= 0
   yoptn = index(opt,'N') /= 0
   yopts = index(opt,'S') /= 0
   yoptt = index(opt,'T') /= 0
   yoptv = index(opt,'V') /= 0
   yoptp = index(opt,'P') /= 0 .and. (.not.yopti)
   ynform = 0
   if( index(opt,'1') /= 0 ) ynform = 1
   if( index(opt,'2') /= 0 ) ynform = 2

   ! Displacement of labels from edge of box
   ! (for X bottom/top, Y left/right, and Y left/right with V option).
   xndsp = 1.2d0
   xmdsp = 0.7d0
   yndsp = 0.7d0
   ymdsp = 1.2d0
   ynvdsp = 0.7d0
   ymvdsp = 0.7d0
   if( xopti ) then
      xndsp = xndsp + 0.3d0
      xmdsp = xmdsp + 0.3d0
   end if
   if( yopti ) then
      yndsp = yndsp + 0.3d0
      ymdsp = ymdsp + 0.3d0
      ynvdsp = ynvdsp + 0.3d0
      ymvdsp = ymvdsp + 0.3d0
   end if

   ! Disable clipping (ticks may be drawn outside, and axis lines
   ! could be badly drawn). The change of CLIPPING_IN_AXES is required
   ! for monitoring the Quick Return in pgline.
   clipping_disabled = .false.
   if( CLIPPING_IN_AXES ) then
      ibuf(1) = 0 ! unset clipping (used only in X11 driver)
      call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr )
      clipping_save = CLIPPING_IN_AXES
      CLIPPING_IN_AXES = .false.
      clipping_disabled = .true.
   end if

   tikl_v = pgxsp(pgid)*0.6d0*((ytrc-yblc)/pgylen(pgid))
   tikl_h = pgxsp(pgid)*0.6d0*((xtrc-xblc)/pgxlen(pgid))

   ! Increase slightly the tick lengths, if the axis width is greater
   ! than unity.
   factor = max( 1.0d0, (pg_axis_lin_width(pgid)+3.0d0)/(1.0d0+3.0d0) )
   tikl_v = tikl_v * factor
   tikl_h = tikl_h * factor

   ! Set appropriate cap and join style for the lines.
   ibuf(1) = 0 ! Cap butt
   ibuf(2) = 0 ! Join miter
   call grexec( grgtyp, SET_LINE_CAP_JOIN_STYLE, rbuf, ibuf, chr, lchr )

   ! Draw box.
   call grrect( xblc, yblc, xtrc, ytrc, filled=.false. )

   ! Draw axes if required.
   if( xopta ) then
      call grmova( xblc, 0.0d0 )
      call grlina( xtrc, 0.0d0 )
   end if
   if( yopta ) then
      call grmova( 0.0d0, yblc )
      call grlina( 0.0d0, ytrc )
   end if

   !========================= X axis ==================================

   ! Length of X tick marks.
   tikl1 = tikl_v
   if( xopti ) tikl1 = -tikl1
   tikl2 = tikl1*0.5d0

   ! Choose X tick intervals.
   ! Major interval: XINT, minor interval: XINT2
   utab(1) = 0.0
   if( xoptl ) then
      xint = sign(1.0d0,xtrc-xblc)
      nsubx = 1
      if( xint > 0.0 ) then
         utab(2:10) = tab(2:10)
      else
         utab(2:10) = 1.0 - tab(2:10)
      end if
   else if( xtick == 0.0 ) then
      xint = max(0.05d0, min(7.0d0*pgxsp(pgid)/pgxlen(pgid), 0.2d0))*(xtrc-xblc)
      xint = pgrnd(xint,nsubx)
   else
      xint = sign(xtick,xtrc-xblc)
      nsubx = max(nxsub,1)
   end if
   if( .not. xopts ) nsubx = 1
   np = int(log10(abs(xint))) - 4
   nv = nint(xint/10.0d0**np)
   xint2 = xint/nsubx
   xoptls = xoptl .and. xopts .and. (abs(xtrc-xblc)<2.0d0)
   xoptls2 = xoptls .and. (abs(xtrc-xblc)<1.0d0)
   xoptls3 = xoptls2 .and. (abs(xtrc-xblc)<0.5d0)

   call tiny_shift_range( xblc, xtrc, xint2 )

   ! Draw X ticks.
   if( xoptt .or. xopts ) then
      call pgbox1(xblc, xtrc, xint2, i1, i2)
      jmax = 1
      if( xoptl .and. xopts ) jmax = 9

      ! Bottom ticks.
      if( xoptb ) then
         if( xoptl ) then
            ! Log Axes
            do i = i1-1, i2
               do j = 1, jmax
                  major = xoptt .and. j==1
                  if( major ) then
                     tikl = tikl1
                  else
                     tikl = tikl2
                  end if
                  xval = (i+utab(j))*xint2
                  if( irange(xblc,xval,xtrc) ) then
                     if( xoptp .and. major ) then
                        call grmova(xval, yblc-tikl2)
                     else
                        call grmova(xval, yblc)
                     end if
                     call grlina( xval, yblc+tikl )
                  end if
                  if( xoptls3 ) then
                     v1 = 10.0d0**xval
                     v2 = 10.0d0**((i+utab(j+1))*xint2)
                     !--------------
                     xval = log10( v1 + 0.25d0*(v2-v1) )
                     if( irange(xblc,xval,xtrc) ) then
                        call grmova( xval, yblc )
                        call grlina( xval, yblc+tikl2 )
                     end if
                     !--------------
                     xval = log10( v1 + 0.5d0*(v2-v1) )
                     if( irange(xblc,xval,xtrc) ) then
                        call grmova( xval, yblc )
                        call grlina( xval, yblc+tikl2 )
                     end if
                     !--------------
                     xval = log10( v1 + 0.75d0*(v2-v1) )
                     if( irange(xblc,xval,xtrc) ) then
                        call grmova( xval, yblc )
                        call grlina( xval, yblc+tikl2 )
                     end if
                  end if
               end do
            end do
         else
            ! Linear Axes
            do i = i1-1, i2
               major = (mod(i,nsubx)==0) .and. xoptt
               if( major ) then
                  tikl = tikl1
               else
                  tikl = tikl2
               end if
               xval = i*xint2
               if( INTEGER_XLABEL(pgid) .and. .not. PGBOX_IN_COLORBAR ) then
                  if( real(nint(xval)) == real(xval) ) then
                     to_print = .true.
                  else
                     to_print = .false.
                  end if
               else
                  to_print = .true.
               end if
               if( to_print ) then
                  if( irange(xblc,xval,xtrc) ) then
                     if( xoptp .and. major ) then
                        call grmova( xval, yblc-tikl2 )
                     else
                        call grmova( xval, yblc )
                     end if
                     call grlina( xval, yblc+tikl )
                  end if
               end if
            end do
         end if
      end if

      ! Axis ticks.
      if( xopta ) then
         do i = i1-1, i2
            do j = 1, jmax
               major = (mod(i,nsubx)==0) .and. xoptt .and. j == 1
               if( major ) then
                  tikl = tikl1
               else
                  tikl = tikl2
               end if
               xval = (i+utab(j))*xint2
               if( irange(xblc,xval,xtrc) ) then
                  call grmova( xval, -tikl )
                  call grlina( xval, tikl )
               end if
            end do
         end do
      end if

      ! Top ticks.
      if( xoptc ) then
         do i = i1-1, i2
            do j = 1, jmax
               major = (mod(i,nsubx)==0) .and. xoptt .and. j == 1
               if( major ) then
                  tikl = tikl1
               else
                  tikl = tikl2
               end if
               xval = (i+utab(j))*xint2
               if( INTEGER_XLABEL(pgid) .and. .not. PGBOX_IN_COLORBAR ) then
                  if( real(nint(xval)) == real(xval) ) then
                     to_print = .true.
                  else
                     to_print = .false.
                  end if
               else
                  to_print = .true.
               end if
               if( to_print ) then
                  if( irange(xblc,xval,xtrc) ) then
                     call grmova( xval, ytrc-tikl )
                     call grlina( xval, ytrc )
                  end if
               end if
            end do
         end do
      end if
   end if

   ! Write X labels.
   if( xoptn .or. xoptm ) then

10 continue
if( USER_TICKS(pgid) ) then
   ! Special case: ticks labels are provided by the user...

   if( USER_TICKS_ALL(pgid) ) then

      ! Below, 'xint2' is used to take all ticks (both major and minor)
      call pgbox1(xblc, xtrc, xint2, i1, i2) ! Get (i1,i2)
      vec_tmp = lbound(USER_TICKS_LABELS(pgid)%labels) ; n1 = vec_tmp(1)
      vec_tmp = ubound(USER_TICKS_LABELS(pgid)%labels) ; n2 = vec_tmp(1)
      if( n1 /= i1 .or. n2 /= i2 ) then
         print *, "(MUESLI MFPLOT:) Warning."
         print *, "   During axis labelling, an incompatibility has been found,"
         print *, "   perhaps due to a change in the X-axis range..."
         print *, "   Cannot process USER LABELS for TICKS (see msSetXAxisUserLabels)"
         print *, "   -> USER LABELS have been deactivated."
         ! deactivating USER LABELS for TICKS
         USER_TICKS(pgid) = .false.
         go to 10
      end if
      do i = i1, i2
         xc = (i*xint2-xblc)/(xtrc-xblc)
         nc = USER_TICKS_LABELS(pgid)%length(i)
         call pgmtxt( 'B', xndsp, xc, 0.5d0,                            &
                      USER_TICKS_LABELS(pgid)%labels(i)(1:nc))
      end do

   else ! only major ticks are re-labeled

      call pgbox1(xblc, xtrc, xint, i1, i2) ! Get (i1,i2)
      vec_tmp = lbound(USER_TICKS_LABELS(pgid)%labels) ; n1 = vec_tmp(1)
      vec_tmp = ubound(USER_TICKS_LABELS(pgid)%labels) ; n2 = vec_tmp(1)
      if( n1 /= i1 .or. n2 /= i2 ) then
         print *, "(MUESLI MFPLOT:) Warning."
         print *, "   During axis labelling, an incompatibility has been found,"
         print *, "   perhaps due to a change in the X-axis range..."
         print *, "   Cannot process USER LABELS for TICKS (see msSetXAxisUserLabels)"
         print *, "   -> USER LABELS have been deactivated."
         ! deactivating USER LABELS for TICKS
         USER_TICKS(pgid) = .false.
         go to 10
      end if
      do i = i1, i2
         xc = (i*xint-xblc)/(xtrc-xblc)
         nc = USER_TICKS_LABELS(pgid)%length(i)
         call pgmtxt( 'B', xndsp, xc, 0.5d0,                            &
                      USER_TICKS_LABELS(pgid)%labels(i)(1:nc))
      end do

   end if

else ! usual case

      ! Get all ticks indices
      call pgbox1(xblc, xtrc, xint2, i1, i2) ! Get (i1,i2)
      X_ALL_TICKS_IND_START(pgid) = i1 ; X_ALL_TICKS_IND_END(pgid) = i2
      call pgbox1(xblc, xtrc, xint, i1, i2) ! Get (i1,i2)
      X_MAJ_TICKS_IND_START(pgid) = i1 ; X_MAJ_TICKS_IND_END(pgid) = i2

      do i = i1, i2
         xc = (i*xint-xblc)/(xtrc-xblc)
         if( xoptl ) then
            call pgnumb(1,nint(i*xint),xnform,clbl,nc)
         else
            call pgnumb(i*nv,np,xnform,clbl,nc)
         end if
         if( INTEGER_XLABEL(pgid) .and. .not. PGBOX_IN_COLORBAR ) then
            num_val = (i*nv)*10.0d0**np
            if( real(nint(num_val)) == real(num_val) ) then
               to_print = .true.
            else
               to_print = .false.
            end if
         else
            to_print = .true.
         end if
         if( to_print ) then
            if( xoptn ) call pgmtxt('B', xndsp, xc, 0.5d0, clbl(1:nc))
            if( xoptm ) call pgmtxt('T', xmdsp, xc, 0.5d0, clbl(1:nc))
         end if
      end do

end if

   end if

   ! Extra X labels for log axes.
   if( xoptls ) then
      call pgbox1(xblc, xtrc, xint2, i1, i2)
      if( xoptls2 ) then
         j2 = 9
         j3 = 1
      else
         j2 = 5
         j3 = 3
      end if
      do i = i1-1, i2
         do j = 2, j2, j3
            xval = (i+utab(j))*xint2
            xc = (xval-xblc)/(xtrc-xblc)
            ki = i
            if( xtrc < xblc ) ki = ki + 1
            if( irange(xblc,xval,xtrc) ) then
               mm = j
               pp = nint(ki*xint2) ! round to nearest integer
               call pgnumb( mm, pp, xnform, clbl, nc )
               if( xoptn ) call pgmtxt('B', xndsp, xc, 0.5d0, clbl(1:nc))
               if( xoptm ) call pgmtxt('T', xmdsp, xc, 0.5d0, clbl(1:nc))
            end if
         end do
         if( xoptls3 ) then
            do j = 1, 9
               v1 = 10.0d0**((i+utab(j))*xint2)
               v2 = 10.0d0**((i+utab(j+1))*xint2)
               ecart = (i+utab(j+1))*xint2 - (i+utab(j))*xint2
               ecart_ratio = ecart/abs(xtrc-xblc)
               if( ecart_ratio > 0.5d0 ) then
                  !--------------
                  xval = log10( v1 + 0.25d0*(v2-v1) )
                  xc = (xval-xblc)/(xtrc-xblc)
                  ki = i
                  if( xtrc < xblc ) ki = ki + 1
                  if( irange(xblc,xval,xtrc) ) then
                     vv = v1 + 0.25d0*(v2-v1)
                     ! need two more decimals to be drawn...
                     pp = -( 3 - nint(log10(vv)) )
                     mm = nint( vv * 10.0d0**( -pp ) )
                     call pgnumb(mm,pp,xnform,clbl,nc)
                     if( xoptn ) call pgmtxt('B', xndsp, xc, 0.5d0, clbl(1:nc))
                     if( xoptm ) call pgmtxt('T', xmdsp, xc, 0.5d0, clbl(1:nc))
                  end if
                  !--------------
                  xval = log10( v1 + 0.5d0*(v2-v1) )
                  xc = (xval-xblc)/(xtrc-xblc)
                  ki = i
                  if( xtrc < xblc ) ki = ki + 1
                  if( irange(xblc,xval,xtrc) ) then
                     vv = v1 + 0.5d0*(v2-v1)
                     ! need two more decimals to be drawn...
                     pp = -( 3 - nint(log10(vv)) )
                     mm = nint( vv * 10.0d0**( -pp ) )
                     call pgnumb(mm,pp,xnform,clbl,nc)
                     if( xoptn ) call pgmtxt('B', xndsp, xc, 0.5d0, clbl(1:nc))
                     if( xoptm ) call pgmtxt('T', xmdsp, xc, 0.5d0, clbl(1:nc))
                  end if
                  !--------------
                  xval = log10( v1 + 0.75d0*(v2-v1) )
                  xc = (xval-xblc)/(xtrc-xblc)
                  ki = i
                  if( xtrc < xblc ) ki = ki + 1
                  if( irange(xblc,xval,xtrc) ) then
                     vv = v1 + 0.75d0*(v2-v1)
                     ! need two more decimals to be drawn...
                     pp = -( 3 - nint(log10(vv)) )
                     mm = nint( vv * 10.0d0**( -pp ) )
                     call pgnumb(mm,pp,xnform,clbl,nc)
                     if( xoptn ) call pgmtxt('B', xndsp, xc, 0.5d0, clbl(1:nc))
                     if( xoptm ) call pgmtxt('T', xmdsp, xc, 0.5d0, clbl(1:nc))
                  end if
               end if
            end do
         end if
      end do
   end if

   !========================= Y axis ==================================

   ! Length of Y tick marks.
   tikl1 = tikl_h
   if( yopti ) tikl1 = -tikl1
   tikl2 = tikl1*0.5d0

   ! Choose Y tick intervals.
   ! Major interval: YINT, minor interval: YINT2
   utab(1) = 0.0
   if( yoptl ) then
      yint = sign(1.0d0,ytrc-yblc)
      nsuby = 1
      if( yint > 0.0 ) then
         utab(2:10) = tab(2:10)
      else
         utab(2:10) = 1.0 - tab(2:10)
      end if
   else if( ytick == 0.0 ) then
      yint = max(0.05d0, min(7.0d0*pgxsp(pgid)/pgylen(pgid), 0.2d0))*(ytrc-yblc)
      yint = pgrnd(yint,nsuby)
   else
      yint  = sign(ytick,ytrc-yblc)
      nsuby = max(nysub,1)
   end if
   if( .not. yopts ) nsuby = 1
   np = int(log10(abs(yint))) - 4
   nv = nint(yint/10.0d0**np)
   yint2 = yint/nsuby
   yoptls = yoptl .and. yopts .and. (abs(ytrc-yblc)<2.0d0)
   yoptls2 = yoptls .and. (abs(ytrc-yblc)<1.0d0)
   yoptls3 = yoptls2 .and. (abs(ytrc-yblc)<0.5d0)

   call tiny_shift_range( yblc, ytrc, yint2 )

   ! Draw Y ticks.
   if( yoptt .or. yopts ) then
      call pgbox1(yblc, ytrc, yint2, i1, i2)
      jmax = 1
      if( yoptl .and. yopts ) jmax = 9

      ! Left ticks.
      if( yoptb ) then
         if( yoptl ) then
            ! Log Axes
            do i = i1-1, i2
               do j = 1, jmax
                  major = yoptt .and. j==1
                  if( major ) then
                     tikl = tikl1
                  else
                     tikl = tikl2
                  end if
                  yval = (i+utab(j))*yint2
                  if( irange(yblc,yval,ytrc) ) then
                     if( yoptp .and. major ) then
                        call grmova( xblc-tikl2, yval )
                     else
                        call grmova( xblc, yval )
                     end if
                     call grlina( xblc+tikl, yval )
                  end if
                  if( yoptls3 ) then
                     v1 = 10.0d0**yval
                     v2 = 10.0d0**((i+utab(j+1))*yint2)
                     !--------------
                     yval = log10( v1 + 0.25d0*(v2-v1) )
                     if( irange(yblc,yval,ytrc) ) then
                        call grmova( xblc, yval )
                        call grlina( xblc+tikl, yval )
                     end if
                     !--------------
                     yval = log10( v1 + 0.5d0*(v2-v1) )
                     if( irange(yblc,yval,ytrc) ) then
                        call grmova( xblc, yval )
                        call grlina( xblc+tikl, yval )
                     end if
                     !--------------
                     yval = log10( v1 + 0.75d0*(v2-v1) )
                     if( irange(yblc,yval,ytrc) ) then
                        call grmova( xblc, yval )
                        call grlina( xblc+tikl, yval )
                     end if
                  end if
               end do
            end do
         else
            ! Linear Axes
            do i = i1-1, i2
               major = (mod(i,nsuby)==0) .and. yoptt
               if( major ) then
                  tikl = tikl1
               else
                  tikl = tikl2
               end if
               yval = i*yint2
if( INTEGER_YLABEL(pgid) .and. .not. PGBOX_IN_COLORBAR ) then
   if( real(nint(yval)) == real(yval) ) then
      to_print = .true.
   else
      to_print = .false.
   end if
else
   to_print = .true.
end if
if( to_print ) then
               if( irange(yblc,yval,ytrc) ) then
                  if( yoptp .and. major ) then
                     call grmova( xblc-tikl2, yval )
                  else
                     call grmova( xblc, yval )
                  end if
                  call grlina( xblc+tikl, yval )
               end if
end if
            end do
         end if
      end if

      ! Axis ticks.
      if( yopta ) then
         do i = i1-1, i2
            do j = 1, jmax
               major = (mod(i,nsuby)==0) .and. yoptt .and. j == 1
               if( major ) then
                  tikl = tikl1
               else
                  tikl = tikl2
               end if
               yval = (i+utab(j))*yint2
               if( irange(yblc,yval,ytrc) ) then
                  call grmova( -tikl, yval )
                  call grlina( tikl, yval )
               end if
            end do
         end do
      end if

      ! Right ticks.
      if( yoptc ) then
         do i = i1-1, i2
            do j = 1, jmax
               major = (mod(i,nsuby)==0) .and. yoptt .and. j == 1
               if( major ) then
                  tikl = tikl1
               else
                  tikl = tikl2
               end if
               yval = (i+utab(j))*yint2
if( INTEGER_YLABEL(pgid) .and. .not. PGBOX_IN_COLORBAR ) then
   if( real(nint(yval)) == real(yval) ) then
      to_print = .true.
   else
      to_print = .false.
   end if
else
   to_print = .true.
end if
if( to_print ) then
               if( irange(yblc,yval,ytrc) ) then
                  call grmova( xtrc-tikl, yval )
                  call grlina( xtrc, yval )
               end if
end if
            end do
         end do
      end if
   end if

   ! Write Y labels.
   if( yoptn .or. yoptm ) then
      call pgbox1(yblc, ytrc, yint, i1, i2)
      do i = i1, i2
         yc = (i*yint-yblc)/(ytrc-yblc)
         if( yoptl ) then
            call pgnumb(1,nint(i*yint),ynform,clbl,nc)
         else
            call pgnumb(i*nv,np,ynform,clbl,nc)
         end if
if( INTEGER_YLABEL(pgid) .and. .not. PGBOX_IN_COLORBAR ) then
   num_val = (i*nv)*10.0d0**np
   if( real(nint(num_val)) == real(num_val) ) then
      to_print = .true.
   else
      to_print = .false.
   end if
else
   to_print = .true.
end if
if( to_print ) then
         if( yoptv ) then
            if( yoptn ) call pgmtxt('LV',ynvdsp,yc,1.0d0,clbl(1:nc))
            if( yoptm ) call pgmtxt('RV',ymvdsp,yc,0.0d0,clbl(1:nc))
         else
            if( yoptn ) call pgmtxt('L',yndsp,yc,0.5d0,clbl(1:nc))
            if( yoptm ) call pgmtxt('R',ymdsp,yc,0.5d0,clbl(1:nc))
         end if
end if
      end do
   end if

   ! Extra Y labels for log axes.
   if( yoptls ) then
      call pgbox1(yblc, ytrc, yint2, i1, i2)
      if( yoptls2 ) then
         j2 = 9
         j3 = 1
      else
         j2 = 5
         j3 = 3
      end if
      do i = i1-1, i2
         do j = 2, j2, j3
            yval = (i+utab(j))*yint2
            yc = (yval-yblc)/(ytrc-yblc)
            ki = i
            if( yblc > ytrc ) ki = ki + 1
            if( irange(yblc,yval,ytrc) ) then
               mm = j
               pp = nint(ki*yint2) ! round to nearest integer
               call pgnumb( mm, pp, ynform, clbl, nc )
               if( yoptv ) then
                  if( yoptn ) call pgmtxt('LV', ynvdsp, yc, 1.0d0, clbl(1:nc))
                  if( yoptm ) call pgmtxt('RV', ymvdsp, yc, 0.0d0, clbl(1:nc))
               else
                  if( yoptn ) call pgmtxt('L', yndsp, yc, 0.5d0, clbl(1:nc))
                  if( yoptm ) call pgmtxt('R', ymdsp, yc, 0.5d0, clbl(1:nc))
               end if
            end if
         end do
         if( yoptls3 ) then
            do j = 1, 9
               v1 = 10.0d0**((i+utab(j))*yint2)
               v2 = 10.0d0**((i+utab(j+1))*yint2)
               ecart = (i+utab(j+1))*yint2 - (i+utab(j))*yint2
               ecart_ratio = ecart/abs(ytrc-yblc)
               if( ecart_ratio > 0.5d0 ) then
                  !--------------
                  yval = log10( v1 + 0.25d0*(v2-v1) )
                  yc = (yval-yblc)/(ytrc-yblc)
                  ki = i
                  if( ytrc < yblc ) ki = ki + 1
                  if( irange(yblc,yval,ytrc) ) then
                     vv = v1 + 0.25d0*(v2-v1)
                     ! need two more decimals to be drawn...
                     pp = -( 3 - nint(log10(vv)) )
                     mm = nint( vv * 10.0d0**( -pp ) )
                     call pgnumb(mm,pp,ynform,clbl,nc)
                     if( yoptv ) then
                        if( yoptn ) call pgmtxt('LV', ynvdsp, yc, 1.0d0, clbl(1:nc))
                        if( yoptm ) call pgmtxt('RV', ymvdsp, yc, 0.0d0, clbl(1:nc))
                     else
                        if( yoptn ) call pgmtxt('L', yndsp, yc, 0.5d0, clbl(1:nc))
                        if( yoptm ) call pgmtxt('R', ymdsp, yc, 0.5d0, clbl(1:nc))
                     end if
                  end if
                  !--------------
                  yval = log10( v1 + 0.5d0*(v2-v1) )
                  yc = (yval-yblc)/(ytrc-yblc)
                  ki = i
                  if( ytrc < yblc ) ki = ki + 1
                  if( irange(yblc,yval,ytrc) ) then
                     vv = v1 + 0.5d0*(v2-v1)
                     ! need two more decimals to be drawn...
                     pp = -( 3 - nint(log10(vv)) )
                     mm = nint( vv * 10.0d0**( -pp ) )
                     call pgnumb(mm,pp,ynform,clbl,nc)
                     if( yoptv ) then
                        if( yoptn ) call pgmtxt('LV', ynvdsp, yc, 1.0d0, clbl(1:nc))
                        if( yoptm ) call pgmtxt('RV', ymvdsp, yc, 0.0d0, clbl(1:nc))
                     else
                        if( yoptn ) call pgmtxt('L', yndsp, yc, 0.5d0, clbl(1:nc))
                        if( yoptm ) call pgmtxt('R', ymdsp, yc, 0.5d0, clbl(1:nc))
                     end if
                  end if
                  !--------------
                  yval = log10( v1 + 0.75d0*(v2-v1) )
                  yc = (yval-yblc)/(ytrc-yblc)
                  ki = i
                  if( ytrc < yblc ) ki = ki + 1
                  if( irange(yblc,yval,ytrc) ) then
                     vv = v1 + 0.75d0*(v2-v1)
                     ! need two more decimals to be drawn...
                     pp = -( 3 - nint(log10(vv)) )
                     mm = nint( vv * 10.0d0**( -pp ) )
                     call pgnumb(mm,pp,ynform,clbl,nc)
                     if( yoptv ) then
                        if( yoptn ) call pgmtxt('LV', ynvdsp, yc, 1.0d0, clbl(1:nc))
                        if( yoptm ) call pgmtxt('RV', ymvdsp, yc, 0.0d0, clbl(1:nc))
                     else
                        if( yoptn ) call pgmtxt('L', yndsp, yc, 0.5d0, clbl(1:nc))
                        if( yoptm ) call pgmtxt('R', ymdsp, yc, 0.5d0, clbl(1:nc))
                     end if
                  end if
               end if
            end do
         end if
      end do
   end if

   call pgebuf()

   if( clipping_disabled ) then
      ! Restore the clipping state at the viewport
      CLIPPING_IN_AXES = clipping_save
      if( CLIPPING_IN_AXES ) then
         ibuf(1) = 1 ! clipping
         call grexec( grgtyp, SET_CLIPPING, rbuf, ibuf, chr, lchr ) ! set clipping
      end if
   end if

contains

   function irange( a, b, c ) result( bool )

      double precision, intent(in) :: a, b, c
      logical                      :: bool
      !----------------------

      logical :: a_b_close, b_c_close

      a_b_close = abs(a-b)<epsilon(1.0)*abs(a)
      b_c_close = abs(b-c)<epsilon(1.0)*abs(c)

      bool = ( ( a<=b .or. a_b_close ) .and. ( b<=c .or. b_c_close ) ) .or. &
             ( ( c<=b .or. b_c_close ) .and. ( b<=a .or. a_b_close ) )

   end function irange

end subroutine
